<?php
// @codingStandardsIgnoreStart

/**
 * GA4 Automation
 *
 * @package StrategySuite
 */

namespace StrategySuite\Analytics;

use Google\Analytics\Admin\V1beta\AnalyticsAdminServiceClient;
use Google\Analytics\Admin\V1beta\DataStream;
use Google\Analytics\Data\V1beta\BetaAnalyticsDataClient;
use Google\Analytics\Data\V1beta\Metric;
use Google\Analytics\Data\V1beta\Dimension;
use Google\Analytics\Data\V1beta\DateRange;
use Google\Analytics\Data\V1beta\Filter;
use Google\Analytics\Data\V1beta\FilterExpression;
use Google\Analytics\Data\V1beta\Filter\StringFilter;
use Google\ApiCore\ApiException;
use Google\Auth\Credentials\ServiceAccountCredentials;
use Google\Ads\GoogleAds\Lib\V20\GoogleAdsClientBuilder;
use Google\Ads\GoogleAds\V20\Services\SearchGoogleAdsRequest;


class GA4_Automation {
	private $property_id;
	private $url;
	private $service_account_file_path;

	private $client;

	/**
	 * The beginning...
	 */
	public function __construct( $property_id, $url, $service_account = 'adwords' ) {
		$this->property_id = $property_id;
		$this->url = $url;
		// $autoload_path = '' === trim($_SERVER['DOCUMENT_ROOT']) ? 'vendor/autoload.php' : dirname($_SERVER['DOCUMENT_ROOT']) . '/vendor/autoload.php';
		// require_once $autoload_path;
		$this->service_account_file_path = STRATEGY_SUITE_PATH . 'oauth/automation-service-account-' . $service_account . '.json';
		$this->client = new AnalyticsAdminServiceClient(
			[
				'credentials'   => $this->service_account_file_path,
			]
		);
	}

	/**
	 * Accepts an array of arguments and checks that all of them are defined.
	 *
	 * @param array $args       The function arguments
	 *
	 * @return bool             Whether the arguments are all defined.
	 */
	protected function all_non_null( $args = [] ) {
		if ( ! is_array( $args ) ) {
			return false;
		}

		foreach ( $args as $arg ) {
			if ( ! $arg ) {
				return false;
			}
		}

		return true;
	}

	protected function to_camel_case( $string ) {
		$string = preg_replace( '/[^a-zA-Z0-9]+/', ' ', $string );
		$string = ucwords( $string );
		$string = str_replace( ' ', '', $string );
		$string = lcfirst( $string );

		return $string;
	}

	protected function create_event_name( $string, $add_suffix = true ) {
		$suffix = 'Submission';
		$max_length = $add_suffix ? 40 - strlen( $suffix ) : 40;
		$event_name = substr( $this->to_camel_case( $string ), 0, $max_length );
		$event_name .= $add_suffix ? $suffix : '';
		return $event_name;
	}

	protected function build_path( $args = [] ) {
		$path = 'properties/' . $this->property_id;
		if ( 0 < count( $args ) ) {
			foreach ( $args as $arg ) {
				if ( is_array( $arg ) && 2 === count( $arg ) ) {
					$path .= '/' . $arg[0] . '/' . $arg[1];
				}
			}
		}
		return $path;
	}

	protected function default_data_stream_template( $args = [] ) {
		$defaults = [];

		if ( isset( $args['exclude'] ) && is_array( $args['exclude'] ) ) {
			foreach ( $args['exclude'] as $exclusion ) {
				if ( isset( $defaults[ $exclusion ] ) ) {
					unset( $defaults[ $exclusion ] );
				}
			}
		}

		return $defaults;
	}

	protected function data_stream_schema( $args = [] ) {
		if ( ! isset( $args['type'] ) ) {
			return null;
		}

		$data_stream = new DataStream();
		switch ( $args['type'] ) {
			case 'Default':
				if ( $this->all_non_null( [ $args['name'] ] ) ) {
					$data_stream->setName( $args['name'] );
				}
				break;
			default:
				break;
		}

		return $data_stream;
	}

	private function get_property( $args = [] ) {
		if ( ! $this->all_non_null( [ $this->property_id ] ) ) {
			return false;
		}

		$property = $this->client->getProperty( 'properties/' . $this->property_id );

		return $property;
	}

	private function get_data_stream( $args = [] ) {
		if ( ! $this->all_non_null( [ $this->property_id ] ) ) {
			return false;
		}

		$data_stream_ids = [];
		foreach ( $args['data_streams'] as $data_stream ) {
			if ( is_array( $data_stream ) && isset( $data_stream['name'] ) ) {
				$data_stream_ids[ $data_stream['name'] ] = false;
			}
		}

		$path = isset( $args['id'] ) ? $this->build_path( [ [ 'dataStreams', $args['id'] ] ] ) : $this->build_path();
		$streams = $this->client->listDataStreams( $path );
		foreach ( $streams as $stream ) {
			$name = $stream?->getDisplayName();
			$id = $stream?->getName();
			// var_dump($name);
			// var_dump(basename($id));
			// if ( isset($data_stream_ids[$name]) ) {
			// $data_stream_ids[$name] = $stream?->getDataStreamId
			// }
		}
		return $streams;
	}

	private function create_conversion_events( $args = [] ) {
	}

	private function create_data_streams( $args = [] ) {
		if ( ! $this->all_non_null( [ $this->property_id, $this->url ] ) || ! is_array( $args ) || ! isset( $args['data_stream'] ) ) {
			return false;
		}

		$data_stream_ids = [];
		foreach ( $args['data_streams'] as $data_stream ) {
			if ( is_array( $data_stream ) && isset( $data_stream['name'] ) ) {
				$data_stream_ids[ $data_stream['name'] ] = false;
			}
		}

		$path = $this->build_path();
		$data_streams = $this->client->listDataStreams( $path );
		foreach ( $data_streams as $data_stream ) {
			$name = $data_stream?->getDisplayName();
			if ( isset( $data_stream_ids[ $name ] ) ) {
				$data_stream_ids[ $name ] = basename( $data_stream?->getName() );
			}
		}

		foreach ( $args['data_streams'] as $data_stream_data ) {
			if ( isset( $data_stream_ids[ $data_stream_data['name'] ] ) && false === $data_stream_ids[ $data_stream['name'] ] ) {

			}
		}
	}

	private function configure_data_stream( $data_stream_id, $args = [] ) {
	}

	/**
	 * Fetch Google Analytics 4 report data for the property.
	 *
	 * @return array|false Returns an associative array of metrics (views, engagement, conversion rate, top source)
	 *                     or false if the property ID or service account file is missing.
	 */
	public function get_report_data( $url ) {
		if ( ! $this->property_id || ! file_exists( $this->service_account_file_path ) ) {
			return false;
		}

		putenv( 'GOOGLE_APPLICATION_CREDENTIALS=' . $this->service_account_file_path );

		$client = new BetaAnalyticsDataClient();

		$response = $client->runReport([
			'property' => 'properties/' . $this->property_id,
			'metrics' => [
				new Metric(['name' => 'screenPageViews']),
				new Metric(['name' => 'averageSessionDuration']),
				new Metric(['name' => 'conversions']),
			],
			'dimensions' => [
				new Dimension(['name' => 'sessionSource']),
			],
			'dateRanges' => [
				new DateRange([
					'start_date' => '7daysAgo',
					'end_date' => 'today',
				]),
			],
			'dimensionFilter' => new FilterExpression([
				'filter' => new Filter([
					'field_name' => 'pagePath',
					'string_filter' => new StringFilter([
						'match_type' => StringFilter\MatchType::EXACT,
						'value' => $url,
					]),
				]),
			]),
		]);

		$views = 0;
		$avg_duration = 0;
		$top_source = '';
		$rows = iterator_to_array( $response->getRows() );

		foreach ( $rows as $index => $row ) {
			$source = $row->getDimensionValues()[0]->getValue();
			$views += (int) $row->getMetricValues()[0]->getValue();
			$avg_duration = (float) $row->getMetricValues()[1]->getValue();
			if ( $index === 0 ) {
				$top_source = $source;
			}
		}

		return [
			'views' => $views,
			'engagement' => gmdate('i\m s\s', (int)$avg_duration),
			'conversion_rate' => 0,
			'top_source' => $top_source,
		];
	}

	/**
	 * Fetch Google Ads metrics for a given customer using the Google Ads API.
	 *
	 * @param string $dev_token   Developer token for Google Ads API.
	 * @param int    $manager_id  Manager (MCC) customer ID.
	 * @param string $customer_id Target customer account ID to fetch metrics from.
	 *
	 * @return array|false Returns an associative array of metrics (clicks, impressions, CTR, avg CPC, conversions)
	 *                     or false on failure.
	 */
	public function get_ads_data( $dev_token, $manager_id, $customer_id ) {
		if ( ! file_exists( $this->service_account_file_path ) ) {
			error_log( 'Google Ads credentials file not found: ' . $this->service_account_file_path );
			return false;
		}

		try {
			$jsonKey = json_decode( file_get_contents( $this->service_account_file_path ), true );

			$credentials = new ServiceAccountCredentials(
				['https://www.googleapis.com/auth/adwords'],
				$jsonKey
			);

			$googleAdsClient = ( new GoogleAdsClientBuilder() )
				->withDeveloperToken($dev_token)
				->withLoginCustomerId($manager_id)
				->withOAuth2Credential($credentials)
				->build();

			$query = "
			SELECT
				metrics.clicks,
				metrics.impressions,
				metrics.ctr,
				metrics.average_cpc,
				metrics.conversions
			FROM customer
			WHERE segments.date DURING LAST_7_DAYS
		";

			$request = new SearchGoogleAdsRequest(
				[
					'customer_id' => $customer_id,
					'query'       => $query,
				]
			);

			$response = $googleAdsClient->getGoogleAdsServiceClient()->search( $request );

			$data = [
				'clicks'       => 0,
				'impressions'  => 0,
				'ctr'          => 0,
				'avg_cpc'      => 0,
				'conversions'  => 0,
			];

			foreach ( $response->iterateAllElements() as $row ) {
				$metrics = $row->getMetrics();
				$data['clicks']       += $metrics->getClicks();
				$data['impressions']  += $metrics->getImpressions();
				$data['ctr']          += $metrics->getCtr() * 100;
				$data['avg_cpc']      += $metrics->getAverageCpc()->getMicros() / 1000000;
				$data['conversions']  += $metrics->getConversions();
			}

			return $data;

		} catch ( ApiException $e ) {
			error_log('Google Ads API Error: ' . $e->getMessage());
			return false;
		}
	}
}
// @codingStandardsIgnoreEnd
