<?php
/**
 * ShortPixel
 *
 * This class customizes the functionality of the ShortPixel plugin in WordPress.
 * It automates the configuration of default settings for the ShortPixel Image Optimizer plugin
 * to ensure it integrates seamlessly with the site's workflow. The class also ensures the automation
 * runs only once by using a flag stored in the database.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class ShortPixel
 *
 * This class extends the StrategySuite\Module and manages customizations for
 * the ShortPixel Image Optimizer plugin, including setting default optimization options
 * and ensuring the automation process runs only once.
 *
 * @package StrategySuite
 */
class ShortPixel extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * Checks if the ShortPixel Image Optimizer plugin is active before registering the module.
	 *
	 * @return bool True if the ShortPixel plugin is active, false otherwise.
	 */
	public function can_register() {
		return is_plugin_active( 'shortpixel-image-optimiser/wp-shortpixel.php' );
	}

	/**
	 * Register hooks for customizing ShortPixel plugin behavior.
	 *
	 * This method registers actions to:
	 * - Add default configuration options for the ShortPixel plugin.
	 * - Ensure these options are only applied once to prevent duplicate settings.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'add_default_options' ] );
	}

	/**
	 * Add default settings for ShortPixel Image Optimizer plugin.
	 *
	 * This method checks if the automation process has already been run (using the
	 * 'lg-short-pixel-automation-run' option). If not, it applies a predefined set of
	 * options to configure the ShortPixel plugin, including image compression settings,
	 * resizing options, backup preferences, and CDN settings.
	 *
	 * Once the settings are applied, the method updates the 'lg-short-pixel-automation-run'
	 * option to prevent the settings from being applied again.
	 *
	 * @return void
	 */
	public function add_default_options() {
		if ( 1 == get_option( 'lg-short-pixel-automation-run' ) ) {
			return;
		}

		$options = array(
			'mediaLibraryViewMode' => 1,
			'quotaExceeded' => 0,
			'compressionType' => 2,
			'processThumbnails' => true,
			'useSmartcrop' => true,
			'CMYKtoRGBconversion' => true,
			'createWebp' => true,
			'createAvif' => true,
			'deliverWebp' => 1,
			'optimizeRetina' => true,
			'optimizeUnlisted' => true,
			'backupImages' => true,
			'resizeImages' => true,
			'resizeType' => 'outer',
			'resizeWidth' => 1920,
			'resizeHeight' => 1080,
			'siteAuthUser' => '',
			'siteAuthPass' => '',
			'autoMediaLibrary' => true,
			'doBackgroundProcess' => true,
			'optimizePdfs' => true,
			'excludePatterns' => [],
			'png2jpg' => 1,
			'excludeSizes' => [],
			// 'currentVersion' => '6.1.1',
			'showCustomMedia' => true,
			'cloudflareZoneID' => '',
			'cloudflareToken' => '',
			'includeNextGen' => false,
			'hasCustomFolders' => 0,
			'httpProto' => 'https',
			'downloadProto' => '',
			'redirectedSettings' => 3,
			'unlistedCounter' => 0,
			'exif' => 0,
			'exif_ai' => 0,
			'CDNDomain' => 'https://spcdn.shortpixel.ai/spio',
			'smartCropIgnoreSizes' => false,
			'useCDN' => false,
		);
		update_option( 'spio_settings', $options );

		// Set the 'lg-short-pixel-automation-run' option to 1 to indicate the options have been added.
		update_option( 'lg-short-pixel-automation-run', 1 );
	}
}
