<?php
/**
 * RSS
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class RSS
 *
 * Handles disabling RSS feeds and related rewrite rules in WordPress.
 */
class RSS extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * Checks if the `ENABLE_RSS` constant is defined and set to `false`,
	 * which determines whether the module should be active.
	 *
	 * @return bool Returns `true` if RSS feeds should be disabled, otherwise `false`.
	 */
	public function can_register() {
		return defined( 'ENABLE_RSS' ) && ! ENABLE_RSS;
	}

	/**
	 * Register hooks for disabling RSS feeds.
	 *
	 * This method adds actions to remove rewrite rules and filter feed requests.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'init', [ $this, 'remove_rewrite_rules' ] );
		add_action( 'template_redirect', [ $this, 'filter_feeds' ] );
	}

	/**
	 * Filters RSS feed requests.
	 *
	 * If a feed request is detected, it redirects the request to prevent access.
	 *
	 * @return void
	 */
	public function filter_feeds() {
		if ( ! is_feed() || is_404() ) {
			return;
		}

		$this->redirect_feed();
	}

	/**
	 * Redirects feed requests with a 410 Gone response.
	 *
	 * This method sets the appropriate HTTP status code and headers to indicate
	 * that RSS feeds are no longer available.
	 *
	 * @return void
	 */
	private function redirect_feed() {
		global $wp_rewrite, $wp_query;

		$wp_query->is_feed = false;
		$wp_query->set_404();
		status_header( 410 );
		// Override the xml+rss header set by WP in send_headers
		header( 'Content-Type: ' . get_option( 'html_type' ) . '; charset=' . get_option( 'blog_charset' ) );
	}

	/**
	 * Registers a filter to disable rewrite rules for feeds.
	 *
	 * This ensures that WordPress does not generate feed-related rewrite rules.
	 *
	 * @return void
	 */
	public function remove_rewrite_rules() {
		add_filter( 'rewrite_rules_array', [ $this, 'disable_feed_rewrites' ] );
	}

	/**
	 * Disables feed-related rewrite rules.
	 *
	 * This method removes any rewrite rules that contain 'feed=' to
	 * prevent WordPress from handling feed requests.
	 *
	 * @param array $rules The existing rewrite rules.
	 *
	 * @return array The modified rewrite rules with feed rules removed.
	 */
	public function disable_feed_rewrites( $rules ) {
		foreach ( $rules as $rule => $rewrite ) {
			if ( strpos( $rewrite, 'feed=' ) !== false ) {
				unset( $rules[ $rule ] );
			}
		}

		return $rules;
	}
}
