<?php
/**
 * FormBuilder
 *
 * This class handles the automated creation of Gravity Forms forms based on
 * predefined JSON templates. It checks for the existence of specific forms and
 * creates them if they don't already exist, leveraging the Gravity Forms API.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use GFAPI;

/**
 * Class FormBuilder
 *
 * This class extends the StrategySuite\Module and manages the setup of forms
 * from templates using the Gravity Forms plugin.
 */
class FormBuilder extends \StrategySuite\Module {

	/**
	 * Directory path where form templates are stored.
	 */
	private const FORM_TEMPLATES_DIR = STRATEGY_SUITE_INC . 'form-templates/';

	/**
	 * Determines if the module can be registered.
	 *
	 * This method checks if the Gravity Forms plugin is active before allowing
	 * the registration of the module.
	 *
	 * @return bool True if the Gravity Forms plugin is active, false otherwise.
	 */
	public function can_register() {
		return is_plugin_active( 'gravityforms/gravityforms.php' );
	}

	/**
	 * Registers the module by hooking into WordPress admin initialization.
	 *
	 * This method triggers the form creation process during the admin_init action.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'create_form_builder' ] );
	}

	/**
	 * Factory method for initiating the form creation process.
	 *
	 * Retrieves all form templates and triggers the form building process
	 * if templates are found.
	 *
	 * @return void
	 */
	public function create_form_builder() {
		$templates = $this->get_all_templates();

		if ( is_array( $templates ) && 0 < count( $templates ) ) {
			$this->build_forms( $templates );
		}
	}

	/**
	 * Retrieves all form templates from the templates directory.
	 *
	 * @return array An array of form templates as objects.
	 */
	private function get_all_templates() {
		$template_files = glob( self::FORM_TEMPLATES_DIR . '*.json' );
		$templates = array();

		foreach ( $template_files as $file ) {
			$template = $this->retrieve_form_template( basename( $file ) );
			if ( ! is_null( $template ) ) {
				$templates[] = $template;
			}
		}

		return $templates;
	}

	/**
	 * Builds forms from provided templates if they do not already exist.
	 *
	 * @param array $form_templates An array of form templates.
	 *
	 * @return void
	 */
	private function build_forms( $form_templates ) {
		if ( ! is_array( $form_templates ) ) {
			return;
		}

		foreach ( $form_templates as $template ) {
			if ( ! $this->form_exists( $template ) ) {
				$this->create_form_from_template( $template );
			}
		}
	}

	/**
	 * Retrieves a specific form template from the templates directory.
	 *
	 * @param string $filename The filename of the form template to retrieve.
	 *
	 * @return object|null The form template object, or null if the file does not exist or is invalid.
	 */
	private function retrieve_form_template( $filename ) {
		if ( file_exists( self::FORM_TEMPLATES_DIR . $filename ) ) {
			$contents = file_get_contents( self::FORM_TEMPLATES_DIR . $filename );
			$decoded = json_decode( $contents );

			return $decoded->{0};
		}

		return null;
	}

	/**
	 * Creates a form in Gravity Forms using the provided template.
	 *
	 * @param object|array $template The form template to create.
	 *
	 * @return void
	 */
	private function create_form_from_template( $template ) {
		if ( ! is_array( $template ) ) {
			$template = $this->convert_to_array( $template );
		}

		GFAPI::add_form( $template );
	}

	/**
	 * Checks if a form already exists in Gravity Forms based on the template title.
	 *
	 * @param object $template The form template to check against existing forms.
	 *
	 * @return bool True if the form exists, false otherwise.
	 */
	private function form_exists( $template ) {

		$forms = GFAPI::get_forms( null );
		$match = false;

		foreach ( $forms as $index => $form ) {
			if ( $template->title == $form['title'] || ( 'Lead Magnet' == $template->title && ( 'Newsletter Form' == $form['title'] || 'Lead Magnet Form' == $form['title'] ) ) || ( 'Contact Us' == $template->title && 'Contact Form' == $form['title'] ) ) {
				$match = true;
				break;
			}
		}

		return $match;
	}

	/**
	 * Recursively converts objects to arrays.
	 *
	 * @param mixed $data The data to convert.
	 *
	 * @return mixed The converted data as an array.
	 */
	private function convert_to_array( $data ) {

		if ( ! is_array( $data ) && is_object( $data ) ) {
			$data = (array) $data;
		}

		foreach ( $data as $key => $value ) {

			if ( is_array( $value ) || is_object( $value ) ) {
				$data[ $key ] = $this->convert_to_array( $value );
			}
		}

		return $data;
	}
}
