<?php
/**
 * Amelia
 *
 * This class handles the integration with the Amelia Booking plugin and Groundhogg CRM for automating
 * appointment and event handling.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class Amelia
 *
 * This class extends the StrategySuite\Module and manages the setup of cron jobs
 * to delete old post revisions in a WordPress site.
 */
class Amelia extends \StrategySuite\Module {


	/**
	 * The base URL for the Pipedrive API.
	 *
	 * This variable stores the base URL used for making API requests to Pipedrive. The value is typically
	 * the Pipedrive API URL, which is 'https://api.pipedrive.com/v1'.
	 *
	 * @var string
	 */
	private $pipedrive_base;

	/**
	 * The API token used for authenticating requests to Pipedrive.
	 *
	 * This variable holds the API token that is used to authenticate requests made to the Pipedrive API.
	 * The value is either taken from the constant `AMELIA_PIPEDRIVE_API_TOKEN` or from the environment
	 * variable `AMELIA_PIPEDRIVE_API_TOKEN` if the constant is not defined.
	 *
	 * @var string
	 */
	private $pipedrive_token;

	/**
	 * Amelia constructor.
	 *
	 * Initializes the Pipedrive base URL and API token. The token is read from the
	 * AMELIA_PIPEDRIVE_API_TOKEN constant, or from the AMELIA_PIPEDRIVE_API_TOKEN
	 * environment variable when the constant is not defined.
	 *
	 * @return void
	 */
	public function __construct() {
		$this->pipedrive_base = 'https://api.pipedrive.com/v1';
		$this->pipedrive_token = defined( 'AMELIA_PIPEDRIVE_API_TOKEN' )
			? AMELIA_PIPEDRIVE_API_TOKEN
			: getenv( 'AMELIA_PIPEDRIVE_API_TOKEN' );
	}

	/**
	 * Determine if the module can be registered.
	 *
	 * @return bool
	 */
	public function can_register() {
		return is_plugin_active( 'wp-amelia/ameliabooking.php' );
	}

	/**
	 * Register hooks for automating the deletion of old revisions and integrating with plugins.
	 *
	 * This method registers filters and actions for integrating with Advanced Custom Fields (ACF),
	 * Groundhogg CRM, and handling custom block patterns based on the active theme.
	 *
	 * @return void
	 */
	public function register() {
		if ( class_exists( 'ACF' ) && class_exists( 'AmeliaBooking\Plugin' ) ) {
			add_action( 'acf/save_post', [ $this, 'add_amelia_defaults' ], 30 );
		}

		add_filter( 'allowed_block_types_all', [ $this, 'amelia_deregister_blocks' ], 20, 2 );
		add_filter( 'amelia_change_ssl_settings', [ $this, 'amelia_support_all_db' ], 20, 1 );

		$theme = wp_get_theme();
		if ( 'leadgen' === $theme->stylesheet ) {
			add_action( 'init', [ $this, 'register_amelia_patterns' ] );
		}

		if ( is_plugin_active( 'groundhogg/groundhogg.php' ) ) {
			add_action( 'AmeliaAppointmentBookingCanceled', [ $this, 'amelia_gh_bm_appointment_canceled' ], 10, 3 );
			add_action( 'AmeliaAppointmentBookingAdded', [ $this, 'amelia_gh_bm_appointment_added' ], 10, 3 );
			add_action( 'AmeliaAppointmentBookingStatusUpdated', [ $this, 'amelia_gh_bm_appointment_status_update' ], 10, 3 );
			add_action( 'AmeliaEventBookingCanceled', [ $this, 'amelia_gh_bm_event_canceled' ], 10, 3 );
			add_action( 'AmeliaEventBookingAdded', [ $this, 'amelia_gh_bm_event_added' ], 10, 3 );
			add_action( 'AmeliaEventBookingStatusUpdated', [ $this, 'amelia_gh_bm_event_status_update' ], 10, 3 );
		}

		add_action( 'amelia_after_outlook_calendar_event_added', [ $this, 'amelia_outlook_calendar_add_teams' ], 10, 3 );

		if ( ! empty( $this->pipedrive_token ) ) {
			add_action( 'amelia_before_booking_added', [ $this, 'send_booking_to_pipedrive' ], 10, 1 );
		}
	}

	/**
	 * Adds a tag to a Groundhogg contact and triggers a benchmark.
	 *
	 * @param array  $bookings The booking details from Amelia.
	 * @param string $tag The tag to apply in Groundhogg.
	 *
	 * @return void
	 */
	public function add_amelia_gh_tag_bench( $bookings, $tag ) {

		$email = isset( $bookings[0]['customer']['email'] ) ? $bookings[0]['customer']['email'] : '';
		$first_name = isset( $bookings[0]['customer']['firstName'] ) ? $bookings[0]['customer']['firstName'] : '';
		$last_name = isset( $bookings[0]['customer']['lastName'] ) ? $bookings[0]['customer']['lastName'] : '';

		$contact = \Groundhogg\get_contactdata( $email );

		// If the commenter is not already a contact, let's create a new contact record for them
		if ( ! \Groundhogg\is_a_contact( $contact ) ) {

			$contact = new \Groundhogg\Contact(
				[
					'email' => $email,
					'first_name' => $first_name,
					'last_name' => $last_name,
				]
			);
		}

		$contact->apply_tag( [ $tag ] );

		// Call this function with your call name to trigger the benchmark
		\Groundhogg\do_plugin_api_benchmark( $tag, $email, false );
	}

	/**
	 * Handle appointment cancellation events in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_appointment_canceled( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaAppointmentBookingCanceled' );
	}

	/**
	 * Handle new appointment bookings in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_appointment_added( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaAppointmentBookingAdded' );
	}

	/**
	 * Handle appointment status updates in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_appointment_status_update( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaAppointmentBookingStatusUpdated' );
	}

	/**
	 * Handle event cancellations in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_event_canceled( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaEventBookingCanceled' );
	}

	/**
	 * Handle new event bookings in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_event_added( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaEventBookingAdded' );
	}

	/**
	 * Handle event status updates in Groundhogg.
	 *
	 * @param mixed $reservation The reservation details.
	 * @param array $bookings The bookings related to the event.
	 * @param mixed $container Additional data.
	 */
	public function amelia_gh_bm_event_status_update( $reservation, $bookings, $container ) {
		$this->add_amelia_gh_tag_bench( $bookings, 'AmeliaEventBookingStatusUpdated' );
	}

	/**
	 * Adds default settings to Amelia after saving ACF options.
	 *
	 * @return void
	 */
	public function add_amelia_defaults() {
		$screen = get_current_screen();
		if ( ! strpos( $screen->id, 'site_settings' ) ) {
			return;
		}

		$amelia_option_settings = ( json_decode( get_option( 'amelia_settings' ) ) );

		$amelia_option_settings->activation->purchaseCodeStore = WP_AMELIA_LICENSE;
		$amelia_option_settings->general->gMapApiKey = GOOGLE_API_TOKEN;

		$design_settings = get_field( 'design_settings', 'options' );
		$business_information = get_field( 'business_information', 'options' );
		$logo_id = isset( $design_settings['logo'] ) ? $design_settings['logo'] : '';

		$amelia_option_settings->company->pictureFullPath = wp_get_attachment_image_url( $logo_id, 'full' );
		$amelia_option_settings->company->pictureThumbPath = wp_get_attachment_image_url( $logo_id ) ? wp_get_attachment_image_url( $logo_id ) : wp_get_attachment_image_url( $logo_id, 'full' );
		$amelia_option_settings->company->name = get_bloginfo( 'name' );
		$amelia_option_settings->company->address = isset( $business_information['address']['address'] ) ? $business_information['address']['address'] : '';
		$amelia_option_settings->company->phone = isset( $business_information['phone'] ) ? $business_information['phone'] : '';
		$amelia_option_settings->company->email = isset( $business_information['email'] ) ? $business_information['email'] : '';
		$amelia_option_settings->company->website = get_bloginfo( 'url' );

		$amelia_option_settings->notifications->mailService = 'wp_mail';
		$amelia_option_settings->notifications->senderName = get_bloginfo( 'name' );
		$amelia_option_settings->notifications->senderEmail = determine_sender_email();

		$amelia_option_settings = json_encode( $amelia_option_settings );

		update_option( 'amelia_settings', $amelia_option_settings );
	}

	/**
	 * Adds Microsoft Teams meetings to Amelia bookings via Outlook Calendar API.
	 *
	 * @param object $event The event object from Outlook Calendar.
	 * @param array  $appointment The appointment details from Amelia.
	 * @param array  $provider Provider information, including access tokens.
	 *
	 * @return void
	 */
	public function amelia_outlook_calendar_add_teams( $event, $appointment, $provider ) {
		if ( is_array( $appointment['service'] ) && isset( $appointment['service']['description'] ) && ! str_contains( $appointment['service']['description'], 'Teams' ) ) {
			return;
		}

		$access_token = $provider['outlookCalendar']['token'];
		$event_id = $event->getId();
		$data = json_decode( $access_token, true );

		$headers = array(
			'Authorization' => 'bearer ' . $data['access_token'],
			'Content-Type' => 'application/json',
		);

		$body = wp_json_encode(
			array(
				'isOnlineMeeting' => true,
				'onlineMeetingProvider' => 'teamsForBusiness',
			)
		);

		$args = array(
			'method' => 'PATCH',
			'headers' => $headers,
			'body' => $body,
		);
		$response = wp_remote_request( "https://graph.microsoft.com/v1.0/me/events/{$event_id}", $args );
	}

	/**
	 * Deregisters specific Amelia blocks from the block editor.
	 *
	 * @param array $allowed_block_types Currently allowed block types.
	 * @param array $editor_context Contextual information about the block editor.
	 *
	 * @return array Modified list of allowed block types.
	 */
	public function amelia_deregister_blocks( $allowed_block_types, $editor_context ) {

		$registered_block_slugs = \WP_Block_Type_Registry::get_instance()->get_all_registered();

		unset( $registered_block_slugs['amelia/booking-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/catalog-booking-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/catalog-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/customer-cabinet-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/employee-cabinet-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/events-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/events-list-booking-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/search-gutenberg-block'] );
		unset( $registered_block_slugs['amelia/step-booking-gutenberg-block'] );

		$allowed_block_types = array_keys( $registered_block_slugs );

		return $allowed_block_types;
	}

	/**
	 * Modifies Amelia SSL settings to support custom database configurations.
	 *
	 * @param array $amelia_ssl_settings The current SSL settings for Amelia.
	 *
	 * @return array Modified SSL settings.
	 */
	public function amelia_support_all_db( $amelia_ssl_settings ) {
		$amelia_ssl_settings['enable'] = defined( 'DB_SSL' ) ? DB_SSL : $amelia_ssl_settings['enable'];
		$amelia_ssl_settings['ca'] = defined( 'DB_SSL' ) ? DB_SSL : $amelia_ssl_settings['ca'];
		$amelia_ssl_settings['verify_cert'] = defined( 'MYSQL_CLIENT_FLAGS' ) ? false : $amelia_ssl_settings['verify_cert'];

		return $amelia_ssl_settings;
	}

	/**
	 * Registers custom block patterns for Amelia bookings.
	 *
	 * @return void
	 */
	public function register_amelia_patterns() {

		register_block_pattern_category(
			'bookings',
			[ 'label' => __( 'Bookings', 'lead-gen' ) ]
		);

		if ( function_exists( 'register_block_pattern' ) ) {
			$block_patterns = array(
				[
					'slug' => 'booking-step-by-step',
					'title' => 'Booking Step By Step',
					'category' => 'bookings',
				],
				[
					'slug' => 'event-list',
					'title' => 'Event List',
					'category' => 'bookings',
				],
			);

			$path = STRATEGY_SUITE_INC . 'patterns/amelia/';

			foreach ( $block_patterns as $block_pattern ) {
				register_block_pattern(
					$block_pattern['category'] . '/' . $block_pattern['slug'],
					array(
						'title' => $block_pattern['title'],
						'content' => file_get_contents( $path . $block_pattern['slug'] . '.html' ),
						'categories' => [ $block_pattern['category'] ],
					)
				);
			}
		}
	}

	/**
	 * Send an Amelia booking payload to Pipedrive.
	 *
	 * Validates and parses the Amelia appointment, finds or creates the Organization,
	 * finds or creates the Person and links it to the Organization, sets the Person job title
	 * using your configured field, sets the Organization website, determines the Lead channel
	 * from UTM source and resolves the option ID, resolves the owner from the provider email,
	 * creates the Lead, and attaches a Note with any provided meeting notes.
	 *
	 * This method logs basic errors and surfaces exceptions thrown by HTTP or API failures.
	 *
	 * @param array|string $appointment Amelia appointment payload. In normal operation this is an array from the Amelia hook,
	 *                                  in test mode you may ignore the parameter since a local JSON string is used.
	 *
	 * @return void
	 *
	 * @throws \Exception When Pipedrive API calls fail or when the payload is invalid before the try catch block handles it.
	 */
	public function send_booking_to_pipedrive( $appointment ) {
		try {
			if ( empty( $appointment ) || empty( $appointment['bookings'] ) || ! is_array( $appointment['bookings'] ) ) {
				throw new \Exception( 'Amelia payload missing bookings' );
			}

			$utm_source = $this->get_cookie_value( 'utm_source' );
			$service_id = isset( $appointment['serviceId'] ) ? $appointment['serviceId'] : null;
			$lead_channel = $this->map_source_channel( $utm_source );

			if ( $service_id ) {
				$service = $this->get_amelia_service_by_id( $service_id );

				if ( isset( $service['categoryId'] ) ) {
					$category_id = $service['categoryId'];
					$category = $this->get_amelia_category_by_id( $category_id );

					if ( isset( $category['name'] ) && stripos( $category['name'], 'Leadium' ) !== false ) {
						$lead_channel = 'Sales Outreach';
					}

					// Don't send room reservations to Pipedrive.
					if ( isset( $category['name'] ) && 'Internal Room Reservations' === $category['name'] ) {
						return;
					}
				}
			}

			$lead_channel_id = $this->get_custom_field_option( 'Source channel', $lead_channel );

			$booking = $appointment['bookings'][0];
			$customer = isset( $booking['customer'] ) ? $booking['customer'] : [];
			$email = isset( $customer['email'] ) ? trim( $customer['email'] ) : '';
			$first_name = isset( $customer['firstName'] ) ? trim( $customer['firstName'] ) : '';
			$last_name = isset( $customer['lastName'] ) ? trim( $customer['lastName'] ) : '';
			$full_name = trim( $first_name . ' ' . $last_name );
			$customer_phone = isset( $customer['phone'] ) ? trim( $customer['phone'] ) : '';

			$company_name = $this->amelia_get_custom_field_by_label( $booking, 'Company Name' );
			if ( empty( $company_name ) ) {
				$company_name = $this->org_name_from_email_domain( $email );
				if ( empty( $company_name ) ) {
					$company_name = $full_name;
				}
			}

			$title_role = $this->amelia_get_custom_field_by_label( $booking, 'Title/Role' );
			if ( empty( $title_role ) ) {
				$title_role = '';
			}

			$company_website = $this->amelia_get_custom_field_by_label( $booking, 'Company Website' );
			if ( empty( $company_website ) ) {
				$company_website = '';
			}

			$notes = $this->amelia_get_custom_field_by_label( $booking, 'Please share anything that will help us prepare for our meeting.' );
			if ( empty( $notes ) ) {
				$notes = '';
			}

			// Always create or find Organization
			$org_id = $this->pipedrive_find_org_by_name( $company_name );
			if ( ! $org_id ) {
				$org_payload = [
					'name' => $company_name,
				];

				if ( ! empty( $company_website ) ) {
					if ( ! preg_match( '~^https?://~i', $company_website ) ) {
						$company_website = 'https://' . $company_website;
					}

					$org_payload['website'] = $company_website;
				}

				$org = $this->send_pipedrive_request( 'POST', 'organizations', $org_payload );

				if ( empty( $org['data']['id'] ) ) {
					error_log( '[Amelia→Pipedrive] Failed to create Pipedrive organization' );
					throw new \Exception( 'Failed to create Pipedrive organization' );
				}

				$org_id = (int) $org['data']['id'];
			}

			// Always create or find Person, link to Org
			$person_id = null;
			if ( $email ) {
				$person_id = $this->pipedrive_find_person_by_email( $email );
			}

			if ( ! $person_id ) {
				$person_payload = [
					'name' => $full_name,
					'org_id' => $org_id,
				];
				if ( empty( $person_payload['name'] ) ) {
					$person_payload['name'] = $email ? $email : 'Unknown Person';
				}
				if ( $email ) {
					$person_payload['email'] = $email;
				}
				if ( ! empty( $customer_phone ) ) {
					$person_payload['phone'] = $customer_phone;
				}
				if ( ! empty( $title_role ) ) {
					$title_id = $this->get_custom_field_option( 'Job Title', null, 'person' );
					$person_payload[ $title_id ] = $title_role;
				}
				$person = $this->send_pipedrive_request( 'POST', 'persons', $person_payload );
				if ( empty( $person['data']['id'] ) ) {
					throw new \Exception( 'Failed to create Pipedrive person' );
				}
				$person_id = (int) $person['data']['id'];
			} else {
				$this->send_pipedrive_request( 'PUT', "persons/{$person_id}", [ 'org_id' => $org_id ] );
			}

			$provider_id = isset( $appointment['providerId'] ) ? $appointment['providerId'] : null;
			$provider_email = null;
			if ( $provider_id ) {
				$provider = $this->get_amelia_provider_by_id( $provider_id );
				$provider_email = $provider['email'];
			}

			$owner_id = $provider_email ? $this->pipedrive_find_user_id_by_email( $provider_email ) : null;

			// Build Lead payload
			$lead_title = $company_name;
			if ( empty( $lead_title ) ) {
				$lead_title = $full_name ? $full_name : $email;
				if ( empty( $lead_title ) ) {
					$lead_title = 'New Lead';
				}
			}
			$lead_payload = [
				'title' => $lead_title,
				'organization_id' => $org_id,
				'person_id' => $person_id,
				'channel' => $lead_channel_id,
			];
			if ( $owner_id ) {
				$lead_payload['owner_id'] = $owner_id;
			}

			// Create the Lead
			$lead = $this->send_pipedrive_request( 'POST', 'leads', $lead_payload );
			$lead_id = $lead['data']['id'];

			if ( empty( $lead_id ) ) {
				throw new \Exception( 'Failed to create Pipedrive lead' );
			}

			if ( ! empty( $notes ) && ! empty( $lead_id ) ) {
				$this->send_pipedrive_request(
					'POST',
					'notes',
					[
						'lead_id' => $lead_id,
						'content' => $notes,
					]
				);
			}
		} catch ( \Throwable $e ) {
			error_log( '[Amelia→Pipedrive] Error: ' . $e->getMessage() );
		}
	}

	/**
	 * Retrieves a custom field value by label from an Amelia booking.
	 *
	 * This method handles retrieving a custom field value by label from an Amelia booking. It handles both
	 * flat arrays of field objects and associative arrays of fields keyed by ID. It also decodes custom fields
	 * if they are stored as a JSON string.
	 *
	 * @param array  $booking The Amelia booking details.
	 * @param string $target_label The label of the custom field to retrieve.
	 *
	 * @return string|null The value of the custom field, or null if not found.
	 */
	private function amelia_get_custom_field_by_label( array $booking, $target_label ) {
		if ( empty( $booking['customFields'] ) ) {
			return null;
		}
		$raw = $booking['customFields'];

		// Amelia often stores customFields as a JSON string
		if ( is_string( $raw ) ) {
			$decoded = json_decode( $raw, true );
			if ( json_last_error() === JSON_ERROR_NONE ) {
				$raw = $decoded;
			}
		}

		// Case 1, array of objects with label and value
		if ( is_array( $raw ) && array_is_list( $raw ) ) {
			foreach ( $raw as $field ) {
				if ( ! empty( $field['label'] ) && strcasecmp( $field['label'], $target_label ) === 0 ) {
					return isset( $field['value'] ) ? trim( (string) $field['value'] ) : null;
				}
			}
		}

		// Case 2, associative array keyed by id, each value may have label and value
		if ( is_array( $raw ) && ! array_is_list( $raw ) ) {
			foreach ( $raw as $field ) {
				if ( is_array( $field ) && ! empty( $field['label'] ) && strcasecmp( $field['label'], $target_label ) === 0 ) {
					return isset( $field['value'] ) ? trim( (string) $field['value'] ) : null;
				}
			}
		}

		return null;
	}

	/**
	 * Search for a Pipedrive Person by exact email using the Persons search endpoint.
	 *
	 * This method searches for a Person in Pipedrive by their email address. It uses the search endpoint
	 * to find a person and returns their ID if found.
	 *
	 * @param string $email The email address to search for.
	 *
	 * @return int|null The ID of the person if found, null otherwise.
	 */
	private function pipedrive_find_person_by_email( $email ) {
		// v1 search supports term, fields, exact_match
		$url = add_query_arg(
			[
				'term' => $email,
				'fields' => 'email',
				'exact_match' => 'true',
				'limit' => 1,
				'start' => 0,
				'api_token' => $this->pipedrive_token,
			],
			"{$this->pipedrive_base}/persons/search"
		);

		$res = wp_remote_get(
			$url,
			[
				'timeout' => 15,
				'headers' => [
					'Accept' => 'application/json',
				],
			]
		);

		if ( is_wp_error( $res ) ) {
			return null;
		}

		$code = wp_remote_retrieve_response_code( $res );
		$body = json_decode( wp_remote_retrieve_body( $res ), true );

		if ( 200 !== $code || empty( $body['data']['items'] ) ) {
			return null;
		}

		$item = $body['data']['items'][0];

		if ( ! empty( $item['item']['id'] ) ) {
			return (int) $item['item']['id'];
		}

		return null;
	}

	/**
	 * Find a Pipedrive Organization by exact name.
	 *
	 * This method searches for an Organization in Pipedrive by its exact name and returns its ID if found.
	 *
	 * @param string $name The name of the organization to search for.
	 *
	 * @return int|null The ID of the organization if found, null otherwise.
	 */
	private function pipedrive_find_org_by_name( $name ) {
		$url = add_query_arg(
			[
				'term' => $name,
				'fields' => 'name',
				'exact_match' => 'true',
				'limit' => 1,
				'start' => 0,
				'api_token' => $this->pipedrive_token,
			],
			"{$this->pipedrive_base}/organizations/search"
		);

		$res = wp_remote_get(
			$url,
			[
				'timeout' => 15,
				'headers' => [ 'Accept' => 'application/json' ],
			]
		);

		if ( is_wp_error( $res ) ) {
			return null;
		}

		$code = wp_remote_retrieve_response_code( $res );
		$body = json_decode( wp_remote_retrieve_body( $res ), true );

		if ( 200 !== $code || empty( $body['data']['items'] ) ) {
			return null;
		}

		$item = $body['data']['items'][0];

		return ! empty( $item['item']['id'] ) ? (int) $item['item']['id'] : null;
	}

	/**
	 * Sends a request to the Pipedrive API.
	 *
	 * This helper function sends an HTTP request to the Pipedrive API using the specified HTTP method.
	 * It automatically includes the API token in the query parameters and handles the response.
	 *
	 * @param string     $method The HTTP method to use (e.g., 'GET', 'POST', 'PUT', etc.).
	 * @param string     $endpoint The API endpoint to send the request to.
	 * @param array|null $payload The request body for POST and PUT requests, or null for GET requests.
	 *
	 * @return array The response body from Pipedrive.
	 *
	 * @throws \Exception If an error occurs during the request or if the response code is not successful.
	 */
	private function send_pipedrive_request( $method, $endpoint, $payload = null ) {
		$url = trailingslashit( $this->pipedrive_base ) . ltrim( $endpoint, '/' );

		if ( false === strpos( $url, 'api_token=' ) && $this->pipedrive_token ) {
			$url = add_query_arg( [ 'api_token' => $this->pipedrive_token ], $url );
		}

		$args = [
			'method' => $method,
			'timeout' => 20,
			'headers' => [
				'Accept' => 'application/json',
				'Content-Type' => 'application/json',
			],
		];

		if ( null !== $payload ) {
			$args['body'] = wp_json_encode( $payload );
		}

		$res = wp_remote_request( $url, $args );

		if ( is_wp_error( $res ) ) {
			error_log( '[Amelia→Pipedrive] HTTP error: ' . esc_html( $res->get_error_message() ) );
			throw new \Exception( 'HTTP error: ' . esc_html( $res->get_error_message() ) );
		}

		$code = wp_remote_retrieve_response_code( $res );
		$body = json_decode( wp_remote_retrieve_body( $res ), true );

		if ( $code < 200 || $code >= 300 ) {
			$msg = isset( $body['error'] ) ? $body['error'] : 'Unexpected response from Pipedrive';
			error_log( esc_html( "[Amelia→Pipedrive] Pipedrive API error, status {$code}: {$msg}" ) );
			throw new \Exception( esc_html( "Pipedrive API error, status {$code}: {$msg}" ) );
		}

		return $body;
	}

	/**
	 * Return a sanitized cookie value or null.
	 *
	 * This method retrieves and sanitizes a cookie value for the given cookie name.
	 *
	 * @param string $name The name of the cookie.
	 *
	 * @return string|null The sanitized cookie value, or null if the cookie is not set or empty.
	 */
	private function get_cookie_value( string $name ): ?string {
		if ( isset( $_COOKIE[ $name ] ) && '' !== $_COOKIE[ $name ] ) {
			return sanitize_text_field( wp_unslash( $_COOKIE[ $name ] ) );
		}

		return null;
	}

	/**
	 * Maps the UTM source to a Pipedrive source channel label.
	 *
	 * This method maps the UTM source parameter to a predefined Pipedrive source channel.
	 * It categorizes sources as Google Ads, Email Marketing, Organic Search, or defaults to "(None)".
	 *
	 * @param string|null $utm_source The value of the utm_source query parameter.
	 *
	 * @return string The corresponding Pipedrive source channel label.
	 */
	private function map_source_channel( ?string $utm_source ): string {
		$src = strtolower( trim( (string) $utm_source ) );
		if ( 'gads' === $src || 'ppc' === $src ) {
			return 'Google Ads';
		}

		if ( 'email' === $src ) {
			return 'Email Marketing';
		}

		if ( '' === $src || null === $utm_source ) {
			return 'Organic Search';
		}

		return '(None)';
	}

	/**
	 * Tries to infer an organization name from an email address.
	 *
	 * This method extracts the domain part from an email address and uses it to generate an organization name.
	 * For example, jane@acme.com becomes Acme.
	 *
	 * @param string|null $email The email address to extract the organization name from.
	 *
	 * @return string|null The inferred organization name, or null if the email is invalid.
	 */
	private function org_name_from_email_domain( ?string $email ): ?string {
		if ( ! $email || strpos( $email, '@' ) === false ) {
			return null;
		}

		$domain = substr( strrchr( $email, '@' ), 1 );
		if ( ! $domain ) {
			return null;
		}

		$host = preg_replace( '/^www\./i', '', $domain );
		$parts = explode( '.', $host );
		$base = $parts[0] ?? '';
		if ( '' === $base ) {
			return null;
		}

		return ucwords( preg_replace( '/[-_]+/', ' ', $base ) );
	}

	/**
	 * Find a Pipedrive user by email and return the user ID or null.
	 *
	 * This method searches for a user in Pipedrive by email. If found, it returns the user's ID.
	 * It first tries the `/users/find` endpoint with the `search_by_email` flag, then falls back to a generic search.
	 *
	 * @param string $email The email address to search for.
	 *
	 * @return int|null The user ID if found, null otherwise.
	 */
	private function pipedrive_find_user_id_by_email( string $email ): ?int {
		// First try: /users/find with search_by_email
		$url = add_query_arg(
			[
				'term' => $email,
				'search_by_email' => 1,
				'exact_match' => 1,
				'limit' => 1,
				'api_token' => $this->pipedrive_token,
			],
			"{$this->pipedrive_base}/users/find"
		);

		$res = wp_remote_get(
			$url,
			[
				'timeout' => 15,
				'headers' => [ 'Accept' => 'application/json' ],
			]
		);
		if ( ! is_wp_error( $res ) ) {
			$code = wp_remote_retrieve_response_code( $res );
			$body = json_decode( wp_remote_retrieve_body( $res ), true );
			if ( 200 === $code && ! empty( $body['data'][0]['id'] ) ) {
				return (int) $body['data'][0]['id'];
			}
		}

		// Fallback: generic term search
		$url2 = add_query_arg(
			[
				'term' => $email,
				'limit' => 1,
				'api_token' => $this->pipedrive_token,
			],
			"{$this->pipedrive_base}/users/find"
		);

		$res2 = wp_remote_get(
			$url2,
			[
				'timeout' => 15,
				'headers' => [ 'Accept' => 'application/json' ],
			]
		);
		if ( ! is_wp_error( $res2 ) ) {
			$code2 = wp_remote_retrieve_response_code( $res2 );
			$body2 = json_decode( wp_remote_retrieve_body( $res2 ), true );
			if ( 200 === $code2 && ! empty( $body2['data'][0]['id'] ) ) {
				return (int) $body2['data'][0]['id'];
			}
		}

		return null;
	}

	/**
	 * Retrieve a Pipedrive custom field key, or resolve a single option ID for that field.
	 *
	 * When $label is null, returns the custom field API key for the given field name.
	 * When $label is provided for an enum type field, returns the integer option ID that matches the label.
	 * Uses the {object}Fields endpoint, for example dealFields, personFields, organizationFields, or leadFields.
	 *
	 * @param string|null $target_field_name The display name of the field as seen in Pipedrive, for example "Source channel".
	 * @param string|null $label             The option label to resolve to its numeric ID, pass null to get the field key instead.
	 * @param string      $object            The object type to inspect, for example 'deal', 'person', 'organization', or 'lead'. Default 'deal'.
	 *
	 * @return string|int|null Field key when $label is null, option ID when $label is provided, null when not found.
	 *
	 * @throws \Exception On HTTP or API errors raised by send_pipedrive_request.
	 */
	protected function get_custom_field_option( $target_field_name = null, $label = null, $object = 'deal' ) {
		$data = $this->send_pipedrive_request( 'GET', $object . 'Fields' );

		if ( empty( $data['data'] ) || ! is_array( $data['data'] ) ) {
			return null;
		}

		foreach ( $data['data'] as $field ) {
			if ( ! empty( $field['name'] ) && $field['name'] === $target_field_name ) {
				if ( is_null( $label ) ) {
					return $field['key'];
				} else {
					foreach ( $field['options'] as $opt ) {
						if ( strcasecmp( $opt['label'], $label ) === 0 ) {
							return (int) $opt['id'];
						}
					}
				}
			}
		}

		return null;
	}

	/**
	 * Fetches the service details by ID from the Amelia database.
	 *
	 * @param int $service_id The service ID.
	 *
	 * @return array|null The service details if found, null otherwise.
	 */
	private function get_amelia_service_by_id( $service_id ) {
		global $wpdb;

		if ( ! $service_id ) {
			return null;
		}

		// Query the database to get service details
		$service = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}amelia_services WHERE id = %d",
				$service_id
			),
			ARRAY_A
		);

		if ( empty( $service ) ) {
			return null;
		}

		return $service;
	}

	/**
	 * Fetches the category details by ID from the Amelia database.
	 *
	 * @param int $category_id The category ID.
	 *
	 * @return array|null The category details if found, null otherwise.
	 */
	private function get_amelia_category_by_id( $category_id ) {
		global $wpdb;

		if ( ! $category_id ) {
			return null;
		}

		// Query the database to get the category details
		$category = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}amelia_categories WHERE id = %d",
				$category_id
			),
			ARRAY_A
		);

		if ( empty( $category ) ) {
			return null;
		}

		return $category;
	}

	/**
	 * Fetches the provider details by ID from the Amelia users table.
	 *
	 * @param int $provider_id The provider ID.
	 *
	 * @return array|null The provider details if found, null otherwise.
	 */
	private function get_amelia_provider_by_id( $provider_id ) {
		global $wpdb;

		if ( ! $provider_id ) {
			return null;
		}

		// Query the database to get the provider details from the amelia_users table
		$provider = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}amelia_users WHERE id = %d AND type = 'provider'",
				$provider_id
			),
			ARRAY_A
		);

		if ( empty( $provider ) ) {
			return null;
		}

		return $provider;
	}
}
