<?php
/**
 * Elementor Copyright Widget.
 *
 * Elementor widget that inserts our standard Copyright text.
 *
 * @since 1.0.0
 *
 * @package StrategySuite
 */

namespace Elementor;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

/**
 * Class Elementor_Copyright_Widget.
 *
 * Elementor widget that inserts our standard Copyright text.
 */
class Elementor_Copyright_Widget extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Copyright widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'copyright';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Copyright widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Copyright', 'plugin-name' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Copyright widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-info';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Copyright widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'theme-elements' ];
	}

	/**
	 * Register Copyright widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'strategy' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'site-title',
			[
				'label' => __( 'Site Title', 'strategy' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => __( 'Enter the title of the website.', 'strategy' ),
			]
		);

		$this->add_control(
			'back-link-text',
			[
				'label' => __( 'Text for Creator Backlink', 'strategy' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => __( 'Crafted with ❤ by Strategy', 'strategy' ),
			]
		);

		$this->add_control(
			'back-link-link',
			[
				'label' => __( 'Creator Backlink', 'strategy' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => __( 'https://strategynewmedia.com', 'strategy' ),
			]
		);
		$this->add_control(
			'remove-backlink',
			[
				'label' => __( 'Remove Back Link', 'strategy' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_section',
			[
				'label' => __( 'Style Section', 'strategy' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'copyright_typography',
				'label' => __( 'Typography', 'strategy' ),
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .copyright',
			]
		);

		$this->add_control(
			'copyright_color',
			[
				'label' => __( 'Text Color', 'strategy' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_TEXT,
				],
				'selectors' => [
					'{{WRAPPER}} .copyright' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render oEmbed widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		echo '<span class="copyright">';

		printf(
			esc_html__( '%1$s %2$s %3$s. All Rights Reserved.', 'strategy' ),
			'&copy;',
			esc_attr( gmdate( 'Y' ) ),
			esc_attr( $settings['site-title'] )
		);
		if ( is_array( $settings ) && ( is_null( $settings['remove-backlink'] ) || '' == $settings['remove-backlink'] ) ) {
			if ( $settings['back-link-link'] && $settings['back-link-text'] ) {
				printf(
					'<br/> %1$s',
					'<a href="' . esc_url( $settings['back-link-link'] ) . '" target="_blank" title="' . esc_attr( $settings['back-link-text'] ) . ' (Opens in a new window)" rel="noreferrer noopener">' . esc_html( $settings['back-link-text'] ) . '</a>'
				);
			} elseif ( $settings['back-link-text'] ) {
				printf(
					'<br/> %1$s',
					esc_attr( $settings['back-link-text'] ),
				);
			} else {
				printf(
					'<br/> %1$s',
					'Crafted with ❤ by <a href="https://strategynewmedia.com" target="_blank" title="A Kansas City Marketing Agency (Opens in a new window)" rel="noreferrer noopener">Strategy</a>'
				);
			}
		}
		echo '</span>';
	}
}

Plugin::instance()->widgets_manager->register( new Elementor_Copyright_Widget() );
