<?php
/**
 * Strategy Analytics
 *
 * Handles analytics automation and tracking script integration.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use GFCommon;
use GFAPI;
use Google\Service\Analytics;

/**
 * Class StrategyAnalytics
 *
 * This class integrates and manages various analytics tracking solutions,
 * including Google Analytics (GA4), Google Tag Manager (GTM), Meta Pixels,
 * CallRail, and Clarity.
 */
class StrategyAnalytics extends \StrategySuite\Module {

	/**
	 * Whether or not the class will output anything
	 *
	 * @var boolean
	 */
	private $active = false;

	/**
	 * Automation settings in the database
	 *
	 * @var array
	 */
	private $automation_settings;

	/**
	 * Other analytics installations in the database
	 *
	 * @var array
	 */
	private $other_installations;

	/**
	 * GTM_Automation object
	 *
	 * @var object GTM_Automation
	 */
	private $gtm_automation;

	/**
	 * Scripts to output into the <head>
	 *
	 * @var array
	 */
	private $head_scripts;

	/**
	 * Scripts to output into the beginning of the <body>
	 *
	 * @var array
	 */
	private $body_scripts;

	/**
	 * Scripts to output at the end the </body>
	 *
	 * @var array
	 */
	private $footer_scripts;

	/**
	 * Determines if the module can be registered.
	 *
	 * Checks if Advanced Custom Fields (ACF) is available.
	 *
	 * @return bool True if ACF is available, otherwise false.
	 */
	public function can_register() {
		return function_exists( 'get_field' );
	}

	/**
	 * Registers hooks for analytics automation and script inclusion.
	 *
	 * Adds actions and filters for:
	 * - Analytics automation settings.
	 * - Google Tag Manager integration.
	 * - Form submission tracking.
	 *
	 * @return void
	 */
	public function register() {
		$this->automation_settings = get_field( 'analytics_automation', 'options' );
		$this->other_installations = get_field( 'other_installations', 'options' );
		$this->head_scripts = [];
		$this->body_scripts = [];
		$this->footer_scripts = [];

		if ( ! is_null( $this->automation_settings ) && ! is_null( $this->other_installations ) ) {
			$this->active = true;
		}

		if ( ! $this->active ) {
			return;
		}

		// TODO: Might add check_automation_settings() and gather_other_installations() here so it doesn't need to be called everywhere

		add_action( 'acf/save_post', [ $this, 'acf_save_caller' ], 12 );
		add_action( 'gform_after_save_form', [ $this, 'gform_analytics_validation' ], 30, 2 );
		add_action( 'wp_head', [ $this, 'head_scripts' ], 5 );
		add_action( 'wp_body_open', [ $this, 'body_scripts' ], 5 );
		add_action( 'wp_footer', [ $this, 'footer_scripts' ], 5 );
		add_filter( 'gform_confirmation', [ $this, 'ga4_form_submission_event' ], 10, 4 );
	}

	/**
	 * Handles actions after saving ACF fields.
	 *
	 * Ensures analytics automation settings are updated when the
	 * site settings are saved.
	 */
	public function acf_save_caller() {
		$screen = get_current_screen();
		if ( ! $this->active || ! strpos( $screen->id, 'site_settings' ) ) {
			return;
		}

		// $this->other_installations = get_field( 'other_installations', 'options' );

		$this->check_automation_settings();
		$this->gather_other_installations();
	}

	/**
	 * Validates and updates analytics settings when a Gravity Form is saved.
	 *
	 * @param array   $form   The form being saved.
	 * @param boolean $is_new Whether the form is new or being updated.
	 */
	public function gform_analytics_validation( $form, $is_new ) {
		if ( ! $this->active ) {
			return;
		}
		$gtm_done = rgar( $form, 'gtm_done' );
		$old_title = rgar( $form, 'gtm_last_title', $form['title'] );
		$new_title = $form['title'];

		if ( $gtm_done && $old_title === $new_title ) {
			return;
		}

		$this->check_automation_settings();

		if ( null !== $this->gtm_automation ) {
			$success = $this->gtm_automation->run_single_form( $new_title );
			if ( $success ) {
				$form['gtm_last_title'] = $new_title;
				$form['gtm_done'] = 1;
				GFAPI::update_form( $form );
			}
		}

		if ( $is_new ) {
			$this->other_installations = get_field( 'other_installations', 'options' );
			$this->gather_other_installations();
		}
	}

	/**
	 * Checks and updates automation settings for analytics tracking.
	 *
	 * If automation settings are enabled, this function ensures that the
	 * correct tracking scripts and GTM configurations are set.
	 */
	private function check_automation_settings() {
		if ( ! $this->active || is_null( $this->automation_settings ) || empty( $this->automation_settings ) ) {
			return;
		}

		if ( '' !== trim( $this->automation_settings['account_id'] ) && '' !== trim( $this->automation_settings['container_id'] ) ) {
			$this->init_automation();
			if ( $this->automation_settings['run_automation'] ) {
				$automation_updates = [];

				// if ( null !== $this->ga4_automation ) {

				// }

				if ( null !== $this->gtm_automation ) {
					$gtm_args = [];
					if ( '' === $this->automation_settings['gtm_public_id'] ) {
						$gtm_args['get_public_id'] = true;
					}
					if ( '' !== $this->automation_settings['ga4_measurement_id'] ) {
						$gtm_args['measurement_id'] = $this->automation_settings['ga4_measurement_id'];
					}
					$gtm_automation_report = $this->gtm_automation->run_default( $gtm_args );
					if ( isset( $gtm_automation_report['public_id'] ) ) {
						$automation_updates['gtm_public_id'] = $gtm_automation_report['public_id'];
					}
					$automation_updates['run_automation'] = false;
				}

				if ( 0 < count( $automation_updates ) ) {
					if ( isset( $automation_updates['gtm_public_id'] ) ) {
						$oi_args = [];
						if ( '' !== trim( $this->automation_settings['gtm_public_id'] ) && $this->automation_settings['gtm_public_id'] !== $automation_updates['gtm_public_id'] ) {
							$oi_args['remove'] = [
								[
									'type'  => 'gtm',
									'id'    => $this->automation_settings['gtm_public_id'],
								],
							];
						}
						$oi_args['add'] = [
							[
								'type'  => 'gtm',
								'id'    => $automation_updates['gtm_public_id'],
							],
						];
						$this->update_other_installations( $oi_args );
					}
					$this->update_automation_settings( $automation_updates );
				}
			}
		}
	}

	/**
	 * Updates automation settings in ACF options.
	 *
	 * @param array $settings The new settings to be updated.
	 */
	private function update_automation_settings( $settings = [] ) {
		if ( isset( $settings['gtm_public_id'] ) ) {
			$this->automation_settings['gtm_public_id'] = $settings['gtm_public_id'];
		}

		if ( isset( $settings['ga4_measurement_id'] ) ) {
			$this->automation_settings['ga4_measurement_id'] = $settings['ga4_measurement_id'];
		}

		if ( isset( $settings['run_automation'] ) ) {
			$this->automation_settings['run_automation'] = $settings['run_automation'];
		}

		if ( ! empty( $settings ) ) {
			update_field( 'analytics_automation', $this->automation_settings, 'options' );
		}
	}

	/**
	 * Initializes Google Tag Manager automation.
	 *
	 * Loads the GTM automation class and initializes it with account and container IDs.
	 */
	private function init_automation() {
		$file = STRATEGY_SUITE_INC . 'noautoload-classes/GTMAutomation.php';
		if ( $file && is_readable( $file ) ) {
			require_once $file;
		}
		$this->gtm_automation = new GTM_Automation( $this->automation_settings['account_id'], $this->automation_settings['container_id'] );
	}

	/**
	 * Collects and organizes third-party analytics installation IDs.
	 *
	 * Retrieves IDs for Google Analytics, Google Tag Manager, Meta Pixels,
	 * CallRail, and Clarity, and prepares them for script injection.
	 */
	private function gather_other_installations() {

		if ( ! $this->active || ! $this->other_installations ) {
			return;
		}

		$id_collection = array(
			'ga4'           => array(),
			'gtm'           => array(),
			'meta_pixels'   => array(),
			'callrail'   => array(),
			'clarity'   => array(),
		);

		foreach ( $this->other_installations as $definition ) {
			if ( isset( $id_collection[ $definition['type'] ] ) ) {
				$id_collection[ $definition['type'] ][] = $definition['id'];
			}
		}

		// Clear the script variables
		$this->head_scripts = array();
		$this->body_scripts = array();

		if ( ! empty( $id_collection['ga4'] ) ) {
			$ga4_head_script = $this->ga4_head_script( $id_collection['ga4'] );
			$this->head_scripts[] = $ga4_head_script;
		}

		if ( ! empty( $id_collection['gtm'] ) ) {
			$gtm_head_script = $this->gtm_head_script( $id_collection['gtm'] );
			$gtm_body_script = $this->gtm_body_script( $id_collection['gtm'] );
			$this->head_scripts[] = $gtm_head_script;
			$this->body_scripts[] = $gtm_body_script;
		}

		if ( ! empty( $id_collection['meta_pixels'] ) ) {
			$meta_pixels_head_script = $this->meta_pixels_head_script( $id_collection['meta_pixels'] );
			$meta_pixels_body_script = $this->meta_pixels_body_script( $id_collection['meta_pixels'] );
			$this->head_scripts[] = $meta_pixels_head_script;
			$this->body_scripts[] = $meta_pixels_body_script;
		}

		if ( ! empty( $id_collection['callrail'] ) ) {
			$callrail_script = $this->callrail_footer_script( $id_collection['callrail'] );
			$this->footer_scripts[] = $callrail_script;
		}

		if ( ! empty( $id_collection['clarity'] ) ) {
			$clarity_head_script = $this->clarity_head_script( $id_collection['clarity'] );
			$this->head_scripts[] = $clarity_head_script;
		}
	}

	/**
	 * Updates the "Other Installations" ACF field with new analytics configurations.
	 *
	 * @param array $args An array containing additions or removals.
	 */
	private function update_other_installations( $args = [] ) {
		$changes = 0;
		if ( isset( $args['remove'] ) ) {
			foreach ( $this->other_installations as $index => $installation ) {
				if ( isset( $installation['type'] ) && isset( $installation['id'] ) ) {
					foreach ( $args['remove'] as $info ) {
						if ( isset( $info['type'] ) && isset( $info['id'] ) && $info['type'] === $installation['type'] && $info['id'] === $installation['id'] ) {
							unset( $this->other_installations[ $index ] );
							$changes++;
							break;
						}
					}
				}
			}
		}

		if ( isset( $args['add'] ) ) {
			foreach ( $args['add'] as $info ) {
				if ( isset( $info['type'] ) && isset( $info['id'] ) ) {
					$add = true;
					foreach ( $this->other_installations as $index => $installation ) {
						if ( isset( $installation['type'] ) && isset( $installation['id'] ) ) {
							if ( $info['type'] === $installation['type'] && $info['id'] === $installation['id'] ) {
								$add = false;
								break;
							}
						}
					}
					if ( $add ) {
						$new_row = [
							'type'  => $info['type'],
							'id'    => $info['id'],
						];
						$this->other_installations[] = $new_row;
						$changes++;
					}
				}
			}
		}

		if ( 0 < $changes ) {
			$new_array = [];
			foreach ( $this->other_installations as $installation ) {
				$new_array[] = $installation;
			}
			$this->other_installations = $new_array;
			update_field( 'other_installations', $this->other_installations, 'options' );
		}
	}

	/**
	 * Outputs analytics scripts in the `<head>` section.
	 */
	public function head_scripts() {

		if ( ! $this->active ) {
			return;
		}

		$this->gather_other_installations();

		if ( is_null( $this->head_scripts ) || empty( $this->head_scripts ) ) {
			return;
		}

		if ( $this->can_track_visitors() ) {
			echo '<!-- Analytics Head Scripts - Begin -->';
			foreach ( $this->head_scripts as $head_script ) {
				echo $head_script; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
			echo '<!-- Analytics Head Scripts - End -->';
		}
	}

	/**
	 * Outputs analytics scripts inside `<body>`.
	 */
	public function body_scripts() {

		if ( ! $this->active ) {
			return;
		}

		if ( is_null( $this->body_scripts ) || empty( $this->body_scripts ) ) {
			return;
		}

		if ( $this->can_track_visitors() ) {
			echo '<!-- Analytics Body Scripts - Begin -->';
			foreach ( $this->body_scripts as $body_script ) {
				echo $body_script; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
			echo '<!-- Analytics Body Scripts - End -->';
		}
	}

	/**
	 * Outputs analytics scripts before the closing `</body>` tag.
	 */
	public function footer_scripts() {

		if ( ! $this->active ) {
			return;
		}

		if ( is_null( $this->footer_scripts ) || empty( $this->footer_scripts ) ) {
			return;
		}

		if ( $this->can_track_visitors() ) {
			echo '<!-- Analytics Footer Scripts - Begin -->';
			foreach ( $this->footer_scripts as $footer_script ) {
				echo $footer_script; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
			echo '<!-- Analytics Footer Scripts - End -->';
		}
	}

	/**
	 * Generates the `<head>` script for Google Analytics 4.
	 *
	 * @param array $ga4_ids GA4 tracking IDs.
	 * @return string The generated GA4 script.
	 */
	private function ga4_head_script( $ga4_ids ) {

		$snippets = '';
		foreach ( $ga4_ids as $ga4_id ) {
			$snippets .= $this->ga4_head_snippet( $ga4_id );
		}

		$script =
		"<script>
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            gtag('js', new Date());"
			. $snippets . '
        </script>
        ';

		return $script;
	}

	/**
	 * Generates the `<head>` script for Google Tag Manager.
	 *
	 * @param array $gtm_ids GTM container IDs.
	 * @return string The generated GTM script.
	 */
	private function gtm_head_script( $gtm_ids ) {

		$scripts = '';
		foreach ( $gtm_ids as $gtm_id ) {
			$scripts .=
			"<script>
                (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
                new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
                j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
                'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
                })(window,document,'script','dataLayer','" . $gtm_id . "');
            </script>
            ";
		}

		return $scripts;
	}

	/**
	 * Generates the `<body>` script for Google Tag Manager.
	 *
	 * @param array $gtm_ids GTM container IDs.
	 * @return string The generated GTM noscript tag.
	 */
	private function gtm_body_script( $gtm_ids ) {

		$snippets = '';
		foreach ( $gtm_ids as $gtm_id ) {
			$snippets .= $this->gtm_body_snippet( $gtm_id );
		}

		$script =
		'<noscript>'
		. $snippets . '
        </noscript>
        ';

		return $script;
	}

	/**
	 * Generates the `<head>` script for Meta Pixels.
	 *
	 * @param array $pixel_ids Meta Pixel IDs.
	 * @return string The generated Meta Pixel script.
	 */
	private function meta_pixels_head_script( $pixel_ids ) {

		$snippets = '';
		foreach ( $pixel_ids as $pixel_id ) {
			$snippets .= $this->meta_pixels_head_snippet( $pixel_id );
		}

		$script =
		"<script>
            !function(f,b,e,v,n,t,s)
            {if(f.fbq)return;n=f.fbq=function(){n.callMethod?
            n.callMethod.apply(n,arguments):n.queue.push(arguments)};
            if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';
            n.queue=[];t=b.createElement(e);t.async=!0;
            t.src=v;s=b.getElementsByTagName(e)[0];
            s.parentNode.insertBefore(t,s)}(window, document,'script',
            'https://connect.facebook.net/en_US/fbevents.js');"
			. $snippets . "
            fbq('track', 'PageView');
        </script>
        ";

		return $script;
	}

	/**
	 * Generates the `<body>` script for Meta Pixels.
	 *
	 * @param array $pixel_ids Meta Pixel IDs.
	 * @return string The generated Meta Pixel noscript tag.
	 */
	private function meta_pixels_body_script( $pixel_ids ) {

		$snippets = '';
		foreach ( $pixel_ids as $pixel_id ) {
			$snippets .= $this->meta_pixels_body_snippet( $pixel_id );
		}

		$script =
		'<noscript>'
		. $snippets . '
        </noscript>
        ';

		return $script;
	}

	/**
	 * Generates the footer script for CallRail tracking.
	 *
	 * @param array $callrail_installations CallRail tracking IDs.
	 * @return string The generated CallRail script.
	 */
	private function callrail_footer_script( $callrail_installations ) {

		$script = '';
		foreach ( $callrail_installations as $install ) {
			$script .= '<script src="//cdn.calltrk.com/companies/' . esc_attr( $install ) . '/12/swap.js"  async defer></script>'; // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript
		}

		return $script;
	}

	/**
	 * Generates the `<head>` script for Microsoft Clarity tracking.
	 *
	 * @param array $clarity_installations Clarity tracking IDs.
	 * @return string The generated Clarity script.
	 */
	private function clarity_head_script( $clarity_installations ) {

		$script = '';
		foreach ( $clarity_installations as $install ) {
			$script .= '<script async defer>(function(c,l,a,r,i,t,y){c[a]=c[a]||function(){(c[a].q=c[a].q||[]).push(arguments)};t=l.createElement(r);t.async=1;t.src="https://www.clarity.ms/tag/"+i;y=l.getElementsByTagName(r)[0];y.parentNode.insertBefore(t,y);})(window, document, "clarity", "script", "' . esc_attr( $install ) . '");</script>';
		}

		return $script;
	}

	/**
	 * Generates a GA4 script snippet for an individual tracking ID.
	 *
	 * @param string $ga4_id GA4 tracking ID.
	 * @return string The GA4 script snippet.
	 */
	private function ga4_head_snippet( $ga4_id ) {

		$snippet =
		"
        gtag('config', '" . $ga4_id . "');";

		return $snippet;
	}

	/**
	 * Generates a GTM noscript iframe for an individual GTM ID.
	 *
	 * @param string $gtm_id GTM container ID.
	 * @return string The GTM noscript tag.
	 */
	private function gtm_body_snippet( $gtm_id ) {

		$snippet =
		'
        <iframe src="https://www.googletagmanager.com/ns.html?id=' . $gtm_id . '" height="0" width="0" style="display:none;visibility:hidden"></iframe>';

		return $snippet;
	}

	/**
	 * Generates a Meta Pixel script snippet for an individual Pixel ID.
	 *
	 * @param string $pixel_id Meta Pixel ID.
	 * @return string The Meta Pixel script snippet.
	 */
	private function meta_pixels_head_snippet( $pixel_id ) {

		$snippet =
		"
        fbq('init', '" . $pixel_id . "');";

		return $snippet;
	}

	/**
	 * Generates a Meta Pixel noscript image for an individual Pixel ID.
	 *
	 * @param string $pixel_id Meta Pixel ID.
	 * @return string The Meta Pixel noscript image tag.
	 */
	private function meta_pixels_body_snippet( $pixel_id ) {

		$snippet =
		'
        <img height="1" width="1" style="display:none" src="https://www.facebook.com/tr?id=' . $pixel_id . '&ev=PageView&noscript=1"/>';

		return $snippet;
	}

	/**
	 * Tracks and sends a GA4 event when a Gravity Form is submitted.
	 *
	 * @param string $confirmation The form confirmation message.
	 * @param array  $form         The Gravity Form being submitted.
	 * @param array  $entry        The submitted entry data.
	 * @param bool   $ajax         Whether the form was submitted via AJAX.
	 * @return string The modified form confirmation message.
	 */
	public function ga4_form_submission_event( $confirmation, $form, $entry, $ajax ) {
		if ( ! $this->can_track_visitors() ) {
			return $confirmation;
		}

		$field_id_email = get_nth_field_id( $form, 'email', 1 );
		$field_id_phone = get_nth_field_id( $form, 'phone', 1 );

		// Create the data_layer array with the event and form details
		$data_layer = [
			'event' => 'customFormSubmission',
			'formTitle' => str_replace( "'", "\'", $form['title'] ),
		];

		// Add email and phone fields conditionally to the array if values exist
		if ( $field_id_email && ! empty( rgar( $entry, $field_id_email ) ) ) {
			$data_layer['email'] = rgar( $entry, $field_id_email );
		}

		if ( $field_id_phone && ! empty( rgar( $entry, $field_id_phone ) ) ) {
			$data_layer['phone'] = rgar( $entry, $field_id_phone );
		}

		// Apply filters to the data array before pushing it to the dataLayer
		$data_layer = apply_filters( 'data_layer_after_form_submission', $data_layer, $form, $entry );

		// Output the final JavaScript to push the data to window.dataLayer
		$new_confirmation_event = 'window.dataLayer = window.dataLayer || []; window.dataLayer.push(' . json_encode( $data_layer ) . ');';

		if ( ! empty( $entry ) && ( rgar( $entry, 'status' ) !== 'spam' ) ) {
			if ( is_array( $confirmation ) ) {
				$url = esc_url_raw( $confirmation['redirect'] );
				$confirmation = 'Thank you for your submission, you will be redirected.';
				$confirmation = sprintf(
					'<div id="gform_confirmation_message_%1$d" class="gform_confirmation_message_%1$d gform_confirmation_message">%2$s</div>',
					absint( $form['id'] ),
					$confirmation . GFCommon::get_inline_script_tag( $new_confirmation_event . "setTimeout(function(){location.replace('$url')}, 4000);" )
				);
			} else {
				$confirmation .= GFCommon::get_inline_script_tag( $new_confirmation_event );
			}
		}

		return $confirmation;
	}

	/**
	 * Determines whether tracking scripts should be loaded.
	 *
	 * Scripts are only loaded in production and staging environments.
	 *
	 * @return bool True if scripts should be loaded, otherwise false.
	 */
	private function can_track_visitors() {
		return WP_ENV == 'production' || WP_ENV == 'staging';
	}
}
