<?php
/**
 * StockPack
 *
 * This class customizes the functionality of the StockPack plugin in WordPress.
 * It modifies available stock providers, updates default plugin options, and handles
 * notifications visibility for specific user roles. The class ensures the StockPack
 * plugin integrates seamlessly with the site's workflow.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class StockPack
 *
 * This class extends the StrategySuite\Module and manages customizations for
 * the StockPack plugin, including filtering stock providers, adjusting plugin
 * options, and modifying captions for media files.
 *
 * @package StrategySuite
 */
class StockPack extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * Checks if the StockPack plugin is active before registering the module.
	 *
	 * @return bool True if the StockPack plugin is active, false otherwise.
	 */
	public function can_register() {
		return is_plugin_active( 'stockpack/stockpack.php' );
	}

	/**
	 * Register hooks for customizing StockPack plugin behavior.
	 *
	 * This method registers filters and actions to:
	 * - Limit available stock providers.
	 * - Change default StockPack options.
	 * - Disable large image threshold resizing.
	 * - Modify captions generated by StockPack.
	 * - Remove StockPack notifications for most users.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'change_default_stockpack_options' ], 10 );
		add_filter( 'stockpack_caption', [ $this, 'stockpack_change_generated_caption' ], 20, 2 );
		add_filter( 'stockpack_generated_caption', [ $this, 'stockpack_change_generated_caption' ], 20, 2 );
		add_filter( 'stockpack_settings_capability', [ $this, 'stockpack_permission' ] );
		add_filter( 'stockpack_notices_capability', [ $this, 'stockpack_permission' ] );
	}

	/**
	 * Remove StockPack notifications for most users.
	 *
	 * Adds a filter to limit StockPack notifications to users with a specific capability.
	 *
	 * @return string The capability for viewing StockPack notifications.
	 */
	public function stockpack_permission() {
		return 'manage_options';
	}

	/**
	 * Change the default StockPack options to preferred settings.
	 *
	 * Automates the customization of StockPack options. If the options haven't been automated yet,
	 * this method:
	 * - Enables file name changes for media uploads.
	 * - Enables premium caption providers.
	 * - Enables standard caption fields.
	 * - Enables captions for featured images.
	 *
	 * Marks the automation process by setting the 'stockpack_options_automated' option to true.
	 *
	 * @return void
	 */
	public function change_default_stockpack_options() {
		if ( get_option( 'stockpack_options_automated' ) === false ) {
			// Update 'stockpack_basics' options
			$stockpack_basics = get_option( 'stockpack_basics', array() );
			if ( empty( $stockpack_basics ) ) {
				$stockpack_basics = [];
			}
			$stockpack_basics['file_name_change'] = 'yes';
			$stockpack_basics['safe_search'] = 'yes'; // Adjust based on your preference
			if ( defined( 'STOCKPACK_TOKEN' ) ) {
				$stockpack_basics['auth_token'] = STOCKPACK_TOKEN;
			}
			$stockpack_basics['providers'] = array(
				'Adobe Stock'    => 'Adobe Stock',
				'Unsplash'       => 'Unsplash',
			);
			update_option( 'stockpack_basics', $stockpack_basics );

			// Update 'stockpack_advanced' options
			$stockpack_advanced = get_option( 'stockpack_advanced', array() );
			if ( empty( $stockpack_advanced ) ) {
				$stockpack_advanced = [];
			}
			$stockpack_advanced['caption_premium_providers'] = 'yes';
			$stockpack_advanced['caption_standard_fields'] = 'yes';
			$stockpack_advanced['caption_featured_image'] = 'yes';
			$stockpack_advanced['license_state'] = 'yes';
			$stockpack_advanced['add_license_state_to_filename'] = 'no';
			update_option( 'stockpack_advanced', $stockpack_advanced );

			// Update 'stockpack_debug' options
			$stockpack_debug = get_option( 'stockpack_debug', array() );
			if ( empty( $stockpack_debug ) ) {
				$stockpack_debug = [];
			}
			$stockpack_debug['download_timeout'] = 30;
			$stockpack_debug['url_debugging'] = 'no';
			update_option( 'stockpack_debug', $stockpack_debug );
			update_option( 'stockpack_options_automated', true );
		}
	}

	/**
	 * Modify the captions generated by StockPack.
	 *
	 * Adds a title attribute "(opens in a new window)" to links within generated captions.
	 *
	 * @param string $caption    The original caption generated by StockPack.
	 * @param mixed  $attachment The attachment object associated with the caption.
	 * @return string The modified caption with updated link titles.
	 */
	public function stockpack_change_generated_caption( $caption, $attachment ) {
		$returned_replace = str_replace( '<a ', '<a title="(opens in a new window)" ', $caption );
		return $returned_replace;
	}
}
