<?php
/**
 * SiteSettings
 *
 * Manages global site settings through Advanced Custom Fields (ACF),
 * including accessibility, business info, design, integrations, and analytics.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class SiteSettings
 *
 * Handles the creation and management of a centralized site settings page using ACF.
 */
class SiteSettings extends \StrategySuite\Module {

	/**
	 * Load order for this module.
	 *
	 * @var int
	 */
	public $load_order = 1;

	/**
	 * Determines if the module can be registered.
	 *
	 * Checks if ACF functions for adding options pages and local field groups exist.
	 *
	 * @return bool True if ACF is available, otherwise false.
	 */
	public function can_register() {
		return function_exists( 'acf_add_options_page' ) && function_exists( 'acf_add_local_field_group' );
	}

	/**
	 * Registers hooks for setting up site settings and managing integrations.
	 *
	 * This function adds necessary ACF actions and filters, including:
	 * - Initializing site settings options page
	 * - Including custom ACF field groups
	 * - Handling Google API key settings
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'init', [ $this, 'setup_site_settings' ] );
		add_action( 'acf/include_fields', [ $this, 'include_fields' ] );

		// Design Settings - Fonts
		// TODO: Future implementation
		// add_filter( 'acf/load_field/name=heading_font', [ $this, 'acf_populate_selected_fonts' ], 20 );
		// add_filter( 'acf/load_field/name=body_font', [ $this, 'acf_populate_selected_fonts' ], 20 );
		// add_filter( 'acf/load_field/name=custom_font_1', [ $this, 'acf_populate_selected_fonts' ], 20 );
		// add_filter( 'acf/load_field/name=custom_font_2', [ $this, 'acf_populate_selected_fonts' ], 20 );

		// Google API Key
		add_action( 'acf/init', [ $this, 'acf_set_google_api' ] );
		add_filter( 'acf/fields/google_map/api', [ $this, 'acf_get_google_api' ] );
	}

	/**
	 * Sets up the "Site Settings" options page in WordPress admin.
	 *
	 * Uses ACF to create a dedicated page for managing site-wide configurations.
	 *
	 * @return void
	 */
	public function setup_site_settings() {
		acf_add_options_page(
			array(
				'page_title' => 'Site Settings',
				'menu_title' => 'Site Settings',
				'menu_slug'  => 'site_settings',
				'capability' => 'manage_options',
				'redirect'   => false,
			)
		);
	}

	/**
	 * Includes ACF local field groups for managing site settings.
	 *
	 * Defines the structure for accessibility, business info, design settings,
	 * integrations, and analytics configurations.
	 *
	 * @return void
	 */
	public function include_fields() {
		acf_add_local_field_group(
			array(
				'key' => 'group_63ea7d09b41aa',
				'title' => 'Site Settings',
				'fields' => array(
					array(
						'key' => 'field_63ea7d0abbc0a',
						'label' => 'Accessibility',
						'name' => '',
						'aria-label' => '',
						'type' => 'tab',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'placement' => 'top',
						'endpoint' => 0,
						'selected' => 0,
					),
					array(
						'key' => 'field_63ea7dd3bbc0b',
						'label' => 'Accessibility Tools',
						'name' => 'accessibility_tools',
						'aria-label' => '',
						'type' => 'group',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'block',
						'sub_fields' => array(
							array(
								'key' => 'field_63ea8086bbc11',
								'label' => 'Enable Accessibility Tools',
								'name' => 'enable_accessibility_tools',
								'aria-label' => '',
								'type' => 'true_false',
								'instructions' => 'Whether or not to enable the accessibility tools.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'message' => '',
								'default_value' => 1,
								'ui_on_text' => 'Enabled',
								'ui_off_text' => 'Disabled',
								'ui' => 1,
							),
							array(
								'key' => 'field_63ea7e18bbc0c',
								'label' => 'Sitemap Link',
								'name' => 'sitemap_link',
								'aria-label' => '',
								'type' => 'link',
								'instructions' => 'Link for sitemap page in your website. Leave blank to disable.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'array',
							),
							array(
								'key' => 'field_63ea7e5bbbc0d',
								'label' => 'Help Link',
								'name' => 'help_link',
								'aria-label' => '',
								'type' => 'link',
								'instructions' => 'Link for help page in your website. Leave blank to disable.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'array',
							),
							array(
								'key' => 'field_63ea7e69bbc0e',
								'label' => 'Feedback Link',
								'name' => 'feedback_link',
								'aria-label' => '',
								'type' => 'link',
								'instructions' => 'Link for feedback page in your website. Leave blank to disable.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'array',
							),
							array(
								'key' => 'field_63ea7e87bbc0f',
								'label' => 'Remove Link Target',
								'name' => 'remove_link_target',
								'aria-label' => '',
								'type' => 'true_false',
								'instructions' => 'This option will reset all your target links to open in the same window or tab.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'message' => '',
								'default_value' => 0,
								'ui_on_text' => 'Enabled',
								'ui_off_text' => 'Disabled',
								'ui' => 1,
							),
							array(
								'key' => 'field_63ea7f63bbc10',
								'label' => 'Remember Settings Expiration',
								'name' => 'remember_settings_expiration',
								'aria-label' => '',
								'type' => 'select',
								'instructions' => 'Define how long your toolbar settings will be remembered.',
								'required' => 1,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'choices' => array(
									1 => '1 Hour',
									6 => '6 Hours',
									12 => '12 Hours',
									24 => '1 Day',
									48 => '2 Days',
									72 => '3 Days',
									168 => '1 Week',
									720 => '1 Month',
								),
								'default_value' => 168,
								'return_format' => 'value',
								'multiple' => 0,
								'allow_null' => 0,
								'ui' => 0,
								'ajax' => 0,
								'placeholder' => '',
							),
						),
					),
					array(
						'key' => 'field_63eead7ed8fbe',
						'label' => 'Business Info',
						'name' => '',
						'aria-label' => '',
						'type' => 'tab',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'placement' => 'top',
						'endpoint' => 0,
						'selected' => 0,
					),
					array(
						'key' => 'field_63eeada0d8fbf',
						'label' => 'Business Information',
						'name' => 'business_information',
						'aria-label' => '',
						'type' => 'group',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'block',
						'sub_fields' => array(
							array(
								'key' => 'field_63eeadbcd8fc0',
								'label' => 'Address',
								'name' => 'address',
								'aria-label' => '',
								'type' => 'google_map',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'center_lat' => '',
								'center_lng' => '',
								'zoom' => '',
								'height' => '',
							),
							array(
								'key' => 'field_63eeae09d8fc1',
								'label' => 'Email',
								'name' => 'email',
								'aria-label' => '',
								'type' => 'email',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_63eeaf1ed8fc2',
								'label' => 'Phone',
								'name' => 'phone',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_63eeaf5cd8fc3',
								'label' => 'Fax',
								'name' => 'fax',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_63eeb116970e6',
								'label' => 'Social',
								'name' => '',
								'aria-label' => '',
								'type' => 'accordion',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'open' => 0,
								'multi_expand' => 0,
								'endpoint' => 0,
							),
							array(
								'key' => 'field_63eeaf65d8fc4',
								'label' => 'Social Media Links',
								'name' => 'social_media',
								'aria-label' => '',
								'type' => 'group',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'layout' => 'block',
								'sub_fields' => array(
									array(
										'key' => 'field_63eeafa0d8fc5',
										'label' => 'Facebook',
										'name' => 'facebook',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeafb0d8fc6',
										'label' => 'Twitter/X',
										'name' => 'twitter',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeafb7d8fc7',
										'label' => 'Instagram',
										'name' => 'instagram',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeafc6d8fc8',
										'label' => 'YouTube',
										'name' => 'youtube',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeafe0d8fc9',
										'label' => 'Google',
										'name' => 'google',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb000d8fcb',
										'label' => 'LinkedIn',
										'name' => 'linkedin',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb027d8fcd',
										'label' => 'Pinterest',
										'name' => 'pinterest',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeaff2d8fca',
										'label' => 'Discord',
										'name' => 'discord',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb015d8fcc',
										'label' => 'GitHub',
										'name' => 'github',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb040d8fce',
										'label' => 'Snapchat',
										'name' => 'snapchat',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb058d8fcf',
										'label' => 'Reddit',
										'name' => 'reddit',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb05fd8fd0',
										'label' => 'Twitch',
										'name' => 'twitch',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb0b9d8fd1',
										'label' => 'Tumblr',
										'name' => 'tumblr',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'field_63eeb0c2d8fd2',
										'label' => 'Etsy',
										'name' => 'etsy',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
									array(
										'key' => 'social_media_links_tiktok',
										'label' => 'TikTok',
										'name' => 'tiktok',
										'aria-label' => '',
										'type' => 'url',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'placeholder' => '',
									),
								),
							),
						),
					),
					array(
						'key' => 'field_63f7e3f5b6350',
						'label' => 'Design',
						'name' => '',
						'aria-label' => '',
						'type' => 'tab',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'placement' => 'top',
						'endpoint' => 0,
						'selected' => 0,
					),
					array(
						'key' => 'field_63f7e40ab6351',
						'label' => 'Design Settings',
						'name' => 'design_settings',
						'aria-label' => '',
						'type' => 'group',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => '',
						'sub_fields' => array(
							array(
								'key' => 'field_64625ecb1713f',
								'label' => 'Logo',
								'name' => 'logo',
								'aria-label' => '',
								'type' => 'image',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'id',
								'library' => 'all',
								'min_width' => '',
								'min_height' => '',
								'min_size' => '',
								'max_width' => '',
								'max_height' => '',
								'max_size' => '',
								'mime_types' => '',
								'preview_size' => 'medium',
							),
							array(
								'key' => 'field_64625f3117140',
								'label' => 'Alternate Logo',
								'name' => 'alternate_logo',
								'aria-label' => '',
								'type' => 'image',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'id',
								'library' => 'all',
								'min_width' => '',
								'min_height' => '',
								'min_size' => '',
								'max_width' => '',
								'max_height' => '',
								'max_size' => '',
								'mime_types' => '',
								'preview_size' => 'medium',
							),
							array(
								'key' => 'field_64625f3e17141',
								'label' => 'Favicon',
								'name' => 'favicon',
								'aria-label' => '',
								'type' => 'image',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'return_format' => 'id',
								'library' => 'all',
								'min_width' => '',
								'min_height' => '',
								'min_size' => '',
								'max_width' => '',
								'max_height' => '',
								'max_size' => '',
								'mime_types' => '',
								'preview_size' => 'medium',
							),
							array(
								'key' => 'field_63f7e40ab6355',
								'label' => 'Template',
								'name' => 'template',
								'aria-label' => '',
								'type' => 'select',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'choices' => array(
									'default' => 'Default',
									'nonprofit' => 'Non-Profit',
								),
								'default_value' => 'default',
								'return_format' => 'value',
								'multiple' => 0,
								'allow_null' => 0,
								'ui' => 0,
								'ajax' => 0,
								'placeholder' => '',
							),
							array(
								'key' => 'field_63f7e40ab6357',
								'label' => 'Colors',
								'name' => 'colors',
								'aria-label' => '',
								'type' => 'group',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'layout' => '',
								'sub_fields' => array(
									array(
										'key' => 'field_63f7e5ebe8f33',
										'label' => 'Primary',
										'name' => 'primary',
										'aria-label' => '',
										'type' => 'color_picker',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'enable_opacity' => 0,
										'return_format' => 'string',
									),
									array(
										'key' => 'field_63f7e5f7e8f37',
										'label' => 'Secondary',
										'name' => 'secondary',
										'aria-label' => '',
										'type' => 'color_picker',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'enable_opacity' => 0,
										'return_format' => 'string',
									),
									array(
										'key' => 'field_63f7e602e8f3f',
										'label' => 'Accent 1',
										'name' => 'accent_1',
										'aria-label' => '',
										'type' => 'color_picker',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'enable_opacity' => 0,
										'return_format' => 'string',
									),
									array(
										'key' => 'field_63f7e605e8f43',
										'label' => 'Accent 2',
										'name' => 'accent_2',
										'aria-label' => '',
										'type' => 'color_picker',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => '',
										'enable_opacity' => 0,
										'return_format' => 'string',
									),
									array(
										'key' => 'field_64626629dad1a',
										'label' => 'Percentage Variations',
										'name' => 'percentage_variations',
										'aria-label' => '',
										'type' => 'number',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'default_value' => 15,
										'min' => '',
										'max' => '',
										'placeholder' => '',
										'step' => '',
										'prepend' => '',
										'append' => '',
									),
								),
							),
							array(
								'key' => 'field_63f7e83f75989',
								'label' => 'Typography',
								'name' => 'typography',
								'aria-label' => '',
								'type' => 'group',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'layout' => 'block',
								'sub_fields' => array(
									array(
										'key' => 'field_63f7e85c7598a',
										'label' => 'Fonts',
										'name' => 'fonts',
										'aria-label' => '',
										'type' => 'repeater',
										'instructions' => 'These fonts will be downloaded and be available for use in the below settings.',
										'required' => 1,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'layout' => 'table',
										'pagination' => 0,
										'min' => 1,
										'max' => 0,
										'collapsed' => '',
										'button_label' => 'Add Row',
										'rows_per_page' => 20,
										'sub_fields' => array(
											array(
												'key' => 'field_63f7e8e47598b',
												'label' => 'Family',
												'name' => 'family',
												'aria-label' => '',
												'type' => 'select',
												'instructions' => '',
												'required' => 1,
												'conditional_logic' => 0,
												'wrapper' => array(
													'width' => '',
													'class' => '',
													'id' => '',
												),
												'choices' => [],
												'default_value' => false,
												'return_format' => 'value',
												'multiple' => 0,
												'allow_null' => 0,
												'allow_in_bindings' => 1,
												'ui' => 1,
												'ajax' => 0,
												'placeholder' => '',
												'parent_repeater' => 'field_63f7e85c7598a',
											),
											array(
												'key' => 'field_63fe7a6e53083',
												'label' => 'Weights',
												'name' => 'weights',
												'aria-label' => '',
												'type' => 'text',
												'instructions' => 'Comma-separated list of font variants (find font variants at Google Fonts)',
												'required' => 1,
												'conditional_logic' => array(
													array(
														array(
															'field' => 'field_63f7e8e47598b',
															'operator' => '!=empty',
														),
													),
												),
												'wrapper' => array(
													'width' => '',
													'class' => '',
													'id' => '',
												),
												'default_value' => '',
												'maxlength' => '',
												'allow_in_bindings' => 1,
												'placeholder' => '200,300,400,500',
												'prepend' => '',
												'append' => '',
												'parent_repeater' => 'field_63f7e85c7598a',
											),
											array(
												'key' => 'field_642d951e23ef8',
												'label' => 'Fallback Font',
												'name' => 'fallback_font',
												'aria-label' => '',
												'type' => 'select',
												'instructions' => '',
												'required' => 1,
												'conditional_logic' => 0,
												'wrapper' => array(
													'width' => '',
													'class' => '',
													'id' => '',
												),
												'choices' => array(
													'arial' => 'Arial (sans-serif)',
													'arial_black' => 'Arial Black (sans-serif)',
													'baskerville' => 'Baskerville (serif)',
													'bradley_hand' => 'Bradley Hand (cursive)',
													'brush_script_mt' => 'Brush Script MT (cursive)',
													'comic_sans_ms' => 'Comic Sans MS (cursive)',
													'courier' => 'Courier (monospace)',
													'courier_new' => 'Courier New (monospace)',
													'garamond' => 'Garamond (serif)',
													'georgia' => 'Georgia (serif)',
													'gill_sans' => 'Gill Sans (sans-serif)',
													'helvetica' => 'Helvetica (sans-serif)',
													'impact' => 'Impact (sans-serif)',
													'lucida' => 'Lucida (monospace)',
													'monaco' => 'Monaco (monospace)',
													'palatino' => 'Palatino (serif)',
													'tahoma' => 'Tahoma (sans-serif)',
													'times_new_roman' => 'Times New Roman (serif)',
													'trebuchet_ms' => 'Trebuchet MS (sans-serif)',
													'verdana' => 'Verdana (sans-serif)',
												),
												'default_value' => false,
												'return_format' => 'value',
												'multiple' => 0,
												'allow_null' => 0,
												'allow_in_bindings' => 1,
												'ui' => 0,
												'ajax' => 0,
												'placeholder' => '',
												'parent_repeater' => 'field_63f7e85c7598a',
											),
										),
									),
									array(
										'key' => 'field_642dd75b7c208',
										'label' => 'Other Fonts',
										'name' => 'other_fonts',
										'aria-label' => '',
										'type' => 'repeater',
										'instructions' => 'Enter the name of any Google Font families that are not included in the Web Fonts API (i.e; Tilt Warp). Fonts must be entered exactly as they are displayed in Google Fonts. Verify your spelling/capitalization at https://fonts.google.com/.',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'layout' => 'table',
										'pagination' => 0,
										'min' => 0,
										'max' => 0,
										'collapsed' => '',
										'button_label' => 'Add Row',
										'rows_per_page' => 20,
										'sub_fields' => array(
											array(
												'key' => 'field_642dd7ba7c209',
												'label' => 'Font',
												'name' => 'font',
												'aria-label' => '',
												'type' => 'text',
												'instructions' => '',
												'required' => 1,
												'conditional_logic' => 0,
												'wrapper' => array(
													'width' => '',
													'class' => '',
													'id' => '',
												),
												'default_value' => '',
												'maxlength' => '',
												'allow_in_bindings' => 1,
												'placeholder' => '',
												'prepend' => '',
												'append' => '',
												'parent_repeater' => 'field_642dd75b7c208',
											),
											array(
												'key' => 'field_642dd8be0779d',
												'label' => 'Fallback Font',
												'name' => 'fallback_font',
												'aria-label' => '',
												'type' => 'select',
												'instructions' => '',
												'required' => 1,
												'conditional_logic' => 0,
												'wrapper' => array(
													'width' => '',
													'class' => '',
													'id' => '',
												),
												'choices' => array(
													'arial' => 'Arial (sans-serif)',
													'arial_black' => 'Arial Black (sans-serif)',
													'baskerville' => 'Baskerville (serif)',
													'bradley_hand' => 'Bradley Hand (cursive)',
													'brush_script_mt' => 'Brush Script MT (cursive)',
													'comic_sans_ms' => 'Comic Sans MS (cursive)',
													'courier' => 'Courier (monospace)',
													'courier_new' => 'Courier New (monospace)',
													'garamond' => 'Garamond (serif)',
													'georgia' => 'Georgia (serif)',
													'gill_sans' => 'Gill Sans (sans-serif)',
													'helvetica' => 'Helvetica (sans-serif)',
													'impact' => 'Impact (sans-serif)',
													'lucida' => 'Lucida (monospace)',
													'monaco' => 'Monaco (monospace)',
													'palatino' => 'Palatino (serif)',
													'tahoma' => 'Tahoma (sans-serif)',
													'times_new_roman' => 'Times New Roman (serif)',
													'trebuchet_ms' => 'Trebuchet MS (sans-serif)',
													'verdana' => 'Verdana (sans-serif)',
												),
												'default_value' => false,
												'return_format' => 'value',
												'multiple' => 0,
												'allow_null' => 0,
												'allow_in_bindings' => 1,
												'ui' => 0,
												'ajax' => 0,
												'placeholder' => '',
												'parent_repeater' => 'field_642dd75b7c208',
											),
										),
									),
									array(
										'key' => 'field_63f7e98e7598c',
										'label' => 'Heading Font',
										'name' => 'heading_font',
										'aria-label' => '',
										'type' => 'select',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'choices' => array(
											'Arial, sans-serif' => 'Arial',
											'Verdana, sans-serif' => 'Verdana',
											'Helvetica, sans-serif' => 'Helvetica',
											'Times New Roman, serif' => 'Times New Roman',
											'Georgia, serif' => 'Georgia',
											'Courier New, monospace' => 'Courier New',
											'Tahoma, sans-serif' => 'Tahoma',
											'Trebuchet MS, sans-serif' => 'Trebuchet MS',
											'Lucida Console, monospace' => 'Lucida Console',
											'Segoe UI, sans-serif' => 'Segoe UI',
											'System-ui, sans-serif' => 'System-ui',
										),
										'default_value' => false,
										'return_format' => 'value',
										'multiple' => 0,
										'allow_null' => 0,
										'ui' => 0,
										'ajax' => 0,
										'placeholder' => '',
									),
									array(
										'key' => 'field_63f7e9a77598d',
										'label' => 'Body Font',
										'name' => 'body_font',
										'aria-label' => '',
										'type' => 'select',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'choices' => array(
											'Arial, sans-serif' => 'Arial',
											'Verdana, sans-serif' => 'Verdana',
											'Helvetica, sans-serif' => 'Helvetica',
											'Times New Roman, serif' => 'Times New Roman',
											'Georgia, serif' => 'Georgia',
											'Courier New, monospace' => 'Courier New',
											'Tahoma, sans-serif' => 'Tahoma',
											'Trebuchet MS, sans-serif' => 'Trebuchet MS',
											'Lucida Console, monospace' => 'Lucida Console',
											'Segoe UI, sans-serif' => 'Segoe UI',
											'System-ui, sans-serif' => 'System-ui',
										),
										'default_value' => false,
										'return_format' => 'value',
										'multiple' => 0,
										'allow_null' => 0,
										'ui' => 0,
										'ajax' => 0,
										'placeholder' => '',
									),
									array(
										'key' => 'field_642d954223ef9',
										'label' => 'Custom Font #1',
										'name' => 'custom_font_1',
										'aria-label' => '',
										'type' => 'select',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'choices' => array(
											'Arial, sans-serif' => 'Arial',
											'Verdana, sans-serif' => 'Verdana',
											'Helvetica, sans-serif' => 'Helvetica',
											'Times New Roman, serif' => 'Times New Roman',
											'Georgia, serif' => 'Georgia',
											'Courier New, monospace' => 'Courier New',
											'Tahoma, sans-serif' => 'Tahoma',
											'Trebuchet MS, sans-serif' => 'Trebuchet MS',
											'Lucida Console, monospace' => 'Lucida Console',
											'Segoe UI, sans-serif' => 'Segoe UI',
											'System-ui, sans-serif' => 'System-ui',
										),
										'default_value' => false,
										'return_format' => 'value',
										'multiple' => 0,
										'allow_null' => 0,
										'ui' => 0,
										'ajax' => 0,
										'placeholder' => '',
									),
									array(
										'key' => 'field_642d956b23efa',
										'label' => 'Custom Font #2',
										'name' => 'custom_font_2',
										'aria-label' => '',
										'type' => 'select',
										'instructions' => '',
										'required' => 0,
										'conditional_logic' => 0,
										'wrapper' => array(
											'width' => '',
											'class' => '',
											'id' => '',
										),
										'choices' => array(
											'Arial, sans-serif' => 'Arial',
											'Verdana, sans-serif' => 'Verdana',
											'Helvetica, sans-serif' => 'Helvetica',
											'Times New Roman, serif' => 'Times New Roman',
											'Georgia, serif' => 'Georgia',
											'Courier New, monospace' => 'Courier New',
											'Tahoma, sans-serif' => 'Tahoma',
											'Trebuchet MS, sans-serif' => 'Trebuchet MS',
											'Lucida Console, monospace' => 'Lucida Console',
											'Segoe UI, sans-serif' => 'Segoe UI',
											'System-ui, sans-serif' => 'System-ui',
										),
										'default_value' => false,
										'return_format' => 'value',
										'multiple' => 0,
										'allow_null' => 0,
										'ui' => 0,
										'ajax' => 0,
										'placeholder' => '',
									),
								),
							),
						),
					),
					array(
						'key' => 'field_63f7eeba60e39',
						'label' => 'Integrations',
						'name' => '',
						'aria-label' => '',
						'type' => 'tab',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'placement' => 'top',
						'endpoint' => 0,
						'selected' => 0,
					),
					array(
						'key' => 'field_63f7eed560e3a',
						'label' => 'Google Webfonts API Key',
						'name' => 'google_webfonts_api_key',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_643718680017a',
						'label' => 'Analytics',
						'name' => '',
						'aria-label' => '',
						'type' => 'tab',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'placement' => 'top',
						'endpoint' => 0,
						'selected' => 0,
					),
					array(
						'key' => 'field_65f31922383f1',
						'label' => 'Analytics Automation',
						'name' => 'analytics_automation',
						'aria-label' => '',
						'type' => 'group',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'block',
						'sub_fields' => array(
							array(
								'key' => 'field_65f3194a383f2',
								'label' => 'Account ID',
								'name' => 'account_id',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => 'The account ID number from Google Tag Manager.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => 'Required for Automation',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_65f319e3383f3',
								'label' => 'Container ID',
								'name' => 'container_id',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => 'The container ID number from Google Tag Manager.',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => 'Required for Automation',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_65f31a43383f4',
								'label' => 'GTM Public ID',
								'name' => 'gtm_public_id',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_65f38d1393b30',
								'label' => 'GA4 Measurement ID',
								'name' => 'ga4_measurement_id',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
							),
							array(
								'key' => 'field_65f31b1bf714c',
								'label' => 'Run Automation',
								'name' => 'run_automation',
								'aria-label' => '',
								'type' => 'true_false',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'message' => 'Enabling will cause the LeadFuel Analytics Automation to run when the site settings are saved.',
								'default_value' => 0,
								'ui' => 0,
								'ui_on_text' => '',
								'ui_off_text' => '',
							),
						),
					),
					array(
						'key' => 'field_6490bf90dc0e7',
						'label' => 'Other Installations',
						'name' => 'other_installations',
						'aria-label' => '',
						'type' => 'repeater',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'table',
						'pagination' => 0,
						'min' => 0,
						'max' => 0,
						'collapsed' => '',
						'button_label' => 'Add Row',
						'rows_per_page' => 20,
						'sub_fields' => array(
							array(
								'key' => 'field_6490bfa3dc0e8',
								'label' => 'Type',
								'name' => 'type',
								'aria-label' => '',
								'type' => 'select',
								'instructions' => '',
								'required' => 1,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'choices' => array(
									'ga4' => 'Google Analytics 4',
									'gtm' => 'Google Tag Manager',
									'meta_pixels' => 'Meta Pixels',
									'callrail' => 'CallRail',
									'clarity' => 'Clarity',
								),
								'default_value' => false,
								'return_format' => 'value',
								'multiple' => 0,
								'allow_null' => 0,
								'allow_in_bindings' => 1,
								'ui' => 0,
								'ajax' => 0,
								'placeholder' => '',
								'parent_repeater' => 'field_6490bf90dc0e7',
							),
							array(
								'key' => 'field_6490bfe5dc0e9',
								'label' => 'ID',
								'name' => 'id',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 1,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
								'parent_repeater' => 'field_6490bf90dc0e7',
							),
						),
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'options_page',
							'operator' => '==',
							'value' => 'site_settings',
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			)
		);
	}

	/**
	 * Retrieves the Google API key for ACF integrations.
	 *
	 * Fetches the API token from environment variables for use in Google Maps fields.
	 *
	 * @return array An array containing the Google API key.
	 */
	public function acf_get_google_api() {
		$api['key'] = GOOGLE_API_TOKEN;
		return $api;
	}

	/**
	 * Sets the Google API key for ACF.
	 *
	 * Updates ACF settings to use the predefined Google API key for integrations.
	 *
	 * @return void
	 */
	public function acf_set_google_api() {
		acf_update_setting( 'google_api_key', GOOGLE_API_TOKEN );
	}
}
