<?php
/**
 * Sentry
 *
 * Integrates Sentry error tracking into WordPress, with custom handling
 * for excluding specific errors.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class Sentry
 *
 * Handles the integration of Sentry error tracking and custom error filtering.
 */
class Sentry extends \StrategySuite\Module {

	/**
	 * Determines if the module can be registered.
	 *
	 * Checks if the Sentry integration is already loaded.
	 *
	 * @return bool True if Sentry is loaded, otherwise false.
	 */
	public function can_register() {
		return defined( 'WP_SENTRY_LOADED' ) && WP_SENTRY_LOADED;
	}

	/**
	 * Registers hooks for filtering errors sent to Sentry.
	 *
	 * This function adds a filter to exclude errors related to The Events Calendar plugin.
	 *
	 * @return void
	 */
	public function register() {
		add_filter( 'wp_sentry_before_send', [ $this, 'do_not_send_tec_errors_to_sentry' ], 10, 2 );
		add_filter( 'wp_sentry_public_options', [ $this, 'ignore_outlook_protection_errors' ], 10, 1 );
		// This one will ignore errors that occur from third-party domains.
		add_action(
			'wp_enqueue_scripts',
			function () {
				$site_url  = home_url();
				$parsed    = parse_url( $site_url );
				$site_host = $parsed['host'] ?? '';

				$js = <<<JS
function wp_sentry_hook(options) {
	options.beforeSend = function (event, hint) {
		// Example: ignore timeouts
		if (hint.originalException === "Timeout") {
			return null;
		}

		// Ignore errors from third-party domains
		const frames = event.exception?.values?.[0]?.stacktrace?.frames || [];
		const lastFrame = frames.length ? frames[frames.length - 1] : {};
		const source = lastFrame.filename || '';
		const allowedHost = '{$site_host}';

		function isFromAllowedDomain(src) {
			try {
				const url = new URL(src);
				return url.hostname === allowedHost || url.hostname.endsWith('.' + allowedHost);
			} catch (e) {
				return false;
			}
		}

		if (source && !isFromAllowedDomain(source)) {
			return null;
		}

		return event;
	};
}
JS;

				wp_add_inline_script( 'wp-sentry-browser', $js, 'before' );
			}
		);
	}

	/**
	 * Filters errors sent to Sentry to exclude those from The Events Calendar.
	 *
	 * This function prevents errors originating from 'events-calendar-pro'
	 * and 'the-events-calendar' from being reported to Sentry.
	 *
	 * @param \Sentry\Event     $event The event object being sent to Sentry.
	 * @param \Sentry\EventHint $hint The event hint, which contains exception data.
	 *
	 * @return \Sentry\Event|null The modified event or null to exclude it.
	 */
	public function do_not_send_tec_errors_to_sentry( $event, $hint ) {
		// Ensure we have an exception object
		if ( isset( $hint->exception ) && method_exists( $hint->exception, 'getFile' ) ) {
			$file = $hint->exception->getFile();

			if ( strpos( $file, 'events-calendar-pro' ) !== false || strpos( $file, 'the-events-calendar' ) !== false ) {
				return null;
			}
		}

		// Check if the error message itself contains a reference to The Events Calendar
		if ( isset( $event->message ) ) {
			if ( strpos( $event->message, 'the-events-calendar' ) !== false || strpos( $event->message, 'events-calendar-pro' ) !== false ) {
				return null;
			}
		}

		// Check stack trace for file paths
		if ( isset( $event->exception ) && isset( $event->exception->values ) ) {
			foreach ( $event->exception->values as $value ) {
				if ( isset( $value->stacktrace->frames ) ) {
					foreach ( $value->stacktrace->frames as $frame ) {
						if ( isset( $frame->filename ) &&
							( strpos( $frame->filename, 'events-calendar-pro' ) !== false ||
							 strpos( $frame->filename, 'the-events-calendar' ) !== false ) ) {
							return null;
						}
					}
				}
			}
		}

		return $event;
	}

	/**
	 * Modify error handling options to ignore specific Outlook protection
	 * and third-party related error messages.
	 *
	 * This function updates the provided options array by adding entries
	 * to the 'ignoreErrors' key. The ignored errors include:
	 * - Non-Error promise rejection captured
	 * - [Cloudflare Turnstile]
	 *
	 * @param array $options Existing options for error handling.
	 * @return array Updated options including ignored error messages.
	 */
	public function ignore_outlook_protection_errors( $options ) {
		$options['ignoreErrors'] = [
			'Non-Error promise rejection captured',
			'[Cloudflare Turnstile]',
		];
		return $options;
	}
}
