<?php
/**
 * LoginPage
 *
 * This class customizes the WordPress login page by modifying the login logo,
 * its URL, and the title text. It allows the use of a custom logo set in the
 * WordPress Customizer or defaults to a predefined logo.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class LoginPage
 *
 * This class extends the StrategySuite\Module and manages the customization
 * of the WordPress login page appearance and behavior.
 */
class LoginPage extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register hooks for customizing the WordPress login page.
	 *
	 * This method registers filters and actions to change the login logo,
	 * its URL, and the logo title text.
	 *
	 * @return void
	 */
	public function register() {
		add_filter( 'login_headerurl', [ $this, 'change_login_logo_url' ] );
		add_filter( 'login_headertext', [ $this, 'change_login_logo_url_title' ] );
		add_action( 'login_head', [ $this, 'change_login_logo' ], PHP_INT_MAX );
	}

	/**
	 * Change the login logo URL to the site's home page.
	 *
	 * This modifies the default WordPress login logo URL to point to the site's home page.
	 *
	 * @since 1.0
	 * @return string The home URL of the WordPress site.
	 */
	public function change_login_logo_url() {
		return home_url();
	}

	/**
	 * Change the login logo title text to the site name.
	 *
	 * This modifies the default title text that appears when hovering over the login logo,
	 * replacing it with the site's name.
	 *
	 * @since 1.0
	 * @return string The name of the WordPress site.
	 */
	public function change_login_logo_url_title() {
		return get_bloginfo( 'name' );
	}

	/**
	 * Change the login logo to a custom logo set in the WordPress Customizer.
	 *
	 * If a custom logo is set in the Customizer, it will be used as the login logo.
	 * Otherwise, a default logo from the plugin's assets will be displayed.
	 *
	 * @since 1.0
	 * @return void
	 */
	public function change_login_logo() {
		if ( has_custom_logo() ) {
			$image = wp_get_attachment_image_src( get_theme_mod( 'custom_logo' ), 'full' );
		} else {
			$image = [
				STRATEGY_SUITE_DIST_URL . 'images/strategy-logo.png',
				166,
			];
		}
		?>
		<style type="text/css">
			.login h1 a {
				background-image: url(<?php echo esc_url( $image[0] ); ?>);
				-webkit-background-size: <?php echo absint( $image[1] ); ?>px;
				background-size: contain;
				width: 100%;
			}
			body {
				background: #e1e1e1;
			}
		</style>
		<?php
	}
}
