<?php
/**
 * Groundhogg
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use Elementor;

/**
 * Class Groundhogg
 *
 * Handles various integrations and customizations for the Groundhogg plugin.
 */
class Groundhogg extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * Checks if the Groundhogg plugin is active.
	 *
	 * @return bool Returns `true` if Groundhogg is active, otherwise `false`.
	 */
	public function can_register() {
		return is_plugin_active( 'groundhogg/groundhogg.php' );
	}

	/**
	 * Register various hooks for customizing Groundhogg functionality.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'groundhogg/replacements/business_address', [ $this, 'format_address' ] );
		add_action( 'groundhogg/email/before_send', [ $this, 'add_utm_parameters_to_email' ], 10, 5 );
		add_action( 'groundhogg/replacements/init', [ $this, 'add_site_logo_replacement' ], 1, 1 );
		add_action( 'admin_init', [ $this, 'maybe_preload_editor_colors' ] );
		add_action( 'groundhogg/email/after_send', [ $this, 'on_email_send' ] );
		add_action( 'rest_api_init', [ $this, 'email_counter_api' ] );
		add_action( 'wp_dashboard_setup', [ $this, 'register_email_count_dashboard_widget' ] );
		add_action( 'acf/save_post', [ $this, 'configure_white_label' ], 20 );
		add_action( 'admin_init', [ $this, 'configure_hollerbox' ] );
	}

	/**
	 * Formats the business address with line breaks for Groundhogg emails.
	 *
	 * @return string The formatted business address.
	 */
	public function format_address() {
		$address_1 = get_option( 'gh_street_address_1' ) ? get_option( 'gh_street_address_1' ) : '';
		$address_2 = get_option( 'gh_street_address_2' ) ? ', ' . get_option( 'gh_street_address_2' ) : '';
		$city = get_option( 'gh_city' ) ? get_option( 'gh_city' ) : '';
		$state = get_option( 'gh_region' ) ? ', ' . get_option( 'gh_region' ) : '';
		$zip = get_option( 'gh_zip_or_postal' ) ? ', ' . get_option( 'gh_zip_or_postal' ) : '';
		$country = get_option( 'gh_country' ) ? ', ' . get_option( 'gh_country' ) : '';

		return $address_1 . $address_2 . '<br>' . $city . $state . $zip . $country;
	}

	/**
	 * Adds UTM parameters to email links before sending.
	 *
	 * @param \Groundhogg_Email $email The email being sent.
	 * @param string            $to The email recipient.
	 * @param string            $subject The email subject.
	 * @param string            $content The email content.
	 * @param string            $headers The email headers.
	 *
	 * @return void
	 */
	public function add_utm_parameters_to_email( $email, &$to, &$subject, &$content, &$headers ) {
		$dom = new \DOMDocument();
		libxml_use_internal_errors( true );
		$dom->loadHTML( $content );
		libxml_clear_errors();
		$links = $dom->getElementsByTagName( 'a' );

		foreach ( $links as $link ) {
			if ( ! ( $link instanceof \DOMElement ) ) {
				continue;
			}

			$utm_params = [
				'utm_source' => 'Strategy Mail',
				'utm_medium' => 'Email',
				'utm_campaign' => $email->get_title() ? $email->get_title() : $subject,
				'utm_content' => $link->textContent, // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				// 'utm_term' => '' // This will be added later if it exists
			];

			$href = $link->getAttribute( 'href' );
			$parsed_url = parse_url( $href );

			$query_params = [];
			if ( isset( $parsed_url['query'] ) ) {
				parse_str( $parsed_url['query'], $query_params );
			}

			if ( isset( $parsed_url['host'] ) && parse_url( home_url(), PHP_URL_HOST ) !== $parsed_url['host'] ) {
				continue;
			}

			if ( isset( $parsed_url['scheme'] ) && in_array( $parsed_url['scheme'], [ 'tel', 'mailto' ] ) ) {
				continue;
			}

			if ( strpos( $href, 'unsubscribe' ) !== false || strpos( $href, 'privacy-policy' ) !== false ) {
				continue;
			}

			// Overwrite existing UTM params if already present in the URL
			foreach ( $utm_params as $key => $value ) {
				if ( isset( $query_params[ $key ] ) ) {
					$utm_params[ $key ] = $query_params[ $key ];
				}
			}

			// Merge existing query params with our UTM params
			$merged_params = array_merge( $query_params, $utm_params );
			$query_string = http_build_query( $merged_params );

			// Rebuild the URL parts safely
			$scheme = isset( $parsed_url['scheme'] ) ? $parsed_url['scheme'] . '://' : '';
			$host   = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
			$path   = isset( $parsed_url['path'] ) ? $parsed_url['path'] : '';
			$new_href = $scheme . $host . $path . '?' . $query_string;
			$link->setAttribute( 'href', $new_href );
		}

		$content = $dom->saveHTML();
	}

	/**
	 * Preloads editor colors if the Groundhogg email editor page is being accessed.
	 *
	 * @return void
	 */
	public function maybe_preload_editor_colors() {
		if ( isset( $_GET['page'] ) && 'gh_emails' === $_GET['page'] ) {
			$this->preload_editor_colors();
		}
	}

	/**
	 * Preloads and updates the email editor color palette.
	 *
	 * This function retrieves the saved email editor color palette and merges it with
	 * additional colors from the LeadGen settings. It ensures that all colors are valid
	 * hex codes and limits the final palette to a maximum of 8 unique colors.
	 *
	 * @return void
	 */
	private function preload_editor_colors() {
		$editor_colors = array_unique( array_map( 'strtolower', get_option( 'gh_email_editor_color_palette', [] ) ) );

		$color_filter = function ( $color ) {
			return is_string( $color ) && preg_match( '/^#([a-fA-F0-9]{3}|[a-fA-F0-9]{6})$/', $color );
		};

		$theme = wp_get_theme();

		if ( 'hello-elementor' === $theme->stylesheet || 'hello-elementor-child' === $theme->stylesheet ) {
			$elementor_colors = array_map(
				function ( $item ) {
					return strtolower( $item['color'] );
				},
				Elementor\Plugin::$instance->kits_manager->get_current_settings( 'system_colors' )
			);
			$editor_colors = array_unique( array_merge( $elementor_colors, $editor_colors ) );
		} elseif ( 'leadgen' === $theme->stylesheet ) {
			$leadgen_settings = get_field( 'design_settings', 'option' );
			$leadgen_colors = isset( $leadgen_settings['colors'] ) ? $leadgen_settings['colors'] : [];
			$filtered_leadgen_colors = array_values( array_filter( $leadgen_colors, $color_filter ) );
			$editor_colors = array_unique( array_merge( $filtered_leadgen_colors, $editor_colors ) );
		}

		update_option( 'gh_email_editor_color_palette', array_slice( $editor_colors, 0, 8 ) );
	}

	/**
	 * Adds a replacement for the site logo URL.
	 *
	 * This function retrieves the site logo from LeadGen settings or the theme's custom logo.
	 * If no logo is found, it falls back to a placeholder image. The final logo URL is then
	 * added as a replacement in the provided replacements object.
	 *
	 * @param object $replacements The replacements object to which the site logo URL will be added.
	 *
	 * @return void
	 */
	public function add_site_logo_replacement( $replacements ) {
		$placeholder_logo_url = 'https://placehold.co/320x70.png?text=Site+Logo';
		$logo = null;
		$logo_url = null;

		$theme = wp_get_theme();

		if ( 'hello-elementor' === $theme->stylesheet || 'hello-elementor-child' === $theme->stylesheet ) {
			$logo = get_theme_mod( 'custom_logo' ) ? get_theme_mod( 'custom_logo' ) : '';
			$logo_url = $logo ? wp_get_attachment_url( $logo ) : $placeholder_logo_url;
		} elseif ( 'leadgen' === $theme->stylesheet ) {
			$leadgen_settings = get_field( 'design_settings', 'option' );
			$logo = isset( $leadgen_settings['logo'] ) ? $leadgen_settings['logo'] : [];
			if ( $logo ) {
				$logo_url = wp_get_attachment_url( $logo ) ? wp_get_attachment_url( $logo ) : wp_get_attachment_url( $theme_logo );
			}
		}

		$selected_logo_url = $logo_url ? $logo_url : $placeholder_logo_url;

		$callback = function () use ( $selected_logo_url ) {
			return $selected_logo_url;
		};

		$replacements->add(
			'site_logo',
			$callback,
			'The site logo URL',
			'Site Logo',
			'other',
			'',
			''
		);
	}

	/**
	 * Registers the REST API endpoint for retrieving email count.
	 *
	 * @return void
	 */
	public function email_counter_api() {
		register_rest_route(
			'email-counter/v1',
			'/count',
			array(
				'methods'               => 'GET',
				'callback'              => [ $this, 'get_count' ],
				'permission_callback'   => '\__return_true',
			)
		);
	}

	/**
	 * Increments the email counter when an email is sent.
	 *
	 * @param string $email The email sent.
	 *
	 * @return void
	 */
	public function on_email_send( $email ) {
		$this->increment_email_counter();
	}

	/**
	 * Increments the email counter for the current month.
	 *
	 * @return void
	 */
	private function increment_email_counter() {
		$data = get_option( 'strategy_mail_counter' );
		if ( ! $data ) {
			$data = [];
		}

		$current_date = gmdate( 'Y-m-d' );
		$year = gmdate( 'Y', strtotime( $current_date ) );
		$month = gmdate( 'm', strtotime( $current_date ) );

		if ( ! isset( $data[ $year ] ) ) {
			$data[ $year ] = [];
		}

		if ( ! isset( $data[ $year ][ $month ] ) ) {
			$data[ $year ][ $month ] = 0;
		}

		$data[ $year ][ $month ]++;
		update_option( 'strategy_mail_counter', $data );
	}

	/**
	 * Retrieves email count over a specific period.
	 *
	 * @param object $request The request data.
	 *
	 * @return \WP_REST_Response The response with email count data.
	 */
	public function get_count( $request ) {
		$response = [];
		$date1 = $request->get_param( 'date1' ) ?? false;
		$date2 = $request->get_param( 'date2' ) ?? false;
		$t1 = strtotime( $date1 );
		$t2 = strtotime( $date2 );
		if ( ! $date1 ) {
			$response = $this->add_error( $response, 'The date1 field must be defined.' );
		} else if ( ! $t1 ) {
			$response = $this->add_error( $response, 'The date1 field is not a valid date.' );
		}

		if ( $date2 ) {
			if ( ! $t2 ) {
				$response = $this->add_error( $response, 'The date2 field is not a valid date.' );
			} else if ( $t1 && $t2 && $t1 > $t2 ) {
				$response = $this->add_error( $response, 'The date1 field must be prior to the date2 field.' );
			}
		}

		if ( isset( $response['errors'] ) && is_array( $response['errors'] ) && 0 < count( $response['errors'] ) ) {
			$response['success'] = false;
			return new WP_REST_Response( $response, 400 );
		}

		$month_map = [ '01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12' ];
		$y1 = gmdate( 'Y', $t1 );
		$m1 = gmdate( 'm', $t1 );
		$response['count'] = 0;
		$data = get_option( 'strategy_mail_counter' );

		if ( $data ) {
			if ( $t2 ) {
				// Get count over range
				$y2 = gmdate( 'Y', $t2 );
				$m2 = gmdate( 'm', $t2 );
				$cur_month = intval( $m1 ) - 1;
				$target_month = intval( $m2 ) - 1;
				$cur_year = intval( $y1 );
				$target_year = intval( $y2 );
				while ( $cur_year < $target_year || $cur_month <= $target_month ) {
					$date = $month_map[ $cur_month ] . '/01/' . $cur_year;
					$response['count'] += $this->get_count_from_data( $data, $date );
					if ( $cur_month < count( $month_map ) - 1 ) {
						$cur_month++;
					} else {
						$cur_month = 0;
						$cur_year++;
					}
				}
			} else {
				// Get count of only m1
				$response['count'] += $this->get_count_from_data( $data, $date1 );
			}
		}

		$response['success'] = true;
		return new WP_REST_Response( $response, 200 );
	}

	/**
	 * Attempts to retrieve the email count from the data. If the count cannot be found, returns 0.
	 *
	 * @param array  $data the data
	 * @param Object $date the date
	 */
	private function get_count_from_data( $data, $date ) {
		$year = gmdate( 'Y', strtotime( $date ) );
		$month = gmdate( 'm', strtotime( $date ) );

		if ( is_array( $data ) && isset( $data[ $year ] ) && isset( $data[ $year ][ $month ] ) ) {
			return $data[ $year ][ $month ];
		}

		return 0;
	}

	/**
	 * Accepts a response array and a message. Appends the message to the errors component of the response.
	 *
	 * @param array  $response the response to be edited
	 * @param string $message the message to provide
	 */
	private function add_error( $response, $message ) {
		if ( isset( $response['errors'] ) ) {
			$response['errors'][] = $message;
		} else {
			$response['errors'] = [ $message ];
		}

		return $response;
	}

	/**
	 * Registers the dashboard widget to display email quota usage.
	 *
	 * @return void
	 */
	public function register_email_count_dashboard_widget() {
		wp_add_dashboard_widget(
			'email_count_dashboard_widget',
			'Email Quota',
			[ $this, 'display_email_count_dashboard_widget' ]
		);
	}

	/**
	 * Displays the email count in the WordPress dashboard widget.
	 *
	 * @return void
	 */
	public function display_email_count_dashboard_widget() {
		$data = get_option( 'strategy_mail_counter' );
		$count = $this->get_count_from_data( $data, gmdate( 'm/d/Y' ) );
		$quota = 100000;

		echo wp_kses_post( "<p><strong>Emails Sent This Month:</strong> $count / $quota</p>" );
	}

	/**
	 * Configures White Label settings for Groundhogg.
	 *
	 * @return void
	 */
	public function configure_white_label() {
		$screen = get_current_screen();
		if ( strpos( $screen->id, 'site_settings' ) == true ) {
			update_option( 'gh_white_label_template_site', 'https://leadfuel.us' );
			update_option( 'gh_white_label_enable', true );
			update_option( 'gh_white_label_branding_name', 'Strategy Mail' );
			update_option( 'gh_white_label_author_name', 'Strategy, LLC' );
			update_option( 'gh_white_label_author_uri', 'https://strategynewmedia.com' );
			update_option( 'gh_white_label_plugin_uri', 'https://strategynewmedia.com' );
		}
	}

	/**
	 * Configures Hollerbox Pro settings.
	 *
	 * @return void
	 */
	public function configure_hollerbox() {
		$option_key = 'hollerbox_settings';
		$existing = get_option( $option_key, [] );

		// Merge with updated values
		$updated = array_merge(
			$existing,
			[
				'license' => '32c911bb3f2fb43865ba14998cae34eb',
				'is_licensed' => true,
				'credit_disabled' => true,
				'telemetry_subscribed' => false,
				'telemetry_email' => 'developers@strategynewmedia.com',
				'license_expiry' => '2026-05-06 23:59:59',
			]
		);

		// Save the updated settings
		update_option( $option_key, $updated );
	}
}
