<?php
/**
 * GravitySMTP
 *
 * Handles SMTP configuration for Gravity Forms and manages default email settings.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class GravitySMTP
 *
 * This class integrates SMTP settings with Gravity Forms by setting default email configurations
 * and controlling which WordPress notifications should be sent or suppressed.
 */
class GravitySMTP extends \StrategySuite\Module {

	/**
	 * Determines if the module can be registered.
	 *
	 * This checks if the GravitySMTP plugin is active before proceeding with registration.
	 *
	 * @return bool True if the GravitySMTP plugin is active, otherwise false.
	 */
	public function can_register() {
		return is_plugin_active( 'gravitysmtp/gravitysmtp.php' );
	}

	/**
	 * Registers hooks for setting default SMTP options in WordPress.
	 *
	 * This ensures that SMTP settings and email notification preferences are configured
	 * if they haven't been set already.
	 *
	 * @return void
	 */
	public function register() {
		if ( 1 == get_option( 'lg-gravitysmtp-automation-run' ) ) {
			return;
		}

		add_action( 'admin_init', [ $this, 'add_default_options' ] );
	}

	/**
	 * Adds default SMTP configuration and notification preferences.
	 *
	 * This method sets:
	 * - Default sender name and email for Amazon SES and generic SMTP configurations.
	 * - Email notifications settings, enabling or disabling specific system emails.
	 * - A flag to prevent redundant executions of this automation.
	 *
	 * @return void
	 */
	public function add_default_options() {
		// Constants
		update_option(
			'gravitysmtp_amazon',
			json_encode(
				[
					'from_name' => get_bloginfo( 'name' ),
					'from_email' => 'no-reply@' . parse_url( get_bloginfo( 'url' ), PHP_URL_HOST ),
				]
			)
		);
		update_option( 'gravitysmtp_generic', json_encode( [ 'from_name' => get_bloginfo( 'name' ) ] ) );

		$email_notifications = [
			'gravitysmtp_email_stopper_change_of_site_admin_email_address_is_attempted' => true,
			'gravitysmtp_email_stopper_site_admin_email_address_is_changed' => true,
			'gravitysmtp_email_stopper_automatic_plugin_updates' => false,
			'gravitysmtp_email_stopper_automatic_theme_updates' => false,
			'gravitysmtp_email_stopper_automatic_core_update' => false,
			'gravitysmtp_email_stopper_full_log_of_background_update_results' => false,
			'gravitysmtp_email_stopper_comment_is_awaiting_moderation' => false,
			'gravitysmtp_email_stopper_comment_is_published' => false,
			'gravitysmtp_email_stopper_a_new_user_is_created_admin' => false,
			'gravitysmtp_email_stopper_a_new_user_is_created_user' => true,
			'gravitysmtp_email_stopper_user_confirms_personal_data_export_or_erasure_request' => true,
			'gravitysmtp_email_stopper_site_admin_sends_link_to_a_personal_data_export' => true,
			'gravitysmtp_email_stopper_site_admin_erases_personal_data_to_fulfill_a_data_erasure_request' => true,
			'gravitysmtp_email_stopper_user_or_administrator_requests_a_password_reset' => true,
			'gravitysmtp_email_stopper_user_resets_their_password' => false,
			'gravitysmtp_email_stopper_user_changes_their_password' => true,
			'gravitysmtp_email_stopper_user_attempts_to_change_their_email_address' => true,
			'gravitysmtp_email_stopper_user_changes_their_email_address' => true,
		];
		update_option( 'gravitysmtp_config', json_encode( $email_notifications ) );

		// Automation
		update_option( 'lg-gravitysmtp-automation-run', 1 );
	}
}
