<?php
/**
 * GravityFormsPipedrive
 *
 * Connects Gravity Forms entries to Pipedrive leads.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Handles Gravity Forms → Pipedrive integration.
 *
 * This module registers form-level and field-level settings to allow toggling Pipedrive
 * integration per form, mapping specific Gravity Form fields to Pipedrive fields, and
 * automatically creating leads in Pipedrive when a form is submitted.
 */
class GravityFormsPipedrive extends \StrategySuite\Module {

	/**
	 * The Pipedrive_Client instance.
	 *
	 * @var Pipedrive_Client
	 */
	private $client;

	/**
	 * Determines if the module can be registered.
	 *
	 * Checks if Gravity Forms is active and if a Pipedrive API token is defined.
	 *
	 * @return bool True if module requirements are met, false otherwise.
	 */
	public function can_register() {
		return is_plugin_active( 'gravityforms/gravityforms.php' ) && defined( 'AMELIA_PIPEDRIVE_API_TOKEN' )
			? AMELIA_PIPEDRIVE_API_TOKEN
			: getenv( 'AMELIA_PIPEDRIVE_API_TOKEN' );
	}

	/**
	 * Registers all hooks for GravityFormsPipedrive.
	 *
	 * @return void
	 */
	public function register() {
		$this->set_pipedrive_client();

		// Form Settings
		add_filter( 'gform_form_settings_fields', [ $this, 'add_settings_field' ], 100, 2 );

		// Field Mapping
		add_action( 'gform_field_advanced_settings', array( $this, 'add_pipedrive_field_setting' ), 10, 2 );
		add_action( 'gform_editor_js', array( $this, 'add_pipedrive_field_setting_js' ) );

		// Submission
		add_action( 'gform_after_submission', array( $this, 'retrieve_form_data' ), 10, 2 );
	}

	/**
	 * Creates and stores a Pipedrive_Client instance.
	 *
	 * @return void
	 */
	private function set_pipedrive_client() {
		if ( is_null( $this->client ) ) {
			require_once STRATEGY_SUITE_INC . '/lib/PipedriveClient.php';
			$this->client = Pipedrive_Client::create_client();
		}
	}

	/**
	 * Adds the "Enable Pipedrive Integration" toggle to Gravity Forms form settings.
	 *
	 * @see https://docs.gravityforms.com/gform_form_settings_fields/
	 *
	 * @param array $fields Current form settings fields.
	 * @param array $form   Current form object.
	 *
	 * @return array Modified form settings fields.
	 */
	public function add_settings_field( $fields, $form ) {
		$fields['form_options']['fields'][] = array(
			'name'          => 'pipedrive_enabled',
			'type'          => 'toggle',
			'label'         => esc_html__( 'Enable Pipedrive Integration' ),
			'default_value' => apply_filters( 'gf_pipedrive_enabled_default', false, $form ),
		);

		return $fields;
	}

	/**
	 * Outputs the Pipedrive mapping dropdown in field advanced settings.
	 *
	 * @param int $position Current position in advanced settings.
	 * @param int $form_id  Current form ID.
	 *
	 * @return void
	 */
	public function add_pipedrive_field_setting( $position, $form_id ) {
		if ( 50 == $position ) {
			?>
			<li class="pipedrive_field_setting field_setting">
				<label for="field_pipedrive_map" class="section_label">
					<?php esc_html_e( 'Pipedrive Mapping', 'strategy-suite' ); ?>
					<?php gform_tooltip( 'form_field_pipedrive_map' ); ?>
				</label>
				<select id="field_pipedrive_map" class="field_pipedrive_map_setting">
					<option value=""><?php esc_html_e( 'Do not map', 'strategy-suite' ); ?></option>
					<?php
					// Auto-populate with Pipedrive fields
					$legacy_fields = array(
						'company' => 'Organization Name',
						'name'    => 'Person Name',
						'email'   => 'Person Email',
						'phone'   => 'Person Phone',
					);

					$pipedrive_fields = [
						'lead'         => $this->client->get_fields( 'lead' ),
						'person'       => $this->client->get_fields( 'person' ),
						'organization' => $this->client->get_fields( 'organization' ),
					];

					$pipedrive_fields = get_transient( 'pipedrive_field_cache' );

					if ( ! $pipedrive_fields ) {
						$pipedrive_fields = [
							'lead'         => $this->client->get_fields( 'lead' ),
							'person'       => $this->client->get_fields( 'person' ),
							'organization' => $this->client->get_fields( 'organization' ),
						];
						set_transient( 'pipedrive_field_cache', $pipedrive_fields, HOUR_IN_SECONDS );
					}

					echo '<optgroup label="Legacy Fields">';
					foreach ( $legacy_fields as $key => $label ) {
						echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
					}
					echo '</optgroup>';

					foreach ( $pipedrive_fields as $group_label => $fields ) {
						if ( empty( $fields ) ) {
							continue;
						}
						echo '<optgroup label="' . esc_attr( ucfirst( $group_label ) . ' Fields' ) . '">';
						foreach ( $fields as $key => $label ) {
							$prefixed = "{$group_label}_{$key}";
							echo '<option value="' . esc_attr( $prefixed ) . '">' . esc_html( $label ) . '</option>';
						}
						echo '</optgroup>';
					}
					?>
				</select>
			</li>
			<?php
		}
	}

	/**
	 * Injects JavaScript into the form editor to handle saving/loading field mappings.
	 *
	 * @return void
	 */
	public function add_pipedrive_field_setting_js() {
		?>
		<script type="text/javascript">
			// Initialize setting in editor
			fieldSettings.pipedrive = ".pipedrive_field_setting";

			// Load setting value when editing a field
			jQuery(document).on("gform_load_field_settings", function(event, field, form){
				jQuery("#field_pipedrive_map").val(field.pipedriveMap || "");

				// Hide the mapping dropdown if integration is not enabled
				if (form && form.pipedrive_enabled) {
					jQuery(".pipedrive_field_setting").show();
				} else {
					jQuery(".pipedrive_field_setting").hide();
				}
			});

			// Save setting when field is updated
			jQuery("#field_pipedrive_map").on("change", function(){
				SetFieldProperty("pipedriveMap", jQuery(this).val());
			});
		</script>
		<?php
	}

	/**
	 * Processes form submission: packages data, maps fields, and attempts to create a lead in Pipedrive.
	 * Adds a note to the Gravity Forms entry with the result of the Pipedrive integration.
	 *
	 * @param array $entry The submitted form entry.
	 * @param array $form  The form object.
	 *
	 * @return void
	 */
	public function retrieve_form_data( $entry, $form ) {
		if ( empty( $form['pipedrive_enabled'] ) ) {
			return;
		}

		$package = $this->package_form_submission( $entry, $form['fields'] );

		$lead_fields  = [];
		$person_fields = [];
		$org_fields   = [];
		$notes        = [];

		foreach ( $package['data'] as $item ) {
			$map = null;

			foreach ( $form['fields'] as $field ) {
				if ( (string) $field->id === (string) $item['id'] ) {
					$map = $this->normalize_legacy_map( rgar( $field, 'pipedriveMap' ) );
					break;
				}
			}

			if ( empty( $map ) ) {
				$notes[] = $item;
				continue;
			}

			$value = is_array( $item['value'] ) ? implode( ' ', wp_list_pluck( $item['value'], 'value' ) ) : $item['value'];

			if ( str_starts_with( $map, 'lead_' ) ) {
				$key = substr( $map, 5 );
				$lead_fields[ $key ] = $value;
			} elseif ( str_starts_with( $map, 'person_' ) ) {
				$key = substr( $map, 7 );
				$person_fields[ $key ] = $value;
			} elseif ( str_starts_with( $map, 'org_' ) || str_starts_with( $map, 'organization_' ) ) {
				$key = preg_replace( '/^(org|organization)_/', '', $map );
				$org_fields[ $key ] = $value;
			} else {
				$notes[] = $item;
			}
		}

		$note_text = $this->format_packaged_data( $notes );

		$lead = $this->client->construct_lead_from_groups( $lead_fields, $person_fields, $org_fields, $note_text );

		if ( $lead && isset( $lead['id'] ) ) {
			$note_text = sprintf(
				'Lead was created in Pipedrive. <a href="https://pipedrive.com/leads/%s" target="_blank">View Lead</a>',
				esc_html( $lead['id'] )
			);
			$sub_type = 'success';
		} else {
			$note_text = 'Lead was not created in Pipedrive due to an error.';
			$sub_type = 'error';
		}

		// Add the note to the Gravity Forms entry
		\GFAPI::add_note(
			$entry['id'],
			get_current_user_id(), // user_id, or 0 for system
			'Pipedrive Integration',
			$note_text,
			'pipedrive',
			$sub_type
		);
	}

	/**
	 * Packages a form submission into a structured array of field data.
	 *
	 * Handles multi-input fields (like Name and Address), serialized values (like checkboxes),
	 * and separates visible data fields from hidden/admin fields.
	 *
	 * @param array $entry  Gravity Forms entry values.
	 * @param array $fields Gravity Forms field objects.
	 *
	 * @return array Structured package with "data" and "admin" keys.
	 */
	private function package_form_submission( $entry, $fields ) {

		$package = array(
			'data'  => array(),
			'admin' => array(),
		);
		foreach ( $fields as $field ) {
			$package_item = array(
				'label' => $field->label,
				'type'  => $field->type,
				'id'    => $field->id,
				'send'  => false,
				'value' => null,
			);

			// We don't want to be packaging file uploads and nested forms
			if ( 'file_upload' === $field->type || 'form' === $field->type ) {
				continue;
			}

			$field_value = rgar( $entry, $field->id ); // safer than $entry[$field->id]

			// Case 1: Serialized data (like multi-selects, checkboxes)
			if ( is_string( $field_value ) && is_serialized( $field_value ) ) {
				$data = maybe_unserialize( $field_value );
				if ( is_array( $data ) ) {
					$list = array();
					foreach ( $data as $index => $item ) {
						if ( ! is_null( $item ) && '' !== trim( $item ) ) {
							$list[] = array(
								'label' => $index,
								'value' => $item,
							);
						}
					}
					if ( ! empty( $list ) ) {
						$package_item['value'] = $list;
					}
				}
			} elseif ( isset( $field->inputs ) && is_array( $field->inputs ) ) {
				// Case 2: Multi-input fields (Name, Address)
				$item_inputs = array();
				foreach ( $field->inputs as $input ) {
					$input_value = rgar( $entry, $input['id'] );
					if ( ! is_null( $input_value ) && '' !== trim( $input_value ) ) {
						$item_inputs[] = array(
							'label' => $input['label'],
							'id'    => $input['id'],
							'value' => $input_value,
						);
					}
				}
				if ( ! empty( $item_inputs ) ) {
					$package_item['value'] = $item_inputs;
				}
			} elseif ( ! is_null( $field_value ) && '' !== trim( $field_value ) ) {
				// Case 3: Simple fields
				$package_item['value'] = $field_value;
			}

			// Only add packaged item if it has value
			if ( ! is_null( $package_item['value'] ) ) {
				// if ( ( isset( $field->isHidden ) && $field->isHidden ) || 'hidden' === $field->type ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				// $package_item['send'] = false;
				// $package['admin'][]   = $package_item;
				// } else {
				$package_item['send'] = true;
				$package['data'][]    = $package_item;
				// }
			}
		}

		return $package;
	}

	/**
	 * Formats packaged field data into HTML for use as a note.
	 *
	 * Includes a timestamp and all submitted field values.
	 *
	 * @param array $data The packaged field data.
	 * @return string HTML-formatted note text.
	 */
	private function format_packaged_data( $data ) {

		$output = '';

		$output .= '<strong>DATE: </strong>' . gmdate( 'Y-m-d' ) . '<br>';
		$output .= '<strong>TIMESTAMP: </strong>' . gmdate( 'h:i:sa' ) . '<br>';

		foreach ( $data as $data_item ) {
			if ( ! isset( $data_item['value'] ) ) {
				continue;
			}

			$output .= '<strong>' . $data_item['label'] . ': </strong>';

			if ( is_array( $data_item['value'] ) ) {
				foreach ( $data_item['value'] as $nested_item ) {
					$output .= '<br>';
					$output .= ' -- ';
					$output .= '<strong>' . $nested_item['label'] . ': </strong>';
					$output .= $nested_item['value'];
				}
			} else {
				$output .= $data_item['value'];
			}

			$output .= '<br>';
		}

		return $output;
	}

	/**
	 * Normalizes legacy Pipedrive field keys to new prefixed versions.
	 *
	 * @param string $key Original mapping key.
	 * @return string Normalized key (e.g., company → org_name).
	 */
	private function normalize_legacy_map( $key ) {
		switch ( $key ) {
			case 'company':
				return 'org_name';
			case 'name':
				return 'person_name';
			case 'email':
				return 'person_email';
			case 'phone':
				return 'person_phone';
			default:
				return $key;
		}
	}
}
