<?php
/**
 * GravityFormsDefaults
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use GFAPI;
use GFCommon;
use GFFormsModel;

/**
 * Handles Gravity Forms automation settings.
 */
class GravityFormsDefaults extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * @return bool
	 */
	public function can_register() {
		return is_plugin_active( 'gravityforms/gravityforms.php' );
	}

	/**
	 * Register hooks for automating Gravity Forms settings.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'gform_after_save_form', [ $this, 'add_gform_defaults' ], 10, 2 );
		add_action( 'gform_post_update_form_meta', [ $this, 'enforce_notification_email_domain' ], 10, 3 );
		add_filter( 'gform_confirmation_anchor', '\__return_true' );
		add_filter( 'gform_submit_button', [ $this, 'add_classes_to_buttons' ], 10, 2 );
		add_filter( 'gform_form_switcher_forms', [ $this, 'show_all_forms_in_switcher' ], 10, 1 );
	}

	/**
	 * Retrieves all Gravity Forms and prepares them for display in a form switcher.
	 *
	 * Fetches every form using the Gravity Forms API, converts each form array into
	 * a standardized object structure, and sorts the results alphabetically by title.
	 *
	 * @param array $forms Existing forms passed into the switcher filter. Not used,
	 *                     as this method replaces the list with all available forms.
	 *
	 * @return array Array of form objects containing id, title, description, and is_active,
	 *               sorted alphabetically by title.
	 */
	public function show_all_forms_in_switcher( $forms ) {
		$all_forms = \GFAPI::get_forms();

		$all_forms_objects = array_map(
			function ( $form ) {
				return (object) [
					'id'          => $form['id'],
					'title'       => $form['title'],
					'description' => $form['description'] ?? '',
					'is_active'   => $form['is_active'] ?? true,
				];
			},
			$all_forms
		);

		usort(
			$all_forms_objects,
			function ( $a, $b ) {
				return strcasecmp( $a->title, $b->title );
			}
		);

		return $all_forms_objects;
	}

	/**
	 * Adds default settings to new Gravity Forms.
	 *
	 * @param array $form  The Gravity Form object.
	 * @param bool  $is_new Indicates if the form is new.
	 *
	 * @return void
	 */
	public function add_gform_defaults( $form, $is_new ) {
		$form_needs_update = false;

		$site_domain = get_site_domain();
		foreach ( $form['notifications'] as &$notification ) {
			if ( ! is_same_domain( $notification['from'], $site_domain ) ) {
				$sender_email = determine_sender_email();
				$notification['from'] = $sender_email;
			}
			$form_needs_update = true;
		}

		if ( $form_needs_update || $is_new ) {
			$form['descriptionPlacement'] = 'below';
			$form['labelPlacement'] = 'top_label';
			$form['subLabelPlacement'] = 'above';
			$form['validationSummary'] = true;
			$form['requiredIndicator'] = 'text';
			GFAPI::update_form( $form );
		}
	}

	/**
	 * Enforces same-domain email for all notifications after form meta is updated.
	 *
	 * @param array  $form_meta The updated form meta.
	 * @param int    $form_id   The ID of the form being updated.
	 * @param string $meta_name The specific meta key being updated (e.g., 'notifications').
	 *
	 * @return void
	 */
	public function enforce_notification_email_domain( $form_meta, $form_id, $meta_name ) {
		if ( 'notifications' !== $meta_name ) {
			return;
		}

		$form = GFAPI::get_form( $form_id );

		if ( ! $form || is_wp_error( $form ) ) {
			return;
		}

		$site_domain = get_site_domain();

		foreach ( $form['notifications'] as &$notification ) {
			if ( ! is_same_domain( $notification['from'], $site_domain ) ) {
				$sender_email = determine_sender_email();
				$notification['from'] = $sender_email;
			}
			$form_needs_update = true;
		}

		if ( $form_needs_update ) {
			remove_action( 'gform_post_update_form_meta', [ $this, 'enforce_notification_email_domain' ], 10 );
			GFAPI::update_form( $form );
			add_action( 'gform_post_update_form_meta', [ $this, 'enforce_notification_email_domain' ], 10, 3 );
		}
	}



	/**
	 * Adds our CSS classes to the gform_button so styling effects it without additional CSS.
	 *
	 * @link https://docs.gravityforms.com/gform_submit_button/
	 * @param string $button The string containing the <input> tag to be filtered.
	 * @param object $form Form Object The form currently being processes.
	 */
	public function add_classes_to_buttons( $button, $form ) {
		$dom = new \DOMDocument();
		$dom->loadHTML( '<?xml encoding="utf-8" ?>' . $button );
		$input = $dom->getElementsByTagName( 'input' )->item( 0 );
		$classes = $input->getAttribute( 'class' );
		$classes .= ' btn btn-primary';
		$input->setAttribute( 'class', $classes );
		return $dom->saveHtml( $input );
	}
}
