<?php
/**
 * EnterpriseWP
 *
 * Handles the removal of unnecessary WordPress bloat and optimizes admin and frontend performance.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class EnterpriseWP
 *
 * Cleans up unnecessary WordPress features to enhance performance and improve the user experience.
 */
class EnterpriseWP extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * @return bool Always returns true.
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Registers hooks for cleaning up WordPress bloat.
	 *
	 * This method removes unnecessary scripts, styles, dashboard widgets, and other
	 * default WordPress features to optimize performance.
	 *
	 * @return void
	 */
	public function register() {
		$this->remove_emojis();
		$this->clean_head_meta();
		$this->optimize_dashboard();
		$this->disable_unwanted_features();
		$this->adjust_roles_and_permissions();
		$this->customize_admin_interface();
		$this->clean_up_scripts_and_styles();
		add_filter( 'upload_mimes', [ $this, 'add_mime_types' ] );
	}

	/**
	 * Removes emojis from WordPress.
	 *
	 * @return void
	 */
	private function remove_emojis() {
		remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
		remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
		remove_action( 'wp_print_styles', 'print_emoji_styles' );
		remove_action( 'admin_print_styles', 'print_emoji_styles' );
		remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
		remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );
		remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
		add_filter( 'emoji_svg_url', '\__return_false' );
		add_filter( 'tiny_mce_plugins', [ $this, 'disable_emojis_tinymce' ] );
		add_filter( 'wp_resource_hints', [ $this, 'disable_emoji_dns_prefetch' ], 10, 2 );
	}

	/**
	 * Removes the TinyMCE emoji plugin.
	 *
	 * @param array $plugins The array of default TinyMCE plugins.
	 *
	 * @return array Filtered array of TinyMCE plugins.
	 */
	public function disable_emojis_tinymce( $plugins ) {
		if ( is_array( $plugins ) && in_array( 'wpemoji', $plugins, true ) ) {
			return array_diff( $plugins, array( 'wpemoji' ) );
		}

		return $plugins;
	}

	/**
	 * Removes the emoji CDN hostname from DNS prefetching hints.
	 *
	 * @param array  $urls URLs to print for resource hints.
	 * @param string $relation_type The relation type the URLs are printed for.
	 *
	 * @return array Filtered array of resource hint URLs.
	 */
	public function disable_emoji_dns_prefetch( $urls, $relation_type ) {
		if ( 'dns-prefetch' === $relation_type ) {
			/** This filter is documented in wp-includes/formatting.php */
			$emoji_svg_url = apply_filters( 'emoji_svg_url', 'https://s.w.org/images/core/emoji/2/svg/' );

			$urls = array_values( array_diff( $urls, array( $emoji_svg_url ) ) );
		}

		return $urls;
	}

	/**
	 * Cleans up unnecessary meta tags in the head section of WordPress.
	 *
	 * @return void
	 */
	private function clean_head_meta() {
		remove_action( 'wp_head', 'wp_generator' );
		remove_action( 'wp_head', 'wlwmanifest_link' );
		remove_action( 'wp_head', 'rsd_link' );
		remove_action( 'wp_head', 'rest_output_link_wp_head' );
		remove_action( 'wp_head', 'wp_oembed_add_discovery_links' );
		remove_action( 'wp_head', 'feed_links_extra', 3 );
		remove_action( 'wp_head', 'feed_links', 2 );
		remove_action( 'wp_head', 'wp_shortlink_wp_head' );
		remove_action( 'template_redirect', 'rest_output_link_header', 11 );
	}

	/**
	 * Optimizes the WordPress dashboard by removing unnecessary widgets.
	 *
	 * @return void
	 */
	private function optimize_dashboard() {
		remove_action( 'welcome_panel', 'wp_welcome_panel' );
		add_action( 'admin_menu', [ $this, 'disable_dashboard_widgets' ], 11 );
		add_action( 'wp_dashboard_setup', [ $this, 'remove_other_dashboard_widgets' ] );
	}

	/**
	 * Disables built-in WordPress dashboard widgets.
	 *
	 * @return void
	 */
	public function disable_dashboard_widgets() {
		// Remove "At a Glance"
		remove_meta_box( 'dashboard_right_now', 'dashboard', 'normal' );
		// Remove "Activity" which includes "Recent Comments"
		remove_meta_box( 'dashboard_activity', 'dashboard', 'normal' );
		// Remove Quick Draft
		remove_meta_box( 'dashboard_quick_press', 'dashboard', 'side' );
		// Remove WordPress Events and News
		remove_meta_box( 'dashboard_primary', 'dashboard', 'core' );
		// Hide "Yoast SEO → Integrations".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_integrations' );
		// Hide "Yoast SEO → Academy".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_page_academy' );
		// Hide "Yoast SEO → Premium".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_licenses' );
		// Hide "Yoast SEO → Workouts".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_workouts' );
		// Hide "Yoast SEO → Redirects".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_redirects' );
		// Hide "Yoast SEO → Support".
		remove_submenu_page( 'wpseo_dashboard', 'wpseo_page_support' );
		// Hide Gravity Forms Help
		remove_submenu_page( 'gf_edit_forms', 'gf_help' );
	}

	/**
	 * Removes additional third-party dashboard widgets.
	 *
	 * @return void
	 */
	public function remove_other_dashboard_widgets() {
		// Removes Site Health Dashboard Widget
		remove_meta_box( 'dashboard_site_health', 'dashboard', 'normal' );
		// Removes Tribe Widget
		remove_meta_box( 'tribe_dashboard_widget', 'dashboard', 'side' );
		// Removes Elementor Widget
		remove_meta_box( 'e-dashboard-overview', 'dashboard', 'normal' );
		// Remove the "Yoast SEO / Wincher: Top Keyphrases" widget.
		remove_meta_box( 'wpseo-wincher-dashboard-overview', 'dashboard', 'normal' );
	}

	/**
	 * Disables unwanted WordPress features.
	 *
	 * @return void
	 */
	private function disable_unwanted_features() {
		add_filter( 'admin_email_check_interval', '\__return_false' );
		add_filter( 'xmlrpc_enabled', '\__return_false' );
		remove_action( 'admin_notices', 'update_nag', 3 );
		remove_action( 'load-update-core.php', 'wp_update_plugins' );
		add_filter( 'pre_site_transient_update_plugins', '\__return_null' );
		add_filter( 'pre_site_transient_update_core', [ $this, 'remove_core_updates' ] );
		add_filter( 'pre_site_transient_update_plugins', [ $this, 'remove_core_updates' ] );
		add_filter( 'pre_site_transient_update_themes', [ $this, 'remove_core_updates' ] );
		add_filter( 'wpseo_enable_notification_post_trash', '\__return_false' );
		add_filter( 'wpseo_enable_notification_post_slug_change', '\__return_false' );
		add_filter( 'wpseo_enable_notification_term_delete', '\__return_false' );
		add_filter( 'wpseo_enable_notification_term_slug_change', '\__return_false' );
		add_filter( 'rocket_set_wp_cache_constant', '\__return_false' );
		remove_action( 'wp_footer', 'the_block_template_skip_link' );
	}

	/**
	 * Disables automatic WordPress core updates.
	 *
	 * @return object Modified update object to prevent updates.
	 */
	public function remove_core_updates() {
		global $wp_version;
		return (object) array(
			'last_checked' => time(),
			'version_checked' => $wp_version,
		);
	}

	/**
	 * Adjusts roles and permissions for editors and custom roles.
	 *
	 * @return void
	 */
	private function adjust_roles_and_permissions() {
		add_filter( 'redirection_role', [ $this, 'change_role_to_edit_redirection' ] );
		add_action( 'admin_init', [ $this, 'add_capabilities_to_editors' ] );
		add_action( 'init', [ $this, 'create_client_roles' ] );
		add_action( 'admin_menu', [ $this, 'remove_built_in_roles' ] );
		// TODO: Can't quite get this to work well.
		// $this->rename_role( 'wpamelia-manager', 'wpamelia-manager', 'Bookings Manager' );
		// $this->rename_role( 'wpamelia-provider', 'wpamelia-provider', 'Bookings Employee' );
		// $this->rename_role( 'wpamelia-customer', 'wpamelia-customer', 'Bookings Customer' );
	}

	/**
	 * Modifies the required capability for managing redirections.
	 *
	 * @param string $role The role required to manage redirections.
	 *
	 * @return string The modified capability.
	 */
	public function change_role_to_edit_redirection( $role ) {
		return 'edit_posts';
	}

	/**
	 * Adds additional capabilities to editors.
	 *
	 * @return void
	 */
	public function add_capabilities_to_editors() {
		$role = get_role( 'editor' );

		if ( is_null( $role ) ) {
			return;
		}

		$caps = [
			// The Events Calendar Pro
			'edit_tribe_event',
			'read_tribe_event',
			'delete_tribe_event',
			'delete_tribe_events',
			'edit_tribe_events',
			'edit_others_tribe_events',
			'delete_others_tribe_events',
			'publish_tribe_events',
			'edit_published_tribe_events',
			'delete_published_tribe_events',
			'delete_private_tribe_events',
			'edit_private_tribe_events',
			'read_private_tribe_events',
			'edit_tribe_venue',
			'read_tribe_venue',
			'delete_tribe_venue',
			'delete_tribe_venues',
			'edit_tribe_venues',
			'edit_others_tribe_venues',
			'delete_others_tribe_venues',
			'publish_tribe_venues',
			'edit_published_tribe_venues',
			'delete_published_tribe_venues',
			'delete_private_tribe_venues',
			'edit_private_tribe_venues',
			'read_private_tribe_venues',
			'edit_tribe_organizer',
			'read_tribe_organizer',
			'delete_tribe_organizer',
			'delete_tribe_organizers',
			'edit_tribe_organizers',
			'edit_others_tribe_organizers',
			'delete_others_tribe_organizers',
			'publish_tribe_organizers',
			'edit_published_tribe_organizers',
			'delete_published_tribe_organizers',
			'delete_private_tribe_organizers',
			'edit_private_tribe_organizers',
			'read_private_tribe_organizers',
			// Misc
			'moderate_comments' => true,
			'manage_categories' => true,
			'unfiltered_html' => true,
			'upload_files' => true,
			'edit_theme_options' => true,
			// WP Rocket
			'rocket_purge_cache' => true,
			'rocket_purge_opcache' => true,
			'rocket_preload_cache' => true,
			'rocket_regenerate_critical_css' => true,
			// Gravity Forms
			'gravityforms_create_form' => true,
			'gravityforms_edit_forms' => true,
			'gravityforms_preview_forms' => true,
			'gravityforms_view_entries' => true,
			'gravityforms_edit_entries' => true,
			'gravityforms_delete_entries' => true,
			'gravityforms_view_entry_notes' => true,
			'gravityforms_edit_entry_notes' => true,
			'gravityforms_export_entries' => true,
			// Groundhogg
			'schedule_broadcasts' => true,
			'cancel_broadcasts' => true,
			'view_broadcasts' => true,
			'add_contacts' => true,
			'edit_contacts' => true,
			'edit_others_contacts' => true,
			'view_contacts' => true,
			'view_others_contacts' => true,
			'delete_contacts' => true,
			'delete_others_contacts' => true,
			'import_contacts' => false,
			'export_contacts' => true,
			'add_emails' => true,
			'view_emails' => true,
			'delete_emails' => false,
			'edit_emails' => true,
			'send_emails' => true,
			'export_emails' => true,
			'import_emails' => true,
			'execute_events' => true,
			'cancel_events' => true,
			'schedule_events' => true,
			'view_events' => true,
			'add_funnels' => false,
			'view_funnels' => false,
			'delete_funnels' => false,
			'edit_funnels' => false,
			'export_funnels' => false,
			'import_funnels' => false,
			'add_tags' => true,
			'delete_tags' => false,
			'edit_tags' => true,
			'manage_tags' => true,
			'view_tags' => true,
			'view_reports' => true,
			'export_reports' => true,
			'perform_bulk_actions' => true,
			'manage_gh_licenses' => false,
			'edit_custom_properties' => true,
			'manage_campaigns' => true,
			'download_files' => true,
			'delete_files' => false,
			'view_previous_imports' => false,
			'view_previous_exports' => false,
			'download_contact_files' => true,
			'view_logs' => true,
			'delete_logs' => false,
			'add_notes' => true,
			'delete_notes' => true,
			'edit_notes' => true,
			'view_notes' => true,
			'delete_others_notes' => true,
			'edit_others_notes' => true,
			'view_others_notes' => true,
			'add_activity' => true,
			'view_activity' => true,
			'edit_activity' => true,
			'delete_activity' => true,
			'add_deals' => false,
			'edit_deals' => false,
			'edit_others_deals' => false,
			'view_deals' => false,
			'view_others_deals' => false,
			'delete_deals' => false,
			'delete_others_deals' => false,
			'won_deals' => false,
			'lost_deals' => false,
			// Amelia
			'amelia_delete_events'             => true,
			'amelia_read_menu'                 => true,
			'amelia_read_dashboard'            => true,
			'amelia_read_whats_new'            => false,
			'amelia_read_lite_vs_premium'      => false,
			'amelia_read_calendar'             => true,
			'amelia_read_appointments'         => true,
			'amelia_read_events'               => true,
			'amelia_read_employees'            => true,
			'amelia_read_services'             => true,
			'amelia_read_resources'            => true,
			'amelia_read_packages'             => true,
			'amelia_read_locations'            => true,
			'amelia_read_coupons'              => true,
			'amelia_read_customers'            => true,
			'amelia_read_finance'              => true,
			'amelia_read_notifications'        => true,
			'amelia_read_others_dashboard'     => true,
			'amelia_read_others_calendar'      => true,
			'amelia_read_others_appointments'  => true,
			'amelia_read_others_services'      => true,
			'amelia_read_others_employees'     => true,
			'amelia_read_others_customers'     => true,
			'amelia_write_dashboard'           => true,
			'amelia_write_calendar'            => true,
			'amelia_write_appointments'        => true,
			'amelia_write_events'              => true,
			'amelia_write_employees'           => true,
			'amelia_write_services'            => true,
			'amelia_write_resources'           => true,
			'amelia_write_packages'            => true,
			'amelia_write_locations'           => true,
			'amelia_write_coupons'             => true,
			'amelia_write_customers'           => true,
			'amelia_write_finance'             => true,
			'amelia_write_notifications'       => true,
			'amelia_write_others_calendar'     => true,
			'amelia_write_others_appointments' => true,
			'amelia_write_others_services'     => true,
			'amelia_write_others_employees'    => true,
			'amelia_write_others_events'       => true,
			'amelia_write_others_finance'      => true,
			'amelia_write_others_dashboard'    => true,
			'amelia_write_status_appointments' => true,
			'amelia_write_status_events'       => true,
			'amelia_write_time_appointments'   => true,
		];

		foreach ( $caps as $cap => $value ) {
			if ( $value ) {
				$role->add_cap( $cap );
			}
		}
	}

	/**
	 * Creates a custom client role with predefined capabilities.
	 *
	 * @return void
	 */
	public function create_client_roles() {
		$role = add_role(
			'client',
			__( 'Client', 'leadgen' ),
			[
				// General
				'read' => true,
				// Pages
				'edit_pages' => true,
				'edit_others_pages' => true,
				'edit_published_pages' => true,
				'read_private_pages' => true,
				'edit_private_pages' => true,
				'publish_pages' => true,
				// Posts
				'edit_posts' => true,
				'edit_others_posts' => true,
				'edit_published_posts' => true,
				'read_private_posts' => true,
				'edit_private_posts' => true,
				'publish_posts' => true,
				// Misc
				'moderate_comments' => true,
				'manage_categories' => true,
				'unfiltered_html' => true,
				'upload_files' => true,
				'edit_theme_options' => true,
				// WP Rocket
				'rocket_purge_cache' => true,
				'rocket_purge_opcache' => true,
				'rocket_preload_cache' => true,
				'rocket_regenerate_critical_css' => true,
				// Gravity Forms
				'gravityforms_create_form' => true,
				'gravityforms_edit_forms' => true,
				'gravityforms_preview_forms' => true,
				'gravityforms_view_entries' => true,
				'gravityforms_edit_entries' => true,
				'gravityforms_delete_entries' => true,
				'gravityforms_view_entry_notes' => true,
				'gravityforms_edit_entry_notes' => true,
				'gravityforms_export_entries' => true,
				// Groundhogg
				'schedule_broadcasts' => true,
				'cancel_broadcasts' => true,
				'view_broadcasts' => true,
				'add_contacts' => true,
				'edit_contacts' => true,
				'edit_others_contacts' => true,
				'view_contacts' => true,
				'view_others_contacts' => true,
				'delete_contacts' => true,
				'delete_others_contacts' => true,
				'import_contacts' => false,
				'export_contacts' => true,
				'add_emails' => true,
				'view_emails' => true,
				'delete_emails' => false,
				'edit_emails' => true,
				'send_emails' => true,
				'export_emails' => true,
				'import_emails' => true,
				'execute_events' => true,
				'cancel_events' => true,
				'schedule_events' => true,
				'view_events' => true,
				'add_funnels' => false,
				'view_funnels' => false,
				'delete_funnels' => false,
				'edit_funnels' => false,
				'export_funnels' => false,
				'import_funnels' => false,
				'add_tags' => true,
				'delete_tags' => false,
				'edit_tags' => true,
				'manage_tags' => true,
				'view_tags' => true,
				'view_reports' => true,
				'export_reports' => true,
				'perform_bulk_actions' => true,
				'manage_gh_licenses' => false,
				'edit_custom_properties' => true,
				'manage_campaigns' => true,
				'download_files' => true,
				'delete_files' => false,
				'view_previous_imports' => false,
				'view_previous_exports' => false,
				'download_contact_files' => true,
				'view_logs' => true,
				'delete_logs' => false,
				'add_notes' => true,
				'delete_notes' => true,
				'edit_notes' => true,
				'view_notes' => true,
				'delete_others_notes' => true,
				'edit_others_notes' => true,
				'view_others_notes' => true,
				'add_activity' => true,
				'view_activity' => true,
				'edit_activity' => true,
				'delete_activity' => true,
				'add_deals' => false,
				'edit_deals' => false,
				'edit_others_deals' => false,
				'view_deals' => false,
				'view_others_deals' => false,
				'delete_deals' => false,
				'delete_others_deals' => false,
				'won_deals' => false,
				'lost_deals' => false,
				// Amelia
				'amelia_delete_events'             => true,
				'amelia_read_menu'                 => true,
				'amelia_read_dashboard'            => true,
				'amelia_read_whats_new'            => false,
				'amelia_read_lite_vs_premium'      => false,
				'amelia_read_calendar'             => true,
				'amelia_read_appointments'         => true,
				'amelia_read_events'               => true,
				'amelia_read_employees'            => true,
				'amelia_read_services'             => true,
				'amelia_read_resources'            => true,
				'amelia_read_packages'             => true,
				'amelia_read_locations'            => true,
				'amelia_read_coupons'              => true,
				'amelia_read_customers'            => true,
				'amelia_read_finance'              => true,
				'amelia_read_notifications'        => true,
				'amelia_read_others_dashboard'     => true,
				'amelia_read_others_calendar'      => true,
				'amelia_read_others_appointments'  => true,
				'amelia_read_others_services'      => true,
				'amelia_read_others_employees'     => true,
				'amelia_read_others_customers'     => true,
				'amelia_write_dashboard'           => true,
				'amelia_write_calendar'            => true,
				'amelia_write_appointments'        => true,
				'amelia_write_events'              => true,
				'amelia_write_employees'           => true,
				'amelia_write_services'            => true,
				'amelia_write_resources'           => true,
				'amelia_write_packages'            => true,
				'amelia_write_locations'           => true,
				'amelia_write_coupons'             => true,
				'amelia_write_customers'           => true,
				'amelia_write_finance'             => true,
				'amelia_write_notifications'       => true,
				'amelia_write_others_calendar'     => true,
				'amelia_write_others_appointments' => true,
				'amelia_write_others_services'     => true,
				'amelia_write_others_employees'    => true,
				'amelia_write_others_events'       => true,
				'amelia_write_others_finance'      => true,
				'amelia_write_others_dashboard'    => true,
				'amelia_write_status_appointments' => true,
				'amelia_write_status_events'       => true,
				'amelia_write_time_appointments'   => true,
				// The Events Calendar Pro
				'edit_tribe_event',
				'read_tribe_event',
				'delete_tribe_event',
				'delete_tribe_events',
				'edit_tribe_events',
				'edit_others_tribe_events',
				'delete_others_tribe_events',
				'publish_tribe_events',
				'edit_published_tribe_events',
				'delete_published_tribe_events',
				'delete_private_tribe_events',
				'edit_private_tribe_events',
				'read_private_tribe_events',
				'edit_tribe_venue',
				'read_tribe_venue',
				'delete_tribe_venue',
				'delete_tribe_venues',
				'edit_tribe_venues',
				'edit_others_tribe_venues',
				'delete_others_tribe_venues',
				'publish_tribe_venues',
				'edit_published_tribe_venues',
				'delete_published_tribe_venues',
				'delete_private_tribe_venues',
				'edit_private_tribe_venues',
				'read_private_tribe_venues',
				'edit_tribe_organizer',
				'read_tribe_organizer',
				'delete_tribe_organizer',
				'delete_tribe_organizers',
				'edit_tribe_organizers',
				'edit_others_tribe_organizers',
				'delete_others_tribe_organizers',
				'publish_tribe_organizers',
				'edit_published_tribe_organizers',
				'delete_published_tribe_organizers',
				'delete_private_tribe_organizers',
				'edit_private_tribe_organizers',
				'read_private_tribe_organizers',
			]
		);
	}

	/**
	 * Removes unnecessary built-in WordPress roles.
	 *
	 * @return void
	 */
	public function remove_built_in_roles() {
		global $wp_roles;

		$roles_to_remove = array( 'contributor', 'author', 'sales_rep', 'sales_manager', 'marketer', 'wpseo_editor', 'wpseo_manager' );

		foreach ( $roles_to_remove as $role ) {
			if ( isset( $wp_roles->roles[ $role ] ) ) {
				$wp_roles->remove_role( $role );
			}
		}
	}

	/**
	 * Renames an existing WordPress role.
	 *
	 * @param string $old_role The old role name.
	 * @param string $new_role The new role name.
	 * @param string $new_display_name The new display name for the role.
	 *
	 * @return void
	 */
	public function rename_role( $old_role, $new_role, $new_display_name ) {
		global $wp_roles;

		// Get the role object.
		if ( isset( $wp_roles->roles[ $old_role ] ) ) {
			$old_role_obj = get_role( $old_role );

			// Get capabilities of the old role.
			$capabilities = $old_role_obj->capabilities;

			// Add new role with the same capabilities.
			add_role( $new_role, $new_display_name, $capabilities );

			// Reassign users from old role to new role.
			$users = get_users( [ 'role' => $old_role ] );
			foreach ( $users as $user ) {
				$user->remove_role( $old_role );
				$user->add_role( $new_role );
			}

			// Remove the old role.
			remove_role( $old_role );
		}
	}

	/**
	 * Customizes the WordPress admin interface.
	 *
	 * @return void
	 */
	private function customize_admin_interface() {
		add_action( 'admin_init', [ $this, 'disable_yoast_notifications' ] );
		add_action( 'admin_head', [ $this, 'remove_help_tabs' ] );
		add_action( 'admin_bar_menu', [ $this, 'remove_toolbar_nodes' ], PHP_INT_MAX );
		add_action( 'admin_menu', [ $this, 'remove_admin_tools_menus' ] );
		add_filter( 'admin_footer_text', [ $this, 'change_admin_footer_text' ] );
		add_filter( 'update_footer', '\__return_empty_string', 11 );
		add_filter( 'user_contactmethods', [ $this, 'update_contact_methods' ], 99 );
		add_action( 'admin_head-profile.php', [ $this, 'user_profile_subject_start' ] );
		add_action( 'admin_footer-profile.php', [ $this, 'user_profile_subject_end' ] );
	}

	/**
	 * Disables notifications from the Yoast SEO plugin.
	 *
	 * @return void
	 */
	public function disable_yoast_notifications() {
		if ( is_plugin_active( 'wordpress-seo/wp-seo.php' ) ) {
			remove_action( 'admin_notices', array( \Yoast_Notification_Center::get(), 'display_notifications' ) );
			remove_action( 'all_admin_notices', array( \Yoast_Notification_Center::get(), 'display_notifications' ) );
			update_user_meta( get_current_user_id(), 'wpseo_ignore_tour', true );
			add_filter( 'wpseo_premium_ads_enabled', '\__return_false' );
			add_filter( 'Yoast\WP\SEO\post_redirect_slug_change', '\__return_true' );
			add_filter( 'Yoast\WP\SEO\term_redirect_slug_change', '\__return_true' );
			add_filter( 'Yoast\WP\SEO\enable_notification_post_trash', '\__return_false' );
			add_filter( 'Yoast\WP\SEO\enable_notification_post_slug_change', '\__return_false' );
			add_filter( 'Yoast\WP\SEO\enable_notification_term_delete', '\__return_false' );
			add_filter( 'Yoast\WP\SEO\enable_notification_term_slug_change', '\__return_false' );
		}
	}

	/**
	 * Removes the help tabs from the WordPress admin.
	 *
	 * @return void
	 */
	public function remove_help_tabs() {
		$screen = get_current_screen();
		$screen->remove_help_tabs();
	}

	/**
	 * Removes items from the WordPress admin toolbar.
	 *
	 * @param \WP_Admin_Bar $wp_admin_bar The WordPress Admin Bar instance.
	 *
	 * @return void
	 */
	public function remove_toolbar_nodes( $wp_admin_bar ) {
		$wp_admin_bar->remove_node( 'wp-logo' );
		$wp_admin_bar->remove_node( 'comments' );
		$wp_admin_bar->remove_node( 'customize' );
		$wp_admin_bar->remove_node( 'customize-background' );
		$wp_admin_bar->remove_node( 'customize-header' );
	}

	/**
	 * Removes unnecessary admin menu items.
	 *
	 * @return void
	 */
	public function remove_admin_tools_menus() {
		remove_submenu_page( 'tools.php', 'tools.php' );
		remove_submenu_page( 'tools.php', 'site-health.php' );
		remove_submenu_page( 'tools.php', 'import.php' );
		remove_submenu_page( 'tools.php', 'export.php' );
	}

	/**
	 * Modifies the admin footer text.
	 *
	 * @param string $default The default footer text.
	 *
	 * @return string The modified footer text.
	 */
	public function change_admin_footer_text( $default ) {
		return 'Crafted with ♥ by Strategy';
	}

	/**
	 * Updates the contact methods available in user profiles.
	 *
	 * @param array $contactmethods The existing contact methods.
	 *
	 * @return array The updated contact methods array.
	 */
	public function update_contact_methods( $contactmethods ) {
		unset( $contactmethods['facebook'] );
		unset( $contactmethods['instagram'] );
		unset( $contactmethods['linkedin'] );
		unset( $contactmethods['myspace'] );
		unset( $contactmethods['pinterest'] );
		unset( $contactmethods['soundcloud'] );
		unset( $contactmethods['tumblr'] );
		unset( $contactmethods['twitter'] );
		unset( $contactmethods['youtube'] );
		unset( $contactmethods['wikipedia'] );

		return $contactmethods;
	}

	/**
	 * Starts output buffering to remove bio section from the profile page.
	 *
	 * @return void
	 */
	public function user_profile_subject_start() {
		ob_start( [ $this,'remove_items_from_bio_box' ] );
	}

	/**
	 * Removes bio-related elements from the WordPress user profile page.
	 *
	 * @param string $buffer The page content buffer.
	 *
	 * @return string The modified page buffer.
	 */
	public function remove_items_from_bio_box( $buffer ) {
		$buffer = str_replace( '<h2>About Yourself</h2>', '', $buffer );
		$buffer = preg_replace( '/<tr class=\"user-profile-picture\"[\s\S]*?<\/tr>/', '', $buffer, 1 );
		return $buffer;
	}

	/**
	 * Ends output buffering for the user profile section.
	 *
	 * @return void
	 */
	public function user_profile_subject_end() {
		ob_end_flush();
	}

	/**
	 * Cleans up unnecessary scripts and styles on the frontend.
	 *
	 * @return void
	 */
	private function clean_up_scripts_and_styles() {
		add_action( 'wp_enqueue_scripts', [ $this, 'dequeue_dashicons' ] );
		// add_action( 'wp_enqueue_scripts', [ $this, 'dequeue_wp_polyfills' ] );
		add_action( 'wp_footer', [ $this, 'dequeue_wp_embed' ] );
		add_action( 'after_setup_theme', [ $this, 'remove_svg_filters' ], PHP_INT_MAX );
		add_action( 'init', [ $this, 'remove_global_styles_from_blocks' ], PHP_INT_MAX );
		// Removes inline css from being enqueued
		remove_filter( 'render_block', 'gutenberg_render_layout_support_flag', PHP_INT_MAX, 2 );
		remove_filter( 'render_block', 'wp_render_elements_support', PHP_INT_MAX, 2 );
		remove_filter( 'render_block', 'gutenberg_render_elements_support', PHP_INT_MAX, 2 );
		add_action( 'wp_default_scripts', [ $this, 'remove_jquery_migrate' ] );
	}

	/**
	 * Removes Dashicons from the frontend for non-logged-in users.
	 *
	 * @return void
	 */
	public function dequeue_dashicons() {
		if ( ! is_admin() && ! is_user_logged_in() ) {
			wp_deregister_style( 'dashicons' );
		}
	}

	/**
	 * Removes WordPress polyfills from the frontend.
	 *
	 * @return void
	 */
	public function dequeue_wp_polyfills() {
		wp_deregister_script( 'wp-polyfill' );
		wp_deregister_script( 'regenerator-runtime' );
	}

	/**
	 * Removes the WP Embed script from WordPress.
	 *
	 * @return void
	 */
	public function dequeue_wp_embed() {
		wp_dequeue_script( 'wp-embed' );
	}

	/**
	 * Removes SVG filters from WordPress.
	 *
	 * @return void
	 */
	public function remove_svg_filters() {
		remove_action( 'wp_body_open', 'wp_global_styles_render_svg_filters' );
		remove_action( 'in_admin_header', 'wp_global_styles_render_svg_filters' );

		// remove render_block filters
		remove_filter( 'render_block', 'wp_render_duotone_support' );
		remove_filter( 'render_block', 'wp_restore_group_inner_container' );
		remove_filter( 'render_block', 'wp_render_layout_support_flag' );
	}

	/**
	 * Removes global styles from Gutenberg blocks.
	 *
	 * @return void
	 */
	public function remove_global_styles_from_blocks() {
		wp_dequeue_style( 'global-styles' );
		wp_dequeue_style( 'wp-block-library' );
		wp_dequeue_style( 'wp-block-library-theme' );
		wp_dequeue_style( 'wc-block-style' );
		wp_dequeue_style( 'global-styles' );
		wp_dequeue_style( 'core-block-supports' );
		remove_action( 'wp_enqueue_scripts', 'wp_enqueue_global_styles' );
		remove_action( 'wp_footer', 'wp_enqueue_global_styles', 1 );
	}

	/**
	 * Removes jQuery Migrate from WordPress frontend scripts.
	 *
	 * @param \WP_Scripts $scripts The WP_Scripts object.
	 *
	 * @return void
	 */
	public function remove_jquery_migrate( $scripts ) {
		if ( ! is_admin() && isset( $scripts->registered['jquery'] ) ) {
			$script = $scripts->registered['jquery'];

			if ( $script->deps ) {
				$script->deps = array_diff( $script->deps, array( 'jquery-migrate' ) );
			}
		}
	}

	/**
	 * Allows additional MIME types to be uploaded in WordPress.
	 *
	 * @param array $mimes The array of supported MIME types.
	 *
	 * @return array The modified array of MIME types.
	 */
	public function add_mime_types( $mimes ) {
		$mimes['svg'] = 'image/svg+xml';
		$mimes['json'] = 'text/plain';
		$mimes['json'] = 'application/json';
		$mimes['ogg'] = 'video/ogg';
		$mimes['csv'] = 'text/csv';
		$mimes['doc'] = 'application/msword';
		$mimes['docx'] = 'application/msword';
		$mimes['msg'] = 'application/vnd.ms-outlook';
		return $mimes;
	}
}
