<?php
/**
 * Elementor Integration
 *
 * Handles Elementor-related modifications and custom widget registrations.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

/**
 * Class Elementor
 *
 * Integrates custom functionality and modifications into Elementor.
 */
class Elementor extends \StrategySuite\Module {

	/**
	 * Determines if the module can be registered.
	 *
	 * This checks if the Elementor plugin is active before initializing the module.
	 *
	 * @return bool True if Elementor is active, false otherwise.
	 */
	public function can_register() {
		return defined( 'ELEMENTOR_PATH' ) && class_exists( 'Elementor\Widget_Base' );
	}

	/**
	 * Registers hooks and filters for Elementor customizations.
	 *
	 * This function registers custom Elementor widgets and applies filters
	 * to modify widget content.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'elementor/widgets/register', [ $this, 'widgets_registered' ] );
		add_filter( 'elementor/widget/render_content', [ $this, 'remove_invalid_attributes_from_menu' ], 10, 2 );
		add_filter( 'hello_elementor_description_meta_tag', '\__return_false' );
		add_filter( 'hello_elementor_enable_skip_link', '\__return_false' );
		add_filter( 'hello_elementor_header_footer', '\__return_false' );
		add_filter( 'hello_elementor_page_title', '\__return_false' );
		add_filter( 'hello_elementor_enqueue_style', '\__return_false' );
		add_filter( 'hello_elementor_enqueue_theme_style', '\__return_false' );

		add_action(
			'admin_menu',
			function () {
				remove_menu_page( 'hello-elementor' );
				remove_submenu_page( 'hello-elementor', 'hello-elementor' );
				remove_submenu_page( 'hello-elementor', 'hello-elementor-theme-builder' );
				remove_submenu_page( 'hello-elementor', 'hello-elementor-ai-site-planner' );
			},
			PHP_INT_MAX
		);

		if ( is_plugin_active( 'wp-rocket/wp-rocket.php' ) ) {
			add_filter( 'rocket_lazyload_background_images', '\__return_false' );
		}

		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

		add_action( 'admin_init', [ $this, 'set_theme_options' ] );
	}

	/**
	 * Enqueue scripts and styles for the accessibility toolbar.
	 *
	 * Registers and enqueues JavaScript and CSS files required for the accessibility toolbar.
	 * Also localizes script options to configure the toolbar behavior.
	 *
	 * @return void
	 */
	public function enqueue_scripts() {
		wp_enqueue_style(
			'strategy-suite-elementor',
			STRATEGY_SUITE_DIST_URL . 'css/elementor-style.css',
			[],
			STRATEGY_SUITE_VERSION
		);
	}

	/**
	 * Registers custom Elementor widgets.
	 *
	 * This method loads and includes additional Elementor widget files if available.
	 *
	 * @return void
	 */
	public function widgets_registered() {
		$template_file = STRATEGY_SUITE_INC . 'noautoload-classes/ElementorCopyright.php';
		if ( $template_file && is_readable( $template_file ) ) {
			require_once $template_file;
		}
	}

	/**
	 * Removes invalid attributes from Elementor's navigation menu widget.
	 *
	 * This function ensures accessibility compliance by modifying the
	 * attributes applied to the menu widget.
	 *
	 * @param string $widget_content The rendered widget content.
	 * @param object $widget The Elementor widget instance.
	 *
	 * @return string The filtered widget content.
	 */
	public function remove_invalid_attributes_from_menu( $widget_content, $widget ) {
		if ( 'nav-menu' === $widget->get_name() ) {
			$settings = $widget->get_settings();
			$widget_content = str_replace( 'migration_allowed="1"', '', $widget_content );
			$widget_content = str_replace( 'migrated="0"', '', $widget_content );
			$widget_content = str_replace( 'role="navigation"', 'aria-label="' . $settings['menu'] . '"', $widget_content );
		}

		return $widget_content;
	}

	/**
	 * Sets default theme options for Elementor's Hello theme.
	 *
	 * This function checks if the theme settings migration has already been completed.
	 * If not, it updates various theme options related to Elementor settings, such as
	 * meta tags, skip links, header/footer settings, page title visibility, and theme styles.
	 * Once the settings are updated, it marks the migration as complete.
	 *
	 * @return void
	 */
	public function set_theme_options() {
		if ( 1 == get_option( 'elementor_theme_settings_migration_done' ) ) {
			return;
		}

		update_option( 'hello_elementor_settings_description_meta_tag', false );
		update_option( 'hello_elementor_settings_skip_link', false );
		update_option( 'hello_elementor_settings_header_footer', false );
		update_option( 'hello_elementor_settings_page_title', false );
		update_option( 'hello_elementor_settings_hello_style', false );
		update_option( 'hello_elementor_settings_hello_theme', false );
		update_option( 'elementor_theme_settings_migration_done', 1 );
	}
}
