<?php
/**
 * DeleteOldRevisions
 *
 * This class handles the automated deletion of old post revisions in WordPress,
 * specifically targeting revisions older than one year. It utilizes WordPress
 * cron schedules to automate the cleanup process on a monthly basis.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use WP_Query;

/**
 * Class DeleteOldRevisions
 *
 * This class extends the StrategySuite\Module and manages the setup of cron jobs
 * to delete old post revisions in a WordPress site.
 */
class DeleteOldRevisions extends \StrategySuite\Module {

	/**
	 * Determine if the module can be registered.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register hooks for automating the deletion of old revisions.
	 *
	 * This method registers the necessary filters and actions to schedule
	 * and execute the cron job responsible for deleting old revisions.
	 *
	 * @return void
	 */
	public function register() {
		add_filter( 'cron_schedules', [ $this,  'add_monthly_interval' ] );
		add_action( 'wp', [ $this,  'schedule_cron' ] );
		add_action( 'delete_old_revisions_cron', [ $this,  'delete_old_revisions' ] );
	}

	/**
	 * Add a monthly interval to the WordPress cron schedule.
	 *
	 * This method adds a new cron schedule interval called 'monthly',
	 * which triggers events approximately every 30.44 days.
	 *
	 * @param array $schedules Existing cron schedules.
	 * @return array Modified cron schedules including the new monthly interval.
	 */
	public function add_monthly_interval( $schedules ) {
		$schedules['monthly'] = array(
			'interval' => 2.628E6, // Seconds in a month
			'display' => __( 'Every Month' ),
		);
		return $schedules;
	}

	/**
	 * Schedule the cron job for deleting old revisions.
	 *
	 * This method checks if the 'delete_old_revisions_cron' event is scheduled.
	 * If not, it schedules the event to run on the newly defined monthly interval.
	 *
	 * @return void
	 */
	public function schedule_cron() {
		if ( ! wp_next_scheduled( 'delete_old_revisions_cron' ) ) {
			wp_schedule_event( time(), 'monthly', 'delete_old_revisions_cron' );
		}
	}

	/**
	 * Delete revisions older than one year from the database.
	 *
	 * This method queries for post revisions that are older than one year
	 * and deletes them from the database to optimize storage and performance.
	 *
	 * @global \wpdb $wpdb WordPress database abstraction object.
	 * @return void
	 */
	public function delete_old_revisions() {
		global $wpdb;

		$time_ago = gmdate( 'Y-m-d H:i:s', apply_filters( 'strategy_suite_revisions_expiration', strtotime( '-1 year' ) ) );

		$revisions = new WP_Query(
			array(
				'post_type' => 'revision',
				'post_status' => 'any',
				'date_query' => array(
					'before' => $time_ago,
				),
				'fields' => 'ids',
				'nopaging' => true,
			)
		);

		if ( $revisions->have_posts() ) {
			$ids = implode( ',', $revisions->posts );
			$wpdb->query( "DELETE FROM $wpdb->posts WHERE ID IN ($ids)" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}
}
