<?php
/**
 * Attribution
 *
 * Handles UTM parameter tracking and pre-filling for Gravity Forms.
 *
 * @package StrategySuite
 */

namespace StrategySuite;

use GFAPI;
use GFFormsModel;

/**
 * Class Attribution
 *
 * This class integrates UTM tracking with Gravity Forms, allowing for
 * automatic population of UTM parameters in form submissions.
 */
class Attribution extends \StrategySuite\Module {

	/**
	 * List of UTM parameters to track.
	 *
	 * @var array
	 */
	private $tracking_params = [
		'utm_id',
		'utm_source',
		'utm_medium',
		'utm_campaign',
		'utm_term',
		'utm_content',
		'gclid',
		'dclid',
		'gbraid',
		'wbraid',
		'fbclid',
		'msclkid',
		'originating_utms',
	];

	/**
	 * Determines if the module can be registered.
	 *
	 * Checks if Gravity Forms is active before enabling UTM tracking.
	 *
	 * @return bool True if Gravity Forms is active, otherwise false.
	 */
	public function can_register() {
		return is_plugin_active( 'gravityforms/gravityforms.php' ) && apply_filters( 'strategy_suite_gf_attribution_enabled', true );
	}

	/**
	 * Registers hooks for UTM tracking and Gravity Forms integration.
	 *
	 * Adds filters and actions for:
	 * - Migrating UTM fields.
	 * - Enqueuing the UTM tracking script.
	 * - Dynamically pre-filling UTM fields.
	 * - Adding UTM fields to Gravity Forms.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'add_utm_fields_data_migration' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

		foreach ( $this->tracking_params as $param ) {
			add_filter( 'gform_field_value_' . $param, [ $this, 'prefill_utm_fields_values' ], 10, 2 );
		}

		add_filter( 'gform_form_settings_fields', [ $this, 'add_gtm_gform_settings_fields' ], 10, 2 );
		add_action( 'gform_after_save_form', [ $this, 'add_utm_fields' ], 99, 2 );
	}

	/**
	 * Adds a custom "GTM Settings" section to the Gravity Forms form settings.
	 *
	 * Appends a checkbox field to the existing form settings that allows administrators
	 * to indicate whether GTM automation has run successfully for the given form.
	 *
	 * @param array $fields Existing form settings fields.
	 * @param array $form   The current form object being edited.
	 *
	 * @return array Modified form settings fields including the GTM Settings section.
	 */
	public function add_gtm_gform_settings_fields( $fields, $form ) {
		$fields[] = array(
			'title'  => __( 'GTM Settings', 'strategy' ),
			'fields' => array(
				array(
					'name'    => 'gtm_settings',
					'type'    => 'checkbox',
					'choices' => array(
						array(
							'name'  => 'gtm_done',
							'label' => __( 'GTM Automation has run successfully', 'strategy' ),
						),
					),
					'value'   => rgar( $form, 'gtm_settings' ),
				),
			),
		);

		return $fields;
	}

	/**
	 * Migrates UTM fields into existing Gravity Forms.
	 *
	 * This ensures that all forms include UTM fields, allowing
	 * for tracking even on older forms.
	 *
	 * @return void
	 */
	public function add_utm_fields_data_migration() {
		if ( 3 == get_option( 'lg_utm_fields_migration_done' ) ) {
			return;
		}

		$forms = GFAPI::get_forms();

		if ( $forms ) {
			foreach ( $forms as $form ) {
				$this->utm_data_migration( $form );
			}
		}

		update_option( 'lg_utm_fields_migration_done', 3 );
	}

	/**
	 * Enqueues the UTM tracking JavaScript file.
	 *
	 * Also localizes the tracking parameters (excluding "originating_utms") for frontend use.
	 */
	public function enqueue_scripts() {
		wp_register_script(
			'lg-attribution',
			STRATEGY_SUITE_DIST_URL . 'js/attribution.js',
			[],
			STRATEGY_SUITE_VERSION,
			true
		);
		wp_enqueue_script( 'lg-attribution' );

		$frontend_tracking_params = array_values( array_diff( $this->tracking_params, [ 'originating_utms' ] ) );

		wp_localize_script(
			'lg-attribution',
			'lgAttribution',
			[
				'utmTrackingParams' => $frontend_tracking_params,
			]
		);
	}

	/**
	 * Prefills Gravity Forms UTM fields dynamically from cookies or query parameters.
	 *
	 * If a UTM parameter is found in a cookie, it takes precedence. If not, it
	 * checks for the parameter in the URL query string.
	 *
	 * @param mixed  $value The current field value.
	 * @param object $field The Gravity Forms field object.
	 *
	 * @return string The populated UTM value.
	 */
	public function prefill_utm_fields_values( $value, $field ) {
		$field_name = $field->inputName; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$cookie_value = isset( $_COOKIE[ $field_name ] ) ? sanitize_text_field( wp_unslash( $_COOKIE[ $field_name ] ) ) : null;
		$query_value = isset( $_GET[ $field_name ] ) ? sanitize_text_field( wp_unslash( $_GET[ $field_name ] ) ) : null;

		return $cookie_value ?? $query_value ?? $value;
	}

	/**
	 * Adds UTM fields dynamically to Gravity Forms if they do not already exist.
	 *
	 * This function ensures all UTM parameters are available as hidden fields
	 * in Gravity Forms, preventing duplicate fields from being added.
	 *
	 * @param array   $form   The current Gravity Form object.
	 * @param boolean $is_new Whether this is a new form being created. False for existing forms.
	 *
	 * @return void
	 */
	public function add_utm_fields( $form, $is_new ) {
		// Avoid duplicating fields by checking if they already exist
		$existing_fields = array_map(
			function ( $field ) {
				return $field->inputName; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			},
			$form['fields']
		);

		$fields_added = false;

		$next_field_id = GFFormsModel::get_next_field_id( $form['fields'] );

		foreach ( $this->tracking_params as $param ) {
			$param_nice_name = ucfirst( str_replace( '_', ' ', $param ) );
			if ( ! in_array( $param, $existing_fields ) ) {
				// Create a hidden field for the parameter
				$form['fields'][] = [
					'id'                => $next_field_id++,
					'type'              => 'hidden',
					'label'             => $param_nice_name,
					'inputName'         => $param,
					'defaultValue'      => '',
					'allowsPrepopulate' => true,
				];
				$fields_added = true;
			}
		}

		if ( ! in_array( 'referer', $existing_fields ) ) {
			$form['fields'][] = [
				'id'           => $next_field_id++,
				'type'         => 'hidden',
				'label'        => 'Referer',
				'inputName'    => 'referer',
				'defaultValue' => '{referer}',
				'allowsPrepopulate' => true,
			];
			$fields_added = true;
		}

		if ( ! in_array( 'gtm_last_title', $existing_fields ) ) {
			$form['fields'][] = [
				'id'                => $next_field_id++,
				'type'              => 'hidden',
				'label'             => 'Last Title',
				'inputName'         => 'gtm_last_title',
				'defaultValue'      => $form['title'],
				'allowsPrepopulate' => true,
			];
			$fields_added = true;
		}

		if ( $fields_added || $is_new ) {
			GFAPI::update_form( $form );
		}
	}

	/**
	 * Removes existing UTM fields from a Gravity Form.
	 *
	 * This ensures that old UTM fields are cleared before re-adding them during migration.
	 *
	 * @param array $form   The Gravity Form object.
	 *
	 * @return void
	 */
	public function utm_data_migration( $form ) {
		$form = $this->remove_duplicate_field_ids( $form );

		// Extract only the necessary field input names
		$utm_fields_to_remove = array_merge( $this->tracking_params, [ 'referer' ] );

		// Filter out fields that match UTM parameters
		$form['fields'] = array_filter(
			$form['fields'],
			function ( $field ) use ( $utm_fields_to_remove ) {
				if ( 'Referrer' === $field->label || 'Submitted URL' === $field->label ) {
					return false;
				}
				return ! in_array( $field->inputName, $utm_fields_to_remove, true ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			}
		);

		// Avoid duplicating fields by checking if they already exist
		$existing_fields = array_map(
			function ( $field ) {
				return $field->inputName; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			},
			$form['fields']
		);

		$fields_added = false;

		$next_field_id = GFFormsModel::get_next_field_id( $form['fields'] );

		foreach ( $this->tracking_params as $param ) {
			$param_nice_name = ucfirst( str_replace( '_', ' ', $param ) );
			if ( ! in_array( $param, $existing_fields ) ) {
				// Create a hidden field for the parameter
				$form['fields'][] = [
					'id'                => $next_field_id++,
					'type'              => 'hidden',
					'label'             => $param_nice_name,
					'inputName'         => $param,
					'defaultValue'      => '',
					'allowsPrepopulate' => true,
				];
				$fields_added = true;
			}
		}

		if ( ! in_array( 'referer', $existing_fields ) ) {
			$form['fields'][] = [
				'id'           => $next_field_id++,
				'type'         => 'hidden',
				'label'        => 'Referer',
				'inputName'    => 'referer',
				'defaultValue' => '{referer}',
				'allowsPrepopulate' => true,
			];
			$fields_added = true;
		}

		GFAPI::update_form( $form );
	}

	/**
	 * Removes duplicate field IDs from a Gravity Form.
	 *
	 * This function ensures that each field ID appears only once in the form's fields array.
	 * If duplicate field IDs are found, only the first occurrence is retained.
	 *
	 * @param array $form The Gravity Form object containing fields.
	 *
	 * @return array The updated form with duplicate field IDs removed.
	 */
	private function remove_duplicate_field_ids( $form ) {
		$seen_field_ids = [];
		$unique_fields = [];

		foreach ( $form['fields'] as $field ) {
			// Ensure each field ID appears only once
			if ( isset( $seen_field_ids[ $field->id ] ) ) {
				continue; // Skip duplicate field IDs
			}

			$seen_field_ids[ $field->id ] = true;
			$unique_fields[] = $field;
		}

		// Update the form with only unique fields
		$form['fields'] = $unique_fields;

		return $form;
	}
}
