<?php
/**
 * Reviews
 *
 * @package StrategySpamProtection
 */

namespace StrategySpamProtection;

use GFCommon;
use GFFormsModel;

/**
 * Creates the StrategySpamProtection post type and shortcodes to output them.
 */
class SpamProtection extends \StrategySpamProtection\Module {
	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		if ( ! defined( 'STRATEGY_ZERO_BOUNCE_API_KEY' ) ) {
			// TODO: Remove from codebase and add as definable constant, allow to set via a db setting as well.
			define( 'STRATEGY_ZERO_BOUNCE_API_KEY', '806f4f95d19942fa9ac93135b7afa901' );
		}

		add_filter( 'gform_ip_address', [ $this, 'cloudflare_gform_ip_address' ] );
		add_filter( 'gform_entry_is_spam', [ $this, 'filter_gform_entry_is_spam_ip_rate_limit' ], 11, 3 );
		add_filter( 'gform_entry_is_spam', [ $this, 'filter_gform_entry_is_spam_zerobounce' ], 25, 3 );
		add_action( 'gform_after_submission', [ $this, 'remove_strategy_mail_hooks_if_spam' ], 1, 2 );
		add_filter( 'gform_form_settings_fields', [ $this, 'add_settings_field' ], 10, 2 );
	}

	/**
	 * Verifies email addresses in Gravity Forms submissions using the ZeroBounce Email Validation API.
	 *
	 * @param boolean $is_spam      Whether the submission is spam.
	 * @param object  $form          The form object that was used for submission.
	 * @param object  $entry         The entry that was submitted.
	 *
	 * @return boolean $is_spam     Whether the submission is spam.
	 */
	public function filter_gform_entry_is_spam_zerobounce( $is_spam, $form, $entry ) {
		if ( ! rgar( $form, 'enableZeroBounce', false ) ) {
			return $is_spam;
		}

		if ( $is_spam ) {
			GFCommon::log_debug( __METHOD__ . '(): Entry marked as spam prior to ZeroBounce check.' );
			return $is_spam;
		}

		// Always not spam for users with administrator role.
		if ( $is_spam && current_user_can( 'manage_options' ) ) {
			GFCommon::log_debug( __METHOD__ . '(): Entry marked as not spam for administrator role.' );
			return false;
		}

		if ( ! defined( 'STRATEGY_ZERO_BOUNCE_API_KEY' ) ) {
			GFCommon::log_debug( __METHOD__ . '(): No ZeroBounce API key was set.' );
			return $is_spam;
		}

		$field_ids = $this->get_email_field_ids( $form );
		if ( empty( $field_ids ) ) {
			GFCommon::log_debug( __METHOD__ . '(): No email fields were in the form.' );
			return $is_spam;
		}

		foreach ( $field_ids as $field_id ) {
			$email    = rgar( $entry, $field_id );

			$ip = rgars( $form, 'personalData/preventIP' ) ? GFFormsModel::get_ip() : rgar( $entry, 'ip' );

			$request_url = add_query_arg(
				array(
					'email'      => $email,
					'ip_address' => $ip,
					'api_key'    => STRATEGY_ZERO_BOUNCE_API_KEY,
				),
				'https://api.zerobounce.net/v2/validate'
			);

			$response = wp_remote_get( $request_url );

			if ( is_wp_error( $response ) ) {
				GFCommon::log_debug( __METHOD__ . '(): ZeroBounce request failed: ' . $response->get_error_message() );
				return $is_spam;
			}

			$status_code = wp_remote_retrieve_response_code( $response );
			$response_body = wp_remote_retrieve_body( $response );
			$data = json_decode( $response_body, true );

			if ( 200 !== $status_code ) {
				GFCommon::log_debug( __METHOD__ . '(): ZeroBounce returned non-200 response: ' . $status_code );
				return $is_spam;
			}

			if ( isset( $data['error'] ) ) {
				GFCommon::log_debug( __METHOD__ . '(): ZeroBounce error - ' . $data['error'] );
				return $is_spam;
			}

			if ( ! is_array( $data ) || empty( $data['status'] ) ) {
				GFCommon::log_debug( __METHOD__ . '(): ZeroBounce response invalid or missing status. Response: ' . print_r( $data, true ) );
				return $is_spam;
			}

			$valid_statuses = [ 'valid', 'catch-all', 'do_not_mail' ];
			$valid_sub_statuses = [ 'greylisted', 'role_based', 'role_based_catch_all' ];

			$status = rgar( $data, 'status' );
			$sub_status = rgar( $data, 'sub_status' );

			if ( 'do_not_mail' === $status && in_array( $sub_status, $valid_sub_statuses ) ) {
				GFCommon::log_debug( __METHOD__ . '(): The email was marked as do_not_mail with an acceptable sub_status.' );
				continue;
			}

			if ( ! in_array( $status, $valid_statuses ) ) {
				GFCommon::set_spam_filter( rgar( $form, 'id' ), 'ZeroBounce', 'The email was invalid.' );
				$is_spam = true;
			}
		}

		return $is_spam;
	}

	/**
	 * Marks submissions as spam if they are within a particular rate limit.
	 *
	 * @param boolean $is_spam      Whether the submission is spam.
	 * @param object  $form          The form object that was used for submission.
	 * @param object  $entry         The entry that was submitted.
	 *
	 * @return boolean $is_spam     Whether the submission is spam.
	 */
	public function filter_gform_entry_is_spam_ip_rate_limit( $is_spam, $form, $entry ) {
		if ( ! rgar( $form, 'enableRateLimiting', false ) ) {
			return $is_spam;
		}

		if ( $is_spam ) {
			GFCommon::log_debug( __METHOD__ . '(): Entry marked as spam prior to IP Rate Limit check.' );
			return $is_spam;
		}

		// Always not spam for users with administrator role.
		if ( current_user_can( 'manage_options' ) ) {
			GFCommon::log_debug( __METHOD__ . '(): Entry marked as not spam for administrator role.' );
			return false;
		}

		$ip_address = rgars( $form, 'personalData/preventIP' ) ? GFFormsModel::get_ip() : rgar( $entry, 'ip' );
		if ( ! filter_var( $ip_address, FILTER_VALIDATE_IP ) ) {
			GFCommon::set_spam_filter( rgar( $form, 'id' ), 'Rate Limit', 'Entry marked as spam because of a malformed or spoofed IP address.' );
			return true;
		}

		$key   = wp_hash( __FUNCTION__ . $ip_address . '/' . rgar( $form, 'id' ) );
		$count = (int) get_transient( $key );

		if ( $count >= 3 ) {
			GFCommon::set_spam_filter( rgar( $form, 'id' ), 'Rate Limit', 'Entry marked as spam because of a rate limited IP address.' );
			return true;
		}

		$count++;
		set_transient( $key, $count, HOUR_IN_SECONDS );

		return false;
	}

	/** Gets all email field IDs from a specific form.
	 *
	 * @param array $form The form object.
	 *
	 * @return array $field_ids     The field ids of all email fields.
	 */
	private function get_email_field_ids( $form ) {
		$field_ids = [];

		foreach ( $form['fields'] as $field ) {
			if ( 'email' == $field->type && $field->isRequired ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$field_ids[] = $field->id;
			}
		}

		return $field_ids;
	}

	/** Return the IP address provided by CloudFlare.
	 *
	 * @param string $ip The IP address Gravity Forms is returning.
	 *
	 * @return array $ip     The modified $ip
	 */
	public function cloudflare_gform_ip_address( $ip ) {
		if ( isset( $_SERVER['CF-Connecting-IP'] ) && ! empty( $_SERVER['CF-Connecting-IP'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['CF-Connecting-IP'] ) );
			GFCommon::log_debug( __METHOD__ . '(): CF-Connecting-IP: ' . $ip );
		} elseif ( isset( $_SERVER['HTTP_CF_CONNECTING_IP'] ) && ! empty( $_SERVER['HTTP_CF_CONNECTING_IP'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CF_CONNECTING_IP'] ) );
			GFCommon::log_debug( __METHOD__ . '(): HTTP_CF_CONNECTING_IP: ' . $ip );
		}

		return $ip;
	}

	/** Prevents Strategy Mail callbacks from running during a Gravity Form submission.
	 *
	 * @param string $entry The Entry Object of the entry that was just created.
	 * @param object $form The Form Object of the current form.
	 *
	 * @return void
	 */
	public function remove_strategy_mail_hooks_if_spam( $entry, $form ) {
		if ( rgar( $entry, 'status' ) === 'spam' ) {
			GFCommon::log_debug( __METHOD__ . '(): Strategy Mail was prevented from running because of this entry is spam.' );
			$this->prevent_contact_creation_after_spam( 'gform_after_submission', [ 'GroundhoggGForms\Steps\GForm::setup', 'GroundhoggGForms\Steps\GForm::complete' ] );
		}
	}

	/** Removes callbacks from a certain hook.
	 *
	 * @param string $hook_name The hook to remove the callbacks from.
	 * @param array  $functions The callback function names to remove.
	 *
	 * @return void
	 */
	private function prevent_contact_creation_after_spam( $hook_name, $functions = [] ) {
		global $wp_filter;

		if ( ! isset( $wp_filter[ $hook_name ] ) ) {
			return;
		}

		// List all callbacks hooked into the specified action
		$hook = $wp_filter[ $hook_name ];
		if ( isset( $hook->callbacks ) && is_array( $hook->callbacks ) ) {
			foreach ( $hook->callbacks as $priority => $callbacks ) {
				foreach ( $callbacks as $callback ) {
					if ( is_array( $callback['function'] ) ) {
						$function_name = get_class( $callback['function'][0] ) . '::' . $callback['function'][1];
					} elseif ( is_object( $callback['function'] ) ) {
						$function_name = get_class( $callback['function'] ) . '::__invoke';
					} else {
						$function_name = $callback['function'];
					}

					// Output the callback information
					// echo 'Hook: ' . $hook_name . ' | Priority: ' . $priority . ' | Function: ' . $function_name . '<br>';

					// Condition to remove certain hooks
					if ( in_array( $function_name, $functions ) ) {
						remove_action( $hook_name, $callback['function'], $priority );
					}
				}
			}
		}
	}

	/**
	 * Adds the Strategy Spam Protection fields to the "Form Options" settings group in GF.
	 *
	 * @see https://docs.gravityforms.com/gform_form_settings_fields/
	 *
	 * @param array $fields Form Settings fields.
	 * @param array $form   The current form.
	 *
	 * @return array
	 */
	public function add_settings_field( $fields, $form = [] ) {

		$fields['form_options']['fields'][] = array(
			'name'          => 'enableRateLimiting',
			'type'          => 'toggle',
			'label'         => esc_html__( 'Prevent spam using rate limiting', 'sps' ),
			'default_value' => apply_filters( 'sps_rate_limiting_default', true, $form ),
		);

		$fields['form_options']['fields'][] = array(
			'name'          => 'enableZeroBounce',
			'type'          => 'toggle',
			'label'         => esc_html__( 'Prevent spam using ZeroBounce', 'sps' ),
			'default_value' => apply_filters( 'sps_zero_bounce_default', true, $form ),
		);

		return $fields;
	}
}
