<?php
/**
 * Reviews
 *
 * @package StrategyPlugin
 */

namespace StrategyPlugin;

use GFAPI;
use StrategyPlugin\Utility;

/**
 * Reviews
 */
class Reviews extends \StrategyPlugin\Module {

	/**
	 * Feedback Form ID
	 *
	 * @var string
	 */
	private $feedback_form_id;

	/**
	 * Feedback Form title
	 *
	 * @var string
	 */
	private $feedback_form_title;

	/**
	 * Locations
	 *
	 * @var array
	 */
	private $locations;

	/**
	 * Current Location ID
	 *
	 * @var string
	 */
	private $current_location_id;

	/**
	 * Rating Threshold
	 *
	 * @var int
	 */
	private $rating_threshold;

	/**
	 * Rating System
	 *
	 * @var int
	 */
	private $rating_system;

	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'init', [ $this, 'register_post_type' ] );
		add_action( 'acf/init', [ $this, 'register_acf_field_groups' ] );
		add_action( 'acf/init', [ $this, 'register_acf_settings_page' ] );
		add_action( 'template_redirect', [ $this, 'set_variables' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueues' ] );
		// add_action( 'wp_footer', [ $this, 'add_modal' ] );

		add_shortcode( 'review_modal', [ $this, 'strategy_review_modal' ] );
		add_shortcode( 'review_us', [ $this, 'strategy_review_content' ] );
	}

	/**
	 * Scaffolds the class properties.
	 *
	 * @return void
	 */
	public function set_variables() {
		$this->feedback_form_title = apply_filters( 'strategy_reviews_review_form_title', 'Review Feedback Form' );
		$this->locations = $this->get_locations();
		$this->feedback_form_id = $this->get_feedback_form_id();
		$this->rating_threshold = get_field( 'rating_threshold', 'options' ) ?? 4;
		$this->rating_system = get_field( 'rating_system', 'options' ) ?? 'stars';
		$this->current_location_id = $this->get_current_location();
	}

	/**
	 * Register the location post type.
	 *
	 * @return void
	 */
	public function register_post_type() {
		register_post_type(
			'location',
			array(
				'labels' => array(
					'name' => 'Locations',
					'singular_name' => 'Location',
					'menu_name' => 'Locations',
					'all_items' => 'All Locations',
					'edit_item' => 'Edit Location',
					'view_item' => 'View Location',
					'view_items' => 'View Locations',
					'add_new_item' => 'Add New Location',
					'new_item' => 'New Location',
					'parent_item_colon' => 'Parent Location:',
					'search_items' => 'Search Locations',
					'not_found' => 'No locations found',
					'not_found_in_trash' => 'No locations found in Trash',
					'archives' => 'Location Archives',
					'attributes' => 'Location Attributes',
					'insert_into_item' => 'Insert into location',
					'uploaded_to_this_item' => 'Uploaded to this location',
					'filter_items_list' => 'Filter locations list',
					'filter_by_date' => 'Filter locations by date',
					'items_list_navigation' => 'Locations list navigation',
					'items_list' => 'Locations list',
					'item_published' => 'Location published.',
					'item_published_privately' => 'Location published privately.',
					'item_reverted_to_draft' => 'Location reverted to draft.',
					'item_scheduled' => 'Location scheduled.',
					'item_updated' => 'Location updated.',
					'item_link' => 'Location Link',
					'item_link_description' => 'A link to a location.',
				),
				'public' => true,
				'show_in_rest' => true,
				'show_ui' => true,
				'show_in_menu' => true,
				'show_in_nav_menus' => true,
				'has_archive' => false,
				'menu_icon' => 'dashicons-location',
				'supports' => array(
					0 => 'title',
					1 => 'editor',
					2 => 'excerpt',
					3 => 'revisions',
					4 => 'thumbnail',
					5 => 'custom-fields',
				),
				'delete_with_user' => false,
			)
		);
	}

	/**
	 * Register the ACF field groups.
	 *
	 * @return void
	 */
	public function register_acf_field_groups() {
		acf_add_local_field_group(
			array(
				'key' => 'group_656787970bd4a',
				'title' => 'Strategy Reviews',
				'fields' => array(
					array(
						'key' => 'field_65678798cfc41',
						'label' => 'Review Platforms',
						'name' => 'review_platforms',
						'aria-label' => '',
						'type' => 'group',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'block',
						'sub_fields' => [
							array(
								'key' => 'field_656787b6cfc42',
								'label' => 'Google',
								'name' => 'google',
								'aria-label' => '',
								'type' => 'url',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'placeholder' => '',
							),
							array(
								'key' => 'field_656787d71cd1e',
								'label' => 'Facebook',
								'name' => 'facebook',
								'aria-label' => '',
								'type' => 'url',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'placeholder' => '',
							),
							array(
								'key' => 'field_656787e51cd1f',
								'label' => 'Yelp!',
								'name' => 'yelp',
								'aria-label' => '',
								'type' => 'url',
								'instructions' => '',
								'required' => 0,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'placeholder' => '',
							),
						],
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'post_type',
							'operator' => '==',
							'value' => 'location',
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			)
		);

		acf_add_local_field_group(
			array(
				'key' => 'group_65fa505911beb',
				'title' => 'Review Us Settings',
				'fields' => array(
					array(
						'key' => 'field_65fa505b8c2d0',
						'label' => 'Rating Threshold',
						'name' => 'rating_threshold',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => 'The minimum rating needed to be presented your review links. Emoji values are 1, 3, and 5.',
						'required' => 1,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_65fa50ab8c2d1',
									'operator' => '!=empty',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => 4,
						'min' => 1,
						'max' => 5,
						'placeholder' => '',
						'step' => 1,
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_65fa50ab8c2d1',
						'label' => 'Rating System',
						'name' => 'rating_system',
						'aria-label' => '',
						'type' => 'radio',
						'instructions' => '',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							'stars' => 'Stars',
							'emojis' => 'Emojis',
						),
						'default_value' => 'stars',
						'return_format' => 'value',
						'allow_null' => 0,
						'other_choice' => 0,
						'layout' => 'vertical',
						'save_other_choice' => 0,
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'options_page',
							'operator' => '==',
							'value' => 'acf-options-review-us-settings',
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			)
		);
	}

	/**
	 * Register the ACF settings page.
	 *
	 * @return void
	 */
	public function register_acf_settings_page() {
		acf_add_options_sub_page(
			array(
				'page_title'     => 'Review Us Settings',
				'menu_title'    => 'Review Us Settings',
				'parent_slug'    => 'edit.php?post_type=location',
			)
		);
	}

	/**
	 * Enqueues the necessary scripts and styles.
	 *
	 * @return void
	 */
	public function enqueues() {
		wp_enqueue_style(
			'strategy_reviews_css',
			Utility\style_url( 'style', 'frontend' ),
			[],
			Utility\get_asset_info( 'style', 'version' ),
		);

		if ( wp_get_theme()->stylesheet === 'hello-elementor-child' ) {
			wp_enqueue_style(
				'elementor_overrides_css',
				Utility\style_url( 'style', 'frontend' ),
				[],
				Utility\get_asset_info( 'style', 'version' ),
			);
		}

		wp_enqueue_script(
			'strategy_reviews_js',
			Utility\script_url( 'scripts', 'frontend' ),
			[ 'jquery' ],
			Utility\get_asset_info( 'scripts', 'version' ),
			true
		);

		wp_localize_script(
			'strategy_reviews_js',
			'strategy_reviews',
			array(
				'form_id' => $this->feedback_form_id,
				'threshold' => $this->rating_threshold,
			)
		);
	}

	/**
	 * Add Review Modal
	 *
	 * @return void
	 */
	public function add_modal() {
		if ( ! empty( $this->locations ) ) {
			echo do_shortcode( '[review_modal]' );
		}
	}

	/**
	 * Determine if the current page is a location page.
	 *
	 * @return string
	 */
	private function is_location_page() {
		return is_singular( 'location' );
	}

	/**
	 * Get Locations List
	 *
	 * @return array
	 */
	private function get_locations() {

		// TODO: Add filter for args
		$raw_locations = get_posts(
			array(
				'post_type'     => 'location',
				'posts_status'  => 'publish',
				'numberposts'   => -1,
				'order'         => 'ASC',
			)
		);

		$locations = array();
		if ( ! empty( $raw_locations ) ) {
			foreach ( $raw_locations as $location ) {
				if ( $this->has_review_links( $location->ID ) ) {
					array_push(
						$locations,
						array(
							'id' => $location->ID,
							'slug' => $location->post_name,
							'name' => $location->post_title,
							'vendors' => $this->get_review_vendors( $location->ID ),
						)
					);
				}
			}
		}

		return $locations;
	}

	/**
	 * Get Current Location
	 *
	 * @return int
	 */
	private function get_current_location() {
		if ( isset( $_GET['review_location'] ) ) {
			return (int) sanitize_text_field( wp_unslash( $_GET['review_location'] ) );
		}

		if ( $this->is_location_page() ) {
			return get_the_ID();
		}

		return null;
	}

	/**
	 * Check if the location has review links.
	 *
	 * @param int $post_id The post ID.
	 * @return bool
	 */
	private function has_review_links( $post_id ) {
		$vendors = $this->get_review_vendors( $post_id );
		foreach ( $vendors as $vendor ) {
			if ( ! empty( $vendor ) ) {
				return true;
			}
		}
	}

	/**
	 * Retrieves the feedback form ID.
	 *
	 * @return int
	 */
	private function get_feedback_form_id() {
		// Check database option
		$id = get_option( 'review_system_form_id' );
		if ( $id && GFAPI::get_form( $id ) ) {
			return $id;
		}

		// Check all forms for title
		$id = $this->get_form_id_by_title( $this->feedback_form_title );
		if ( $id && GFAPI::get_form( $id ) ) {
			update_option( 'review_system_form_id', $id );
			return $id;
		}

		return null;
	}

	/**
	 * Retrieves the form ID by title.
	 *
	 * @param string $title The form title.
	 * @return int
	 */
	private function get_form_id_by_title( $title ) {

		$forms = GFAPI::get_forms();
		foreach ( $forms as $form ) {
			if ( isset( $form['title'] ) && isset( $form['id'] ) && $title == $form['title'] ) {
				return $form['id'];
			}
		}

		return false;
	}

	/**
	 * Get Review Vendors
	 *
	 * @param int $post_id The post ID.
	 * @return array
	 */
	private function get_review_vendors( $post_id ) {
		$review_object = get_field_object( 'review_platforms', $post_id );

		if ( ! $review_object ) {
			return [];
		}

		$review_values = get_field( 'review_platforms', $post_id );
		foreach ( $review_object['sub_fields'] as $key => $sub_field ) {
			$review_object['sub_fields'][ $key ]['value'] = $review_values[ $sub_field['name'] ];
		}
		return $review_object['sub_fields'];
	}

	/**
	 * Review Modal
	 *
	 * @return string
	 */
	public function strategy_review_modal() {
		ob_start();
		?>
	
		<div id="review-modal" aria-hidden="true">
			<div class="modal-overlay" tabindex="-1" data-micromodal-close>
				<div id="review-modal--display" class="rounded-xs" role="dialog" aria-modal="true" aria-labelledby="review-modal--title">
					<header>
						<h2 id="review-modal__title">
							Leave a Review
						</h2>
						<button aria-label="Close modal" data-micromodal-close class="btn btn-tertiary">✖</button>
					</header>
					<?php echo do_shortcode( '[review_us]' ); ?>
				</div>
			</div>
		</div>
	
		<?php
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}

	/**
	 * Review Content
	 *
	 * @return string
	 */
	public function strategy_review_content() {
		ob_start();
		?>
	
		<div id="review-content" class="review-content">
			<?php
			$design_settings = get_field( 'design_settings', 'options' );
			$logo = get_theme_mod( 'custom_logo' );

			if ( $logo ) {
				echo wp_get_attachment_image( $logo, 'full', false, [ 'class' => 'review-content__logo' ] );
			} else if ( $placeholder_dims && count( array_filter( $placeholder_dims ) ) === 2 ) {
				$placeholder_dims = '200x100';
				?>
				<img class="review-content__logo" src="<?php echo esc_url( 'https://placehold.co/' . $placeholder_dims ); ?>" alt="Placeholder - Please Replace Me" title="Placeholder - Please Replace Me" />
				<?php
			} else {
				bloginfo( 'name' );
			}
			?>

			<div class="review-content__error" id="review-error" role="alert">
				<p class="review-content__message" id="review-error-message"></p>
			</div>
			<div id="review-content__1" class="review-content__page" data-value="1">
				<?php $this->review_page_1(); ?>
			</div>
			<div id="review-content__2" class="review-content__page" data-value="2">
				<?php $this->review_page_2(); ?>
			</div>
			<div id="review-content__3" class="review-content__page" data-value="3">
				<?php $this->review_page_3(); ?>
			</div>
			<div id="review-content__4" class="review-content__page" data-value="4">
				<?php $this->review_page_4(); ?>
			</div>
			<div class="review-content__pagination">
				<a id="review-prev-btn" class="btn btn-tertiary review-content__pagination--prev">Prev</a>
				<a id="review-next-btn" class="btn btn-primary review-content__pagination--next">Next</a>
				<a id="review-direct-btn" class="btn btn-primary review-content__pagination--next">Leave A Review</a>
			</div>
		</div>
	
		<?php
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}

	/**
	 * Review Page 1
	 *
	 * @return void
	 */
	private function review_page_1() {
		?>
		<div class="content--container">
			<h2 class="h2">Review Us</h2>
			<p>Please select the location you have received service at, and let us know how we did.</p>
			<div class="content__location">
				<label class="sr-only" for="content__location--select">Select your location:</label>
				<select id="content__location--select">
					<?php if ( is_array( $this->locations ) && count( $this->locations ) > 1 ) { ?>
						<option value="">Select Location</option>
					<?php } ?>
					<?php foreach ( $this->locations as $location ) { ?>
						<option value="<?php echo esc_attr( $location['id'] ); ?>" <?php echo ! is_null( $this->current_location_id ) && $this->current_location_id === $location['id'] ? 'selected' : null; ?>><?php echo esc_html( $location['name'] ); ?></option>
					<?php } ?>
				</select>
			</div>
		</div>
		<?php
	}

	/**
	 * Review Page 2
	 *
	 * @return void
	 */
	public function review_page_2() {
		?>
		<div class="content--container">
			<h2 class="h2">Rate Us</h2>
			<p>Your feedback is very important to us. How would you rate the service you received?</p>
			<div class="content__rating">
				<?php if ( 'stars' === $this->rating_system ) { ?>
					<input class="review-star" type="radio" id="review-star1" value="1">
					<label for="review-star1"></label>
					<input class="review-star" type="radio" id="review-star2" value="2">
					<label for="review-star2"></label>
					<input class="review-star" type="radio" id="review-star3" value="3">
					<label for="review-star3"></label>
					<input class="review-star" type="radio" id="review-star4" value="4">
					<label for="review-star4"></label>
					<input class="review-star" type="radio" id="review-star5" value="5">
					<label for="review-star5"></label>
				<?php } else { ?>
					<input class="review-star emoji" type="radio" id="review-star1" value="1">
					<label for="review-star1" class="review__emoji">
						<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path d="M464 256A208 208 0 1 0 48 256a208 208 0 1 0 416 0zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zM174.6 384.1c-4.5 12.5-18.2 18.9-30.7 14.4s-18.9-18.2-14.4-30.7C146.9 319.4 198.9 288 256 288s109.1 31.4 126.6 79.9c4.5 12.5-2 26.2-14.4 30.7s-26.2-2-30.7-14.4C328.2 358.5 297.2 336 256 336s-72.2 22.5-81.4 48.1zM144.4 208a32 32 0 1 1 64 0 32 32 0 1 1 -64 0zm192-32a32 32 0 1 1 0 64 32 32 0 1 1 0-64z"/></svg>    
					</label>
					<input class="review-star emoji" type="radio" id="review-star3" value="3">
					<label for="review-star3" class="review__emoji">
						<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path d="M464 256A208 208 0 1 1 48 256a208 208 0 1 1 416 0zM256 0a256 256 0 1 0 0 512A256 256 0 1 0 256 0zM176.4 240a32 32 0 1 0 0-64 32 32 0 1 0 0 64zm192-32a32 32 0 1 0 -64 0 32 32 0 1 0 64 0zM184 328c-13.3 0-24 10.7-24 24s10.7 24 24 24H328c13.3 0 24-10.7 24-24s-10.7-24-24-24H184z"/></svg>
					</label>
					<input class="review-star emoji" type="radio" id="review-star5" value="5">
					<label for="review-star5" class="review__emoji">
						<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path d="M464 256A208 208 0 1 0 48 256a208 208 0 1 0 416 0zM0 256a256 256 0 1 1 512 0A256 256 0 1 1 0 256zm177.6 62.1C192.8 334.5 218.8 352 256 352s63.2-17.5 78.4-33.9c9-9.7 24.2-10.4 33.9-1.4s10.4 24.2 1.4 33.9c-22 23.8-60 49.4-113.6 49.4s-91.7-25.5-113.6-49.4c-9-9.7-8.4-24.9 1.4-33.9s24.9-8.4 33.9 1.4zm40-89.3l0 0 0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0zm160 0l0 0-.2-.2c-.2-.2-.4-.5-.7-.9c-.6-.8-1.6-2-2.8-3.4c-2.5-2.8-6-6.6-10.2-10.3c-8.8-7.8-18.8-14-27.7-14s-18.9 6.2-27.7 14c-4.2 3.7-7.7 7.5-10.2 10.3c-1.2 1.4-2.2 2.6-2.8 3.4c-.3 .4-.6 .7-.7 .9l-.2 .2 0 0 0 0 0 0c-2.1 2.8-5.7 3.9-8.9 2.8s-5.5-4.1-5.5-7.6c0-17.9 6.7-35.6 16.6-48.8c9.8-13 23.9-23.2 39.4-23.2s29.6 10.2 39.4 23.2c9.9 13.2 16.6 30.9 16.6 48.8c0 3.4-2.2 6.5-5.5 7.6s-6.9 0-8.9-2.8l0 0 0 0 0 0z"/></svg>
					</label>
				<?php } ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Review Page 4
	 *
	 * @return void
	 */
	public function review_page_3() {
		?>
		<div class="content--container" id="review-page">
			<h2 class="h2">Leave A Review</h2>
			<p>Please leave a review on any of the platforms listed below by clicking or tapping on one.</p>
			<?php foreach ( $this->locations as $location ) { ?>
				<div class="content__review-vendors" data-location="<?php echo esc_attr( $location['id'] ); ?>">
					<?php foreach ( $location['vendors'] as $vendor ) { ?>
						<?php if ( ! empty( $vendor['value'] ) ) { ?>
							<a class="review-vendor__link" href="<?php echo esc_url( $vendor['value'] ); ?>" target="_blank" rel="noopener" data-vendor="<?php echo esc_attr( $vendor['name'] ); ?>">
								<?php if ( 'facebook' === $vendor['name'] ) { ?>
									<svg class="review-vendor__icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><!--! Font Awesome Free 6.3.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free (Icons: CC BY 4.0, Fonts: SIL OFL 1.1, Code: MIT License) Copyright 2023 Fonticons, Inc. --><path d="M279.14 288l14.22-92.66h-88.91v-60.13c0-25.35 12.42-50.06 52.24-50.06h40.42V6.26S260.43 0 225.36 0c-73.22 0-121.08 44.38-121.08 124.72v70.62H22.89V288h81.39v224h100.17V288z"/></svg>
								<?php } ?>
								<?php if ( 'yelp' === $vendor['name'] ) { ?>
									<svg class="review-vendor__icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><!--! Font Awesome Free 6.3.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free (Icons: CC BY 4.0, Fonts: SIL OFL 1.1, Code: MIT License) Copyright 2023 Fonticons, Inc. --><path d="M42.9 240.32l99.62 48.61c19.2 9.4 16.2 37.51-4.5 42.71L30.5 358.45a22.79 22.79 0 0 1-28.21-19.6 197.16 197.16 0 0 1 9-85.32 22.8 22.8 0 0 1 31.61-13.21zm44 239.25a199.45 199.45 0 0 0 79.42 32.11A22.78 22.78 0 0 0 192.94 490l3.9-110.82c.7-21.3-25.5-31.91-39.81-16.1l-74.21 82.4a22.82 22.82 0 0 0 4.09 34.09zm145.34-109.92l58.81 94a22.93 22.93 0 0 0 34 5.5 198.36 198.36 0 0 0 52.71-67.61A23 23 0 0 0 364.17 370l-105.42-34.26c-20.31-6.5-37.81 15.8-26.51 33.91zm148.33-132.23a197.44 197.44 0 0 0-50.41-69.31 22.85 22.85 0 0 0-34 4.4l-62 91.92c-11.9 17.7 4.7 40.61 25.2 34.71L366 268.63a23 23 0 0 0 14.61-31.21zM62.11 30.18a22.86 22.86 0 0 0-9.9 32l104.12 180.44c11.7 20.2 42.61 11.9 42.61-11.4V22.88a22.67 22.67 0 0 0-24.5-22.8 320.37 320.37 0 0 0-112.33 30.1z"/></svg>
								<?php } ?>
								<?php if ( 'google' === $vendor['name'] ) { ?>
									<svg class="review-vendor__icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 488 512"><!--! Font Awesome Free 6.3.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free (Icons: CC BY 4.0, Fonts: SIL OFL 1.1, Code: MIT License) Copyright 2023 Fonticons, Inc. --><path d="M488 261.8C488 403.3 391.1 504 248 504 110.8 504 0 393.2 0 256S110.8 8 248 8c66.8 0 123 24.5 166.3 64.9l-67.5 64.9C258.5 52.6 94.3 116.6 94.3 256c0 86.5 69.1 156.6 153.7 156.6 98.2 0 135-70.4 140.8-106.9H248v-85.3h236.1c2.3 12.7 3.9 24.9 3.9 41.4z"/></svg>
								<?php } ?>
							</a>
						<?php } ?>
					<?php } ?>
				</div>
			<?php } ?>
		</div>
		<div class="content--container" id="feedback-page">
			<h2 class="h2">Tell Us What Went Wrong</h2>
			<p>We are sorry to hear that you didn't have the greatest experience. Please provide some feedback about your experience so that we can improve our services.</p>
			<?php echo do_shortcode( '[gravityform id="' . $this->feedback_form_id . '" title="false" ajax="true"]' ); ?>
			<div class="content__review-divider"><span class="content__review-divider-text">OR</span></div>
		</div>
		<?php
	}

	/**
	 * Review Page 4
	 *
	 * @return void
	 */
	public function review_page_4() {
		?>
		<div class="content--container">
			<h3>Thank You</h3>
			<p>We appreciate you taking the time to review our services. We are always looking for ways to improve our business and provide the best customer satisfaction possible.</p>
		</div>
		<?php
	}
}
