<?php
/**
 * Builds forms from preexisting templates when spun-up on a site.
 *
 * @package StrategyPlugin
 */

namespace StrategyPlugin;

use GFAPI;

/**
 * Builds forms from preexisting templates when spun-up on a site.
 */
class ReviewFormBuilder extends \StrategyPlugin\Module {

	/**
	 * Used to alter the order in which clases are initialized.
	 *
	 * Lower number will be initialized first.
	 *
	 * @note This has no correlation to the `init` priority. It's just a way to allow certain classes to be initialized before others.
	 *
	 * @var int The priority of the module.
	 */
	public $load_order = 10;

	private const FORM_TEMPLATES_DIR = STRATEGY_REVIEWS_PATH . '/forms/';

	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return is_plugin_active( 'gravityforms/gravityforms.php' );
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'create_form_builder' ] );
	}

	/**
	 * Public factory method for instantiating the Form_Builder class. The class is
	 * dependent on Gravity Forms, so it will return null if the plugin is not active.
	 *
	 * @return void
	 */
	public function create_form_builder() {
		$this->validate_template_dir();

		$templates = $this->get_all_templates();

		if ( is_array( $templates ) && 0 < count( $templates ) ) {
			$this->build_forms( $templates );
		}
	}

	/**
	 * Triggers the build process for the form templates if their associated forms
	 * don't already exist.
	 *
	 * @param array $form_templates      An array of form templates
	 */
	private function build_forms( $form_templates ) {

		if ( ! is_array( $form_templates ) ) {
			return;
		}

		foreach ( $form_templates as $template ) {
			if ( ! $this->form_exists( $template ) ) {
				$this->create_form_from_template( $template );
			}
		}
	}

	/**
	 * Retrieves all templates that are in the templates directory.
	 *
	 * @return array                    Returns an array of form templates
	 */
	private function get_all_templates() {

		$template_files = glob( self::FORM_TEMPLATES_DIR . '*.json' );
		$templates = array();

		foreach ( $template_files as $file ) {
			$template = $this->retrieve_form_template( basename( $file ) );
			if ( ! is_null( $template ) ) {
				$templates[] = $template;
			}
		}

		return $templates;
	}

	/**
	 * Retrieves a form template from the templates directory.
	 *
	 * @param string $filename          The filename of the template to be retrieved
	 *
	 * @return Object                   Returns the template form object or null if not found
	 */
	private function retrieve_form_template( $filename ) {

		if ( file_exists( self::FORM_TEMPLATES_DIR . $filename ) ) {
			$contents = file_get_contents( self::FORM_TEMPLATES_DIR . $filename );
			$decoded = json_decode( $contents );

			return $decoded->{0};
		}

		return null;
	}

	/**
	 * Makes sure that the template is formatted correctly, and then adds the form to the site via GFAPI.
	 *
	 * @param Object|array $template    The template of the form being created
	 */
	private function create_form_from_template( $template ) {

		if ( ! is_array( $template ) ) {
			$template = $this->convert_to_array( $template );
		}

		GFAPI::add_form( $template );
	}

	/**
	 * Checks if a form currently exists in Gravity Forms based on some core descriptors of the form.
	 *
	 * @param Object $template          The template of the form being checked for
	 *
	 * @return bool                     Return true if the form already exists, otherwise false
	 */
	private function form_exists( $template ) {

		$forms = GFAPI::get_forms( null );
		$match = false;

		foreach ( $forms as $index => $form ) {
			if ( $template->title == $form['title'] ) {
				$match = true;
				break;
			}
		}

		return $match;
	}

	/**
	 * Recursively converts a piece of data from an object to an array.
	 *
	 * @param mixed $data               The data being converted
	 *
	 * @return mixed                    Returns the modified data
	 */
	private function convert_to_array( $data ) {

		if ( ! is_array( $data ) && is_object( $data ) ) {
			$data = (array) $data;
		}

		foreach ( $data as $key => $value ) {

			if ( is_array( $value ) || is_object( $value ) ) {
				$data[ $key ] = $this->convert_to_array( $value );
			}
		}

		return $data;
	}

	/**
	 * Ensures that the template directory exists. Creates it if it is not found.
	 */
	private function validate_template_dir() {

		if ( ! file_exists( self::FORM_TEMPLATES_DIR ) || ! is_dir( self::FORM_TEMPLATES_DIR ) ) {
			mkdir( self::FORM_TEMPLATES_DIR );
		}
	}
}
