<?php
/**
 * Creates pages automatically.
 *
 * @package StrategyPlugin
 */

namespace StrategyPlugin;

 /**
  * Generates templated pages.
  */
class PageGenerator extends \StrategyPlugin\Module {

	/**
	 * Array for tracking whether or not each target page exists on the site.
	 *
	 * @var array $pages        Contains boolean values indicating the presence of pages
	 */
	private $pages = array(
		'review-us'   => false,
		'referrals'   => false,
	);

	/**
	 * Whether to use patterns for the pages
	 *
	 * @var boolean $use_patterns
	 */
	private $use_patterns = false;

	/**
	 * Used to alter the order in which clases are initialized.
	 *
	 * Lower number will be initialized first.
	 *
	 * @note This has no correlation to the `init` priority. It's just a way to allow certain classes to be initialized before others.
	 *
	 * @var int The priority of the module.
	 */
	public $load_order = 20;

	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		$theme = wp_get_theme();
		$this->use_patterns = 'leadgen' === $theme->stylesheet || 'hello-elementor-child' === $theme->stylesheet;

		add_action( 'admin_init', [ $this, 'generate_pages' ] );
	}

	/**
	 * Checks through the existing pages on the site for the target pages.
	 */
	private function validate_existing_pages() {

		$existing_pages = get_pages();
		foreach ( $existing_pages as $existing_page ) {
			switch ( $existing_page->post_name ) {
				case 'review-us':
					$this->pages['review-us'] = true;
					break;
				case 'referrals':
					$this->pages['referrals'] = true;
					break;
				default:
					break;
			}
		}
	}

	/**
	 * Generates a page with the given slug, title, and content.
	 *
	 * @param string $slug    The slug of the page.
	 * @param string $title   The title of the page.
	 * @param string $content The content of the page.
	 */
	private function generate_page( $slug, $title, $content ) {
		if ( ! $slug || ! $title || ! $content ) {
			return;
		}

		if ( ! $this->pages[ $slug ] ) {
			$page = array(
				'post_author'   => 1,
				'post_title'    => $title,
				'post_name'     => $slug,
				'post_status'   => 'publish',
				'post_content'  => $content,
				'post_type'     => 'page',
			);

			wp_insert_post( $page );
			$this->pages[ $slug ] = true;
		}
	}

	/**
	 * If target pages are not already present on the site, create them.
	 */
	public function generate_pages() {
		$theme = wp_get_theme();
		$this->validate_existing_pages();

		switch ( $theme->stylesheet ) {
			case 'leadgen':
				$this->generate_page( 'review-us', 'Review Us', '<!-- wp:pattern {"slug":"reviews/reviews-section"} /-->' );
				$this->generate_page( 'referrals', 'Referrals', '<!-- wp:pattern {"slug":"reviews/referrals-section"} /-->' );
				break;
			case 'hello-elementor-child':
				$this->generate_page( 'review-us', 'Review Us', '[review_us]' );
				$this->generate_page( 'referrals', 'Referrals', '[referrals]' );
				break;
			default:
				break;
		}
	}
}
