<?php
/**
 * Block Custom CSS Registry.
 *
 * Collects and outputs compiled, block scoped custom CSS for the frontend.
 * CSS is aggregated during block rendering and emitted once per request
 * to improve performance and avoid duplicated inline styles.
 *
 * @package StrategyBlocks
 */

namespace StrategyBlocks;

/**
 * Class BlockCustomCSS
 *
 * Collects and outputs compiled, block scoped custom CSS for the frontend.
 *
 * This class acts as a request scoped registry. Individual blocks add their
 * compiled CSS during render, and the combined CSS is output once per request
 * as a single <style> tag to improve frontend performance.
 *
 * Usage pattern:
 * - Call ::add() during block rendering to register compiled CSS
 * - Call ::output() once (typically in wp_head) to emit the final styles
 *
 * CSS entries are deduplicated using a hash to prevent duplicate output
 * when identical CSS is generated multiple times on a page.
 *
 * @package StrategyBlocks
 */
class BlockCustomCSS {


	/**
	 * Collected CSS rules for the current request.
	 *
	 * Stored as a hash keyed array to prevent duplicate CSS output.
	 *
	 * @var array<string, string>
	 */
	private static $css = [];

	/**
	 * Add compiled CSS to the registry.
	 *
	 * Empty or whitespace only CSS strings are ignored. CSS is deduplicated
	 * using an md5 hash of the content.
	 *
	 * @param string $css Compiled CSS string scoped to a block instance.
	 * @return void
	 */
	public static function add( string $css ) {
		if ( trim( $css ) ) {
			$hash = md5( $css );
			self::$css[ $hash ] = $css;
		}
	}

	/**
	 * Output all collected CSS as a single inline <style> tag.
	 *
	 * Intended to be called once per request, typically hooked into wp_head.
	 * If no CSS has been registered, this method outputs nothing.
	 *
	 * @return void
	 */
	public static function output() {
		if ( empty( self::$css ) ) {
			return;
		}

		echo '<style id="lg-block-custom-css">';
		echo wp_kses( implode( array_values( self::$css ) ), [] );
		echo '</style>';
	}
}
