<?php
/**
 * Polyline
 *
 * PHP Version 5.3
 *
 * A simple class to handle polyline-encoding for Google Maps
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @category  Mapping
 * @package   Polyline
 * @author    E. McConville <emcconville@emcconville.com>
 * @copyright 2009-2015 E. McConville
 * @license   http://www.gnu.org/licenses/lgpl.html LGPL v3
 * @version   GIT: $Id$
 * @link      https://github.com/emcconville/google-map-polyline-encoding-tool
 */

namespace StrategyBlocks;

/**
 * Polyline encoding & decoding class
 *
 * Convert list of points to encoded string following Google's Polyline
 * Algorithm.
 *
 * @category Mapping
 * @package  Polyline
 * @author   E. McConville <emcconville@emcconville.com>
 * @license  http://www.gnu.org/licenses/lgpl.html LGPL v3
 * @link     https://github.com/emcconville/google-map-polyline-encoding-tool
 */
class Polyline {

	/**
	 * Default precision level of 1e-5.
	 *
	 * Overwrite this property in extended class to adjust precision of numbers.
	 * !!!CAUTION!!!
	 * 1) Adjusting this value will not guarantee that third party
	 *    libraries will understand the change.
	 * 2) Float point arithmetic IS NOT real number arithmetic. PHP's internal
	 *    float precision may contribute to undesired rounding.
	 *
	 * @var int $precision
	 */
	protected static $precision = 5;

	/**
	 * Apply Google Polyline algorithm to list of points.
	 *
	 * @param array $points List of points to encode. Can be a list of tuples,
	 *                      or a flat, one-dimensional array.
	 *
	 * @return string encoded string
	 */
	final public static function encode( $points ) {
		$encoded = '';
		$last_lat = 0;
		$last_lng = 0;

		foreach ( $points as $point ) {
			$lat = $point['lat'];
			$lng = $point['lng'];

			$lat_diff = $lat - $last_lat;
			$lng_diff = $lng - $last_lng;

			$lat_diff_encoded = round( $lat_diff * 1e5 );
			$lng_diff_encoded = round( $lng_diff * 1e5 );

			$encoded .= self::encode_value( $lat_diff_encoded ) . self::encode_value( $lng_diff_encoded );

			$last_lat = $lat;
			$last_lng = $lng;
		}

		return $encoded;
	}

	/**
	 * Generates an encoded polyline string.
	 *
	 * @param int $value The value to encode
	 *
	 * @return string The encoded polyline string
	 */
	final public static function encode_value( $value ) {
		$encoded = '';
		$value <<= 1;
		if ( $value < 0 ) {
			$value = ~$value;
		}
		while ( $value >= 0x20 ) {
			$encoded .= chr( ( 0x20 | ( $value & 0x1F ) ) + 63 );
			$value >>= 5;
		}
		$encoded .= chr( $value + 63 );

		return $encoded;
	}

	/**
	 * Reverse Google Polyline algorithm on encoded string.
	 *
	 * @param string $string Encoded string to extract points from.
	 *
	 * @return array points
	 */
	final public static function decode( $string ) {
		$points = array();
		$index = 0;
		$i = 0;
		$previous = array( 0, 0 );
		$string_len = strlen( $string );
		while ( $i < $string_len ) {
			$shift = 0x00;
			$result = 0x00;
			do {
				$bit = ord( substr( $string, $i++ ) ) - 63;
				$result |= ( $bit & 0x1f ) << $shift;
				$shift += 5;
			} while ( $bit >= 0x20 );

			$diff = ( $result & 1 ) ? ~( $result >> 1 ) : ( $result >> 1 );
			$number = $previous[ $index % 2 ] + $diff;
			$previous[ $index % 2 ] = $number;
			$index++;
			$points[] = $number * 1 / pow( 10, static::$precision );
		}
		return $points;
	}

	/**
	 * Reduce multi-dimensional to single list
	 *
	 * @param array $array Subject array to flatten.
	 *
	 * @return array flattened
	 */
	final public static function flatten( $array ) {
		$flatten = array();
		array_walk_recursive(
			$array, // @codeCoverageIgnore
			function ( $current ) use ( &$flatten ) {
				$flatten[] = $current;
			}
		);
		return $flatten;
	}

	/**
	 * Concat list into pairs of points
	 *
	 * @param array $list One-dimensional array to segment into list of tuples.
	 *
	 * @return array pairs
	 */
	final public static function pair( $list ) {
		return is_array( $list ) ? array_chunk( $list, 2 ) : array();
	}
}
