import { __ } from '@wordpress/i18n';
import { useState, useEffect, useRef } from '@wordpress/element';
import { InspectorControls, BlockControls } from '@wordpress/block-editor';
import { ToolbarButton, ToolbarGroup } from "@wordpress/components";
import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import TinyMCEEditor from '../components/TinyMCEEditor';
import { getBlockStyleClasses, getNativeWPClasses, getVisibilityClasses } from '../components/blockStyleUtils';
import VisibilityControls from "../components/visibility/VisibilityControls";

const Edit = ({ attributes, setAttributes, clientId }) => {
	const {
		content = '',
		animationEntrance = '',
		animationDuration = '',
		animationDelay = ''
	} = attributes;

	const [isEditing, setIsEditing] = useState(false);
	const [isFocused, setIsFocused] = useState(false);
	const containerRef = useRef(null);
	const editorId = `leadgen-editor-${clientId}`;

	useEffect(() => {
		const handleClickOutside = (event) => {
			if (containerRef.current && !containerRef.current.contains(event.target)) {
				setIsFocused(false);
			}
		};

		document.addEventListener('mousedown', handleClickOutside);
		return () => {
			document.removeEventListener('mousedown', handleClickOutside);
		};
	}, []);

	const onClickText = () => {
		if (!isFocused) {
			setIsFocused(true);
		} else {
			setIsEditing(true);
		}
	}

	const onClickClose = () => {
		setIsEditing(false);
		setIsFocused(true);
	}

	const combinedClasses = [
		'lg-block',
		'lg-block__content',
		'body-font',
		getBlockStyleClasses(attributes),
		getNativeWPClasses(attributes),
		getVisibilityClasses(attributes),
	].filter(Boolean).join(' ');

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={combinedClasses}
		>
			<BlockControls>
				<ToolbarGroup>
					<ToolbarButton
						icon="edit"
						label={__('Edit', 'lead-gen')}
						onClick={() => setIsEditing(true)}
					/>
				</ToolbarGroup>
			</BlockControls>
			<InspectorControls>
				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>
			<div
				ref={containerRef}
				className={combinedClasses}
				dangerouslySetInnerHTML={{
					__html: content || '<p>Click to edit...</p>',
				}}
				onClick={onClickText}
			/>
			{isEditing && (
				<TinyMCEEditor
					id={editorId}
					value={content}
					onSave={(val) => setAttributes({ content: val })}
					onClose={onClickClose}
				/>
			)}
		</AnimatedBlockWrapper >
	);
};

export default Edit;
