import { useState } from "@wordpress/element";

export default function PricingListPreview({ attributes }) {
	const {
		columns,
		features,
		pricing,
		featureLabelPlacement,
	} = attributes;

	const [toggle, setToggle] = useState(false);

	const onChangeToggle = () => {
		setToggle(!toggle);
	}

	const formatPrice = (price, currencyFormat) => {
		if (!price) return '';

		const hasSeparator = price.toString().includes('.') || price.toString().includes(',');
		let normalized = parseFloat(price.toString().replace(',', '.'));
		if (isNaN(normalized)) normalized = 0;
		normalized = hasSeparator ? normalized.toFixed(2) : Math.trunc(normalized).toString();

		if (currencyFormat === 'us') {
			if (hasSeparator) {
				const parts = normalized.split('.');
				return (
					<>
						{parts[0]}
						<sup>{parts[1]}</sup>
					</>
				);
			}
			return normalized;
		} else {
			return hasSeparator ? normalized.replace('.', ',') : normalized;
		}
	};

	const getPrice = (col) => {
		if (pricing['yearly'] && (!pricing['monthly'] || toggle)) {
			return pricing['yearly'] ? col['priceYear'] : '';
		}

		return pricing['monthly'] ? col['price'] : '';
	}

	const formatPeriod = () => {
		if (pricing['yearly'] && (!pricing['monthly'] || toggle)) {
			return pricing['yearly'] ? pricing['yearPeriod'] : '';
		}

		return pricing['monthly'] ? pricing['monthPeriod'] : '';
	}

	return (
		<div className="pricing-list">
			{pricing['yearly'] && pricing['monthly'] && (
				<div className="pricing-toggle">
					<fieldset className="pricing-toggle__container">
						<legend className="screen-reader-text">
							Billing period
						</legend>
						<label>
							<input type="radio" id="yearly" name="period" value="yearly" checked={toggle} onChange={onChangeToggle} />
							<span>Yearly</span>
						</label>
						<label>
							<input type="radio" id="monthly" name="period" value="monthly" checked={!toggle} onChange={onChangeToggle} />
							<span>Monthly</span>
						</label>
						<div className="pricing-toggle__background" aria-hidden="true"></div>
					</fieldset>
				</div>
			)}

			<div className="pricing-list__wrapper">
				{featureLabelPlacement === 'sidebar' && (
					<div className="pricing-list__sidebar">
						<div className="pricing-list__cell"></div>
						{features.map((feature, fi) => (
							<div key={fi} className="pricing-list__cell">
								{feature.title}
							</div>
						))}
					</div>
				)}

				<div className={`pricing-list__body${featureLabelPlacement === 'inline' ? ' pricing-list__body--full' : ''}`}>
					<div className="pricing-list__slider">
						<div className="swiper-wrapper">
							{columns.map((col, ci) => (
								<div key={ci} className={`pricing-list__item${col['best'] ? ' pricing-list__item--best' : ''}`}>
									<div className="pricing-list__cell">
										<h3 className="pricing-list__title">{col.title}</h3>
										{pricing['yearly'] && pricing['monthly'] && (
											<div className="pricing-list__price">
												{pricing.currency && <span className="pricing-list__price-currency">{pricing.currency === 'custom' ? pricing.customCurrency : pricing.currency}</span>}
												<span className="pricing-list__price-value">{formatPrice(String(getPrice(col) || 0), pricing.currencyFormat)}</span>
												<span className="pricing-list__price-period">{formatPeriod()}</span>
											</div>
										)}
									</div>

									{features.map((feature, fi) => {
										const item = feature.items[ ci ] ?? {};
										return (
											<div key={fi} className="pricing-list__cell pricing-list__result">
												{ item?.icon && (
													<span className="icon-list__icon">
														{item?.included ? '✓' : '✗'}
													</span>
												)}
												{featureLabelPlacement === 'inline' && (
													<span className="pricing-list__feature-label">
														{feature.title}
													</span>
												)}
												{item?.value || ''}
											</div>
										)
									})}

									<div className="pricing-list__cell pricing-list__cell--button">
										<a href={col.buttonLink} className="btn btn-primary">
											{col.buttonText}
										</a>
									</div>
								</div>
							))}
						</div>
					</div>
				</div>
			</div>
		</div>
	)
};
