import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { InspectorControls } from '@wordpress/block-editor';
import {PanelBody, TextControl, Button, ToggleControl, SelectControl} from '@wordpress/components';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import PricingListPreview from "./pricing-list-preview";
import VisibilityControls from "../components/visibility/VisibilityControls";
import AnimationControls from "../components/animations/AnimationControl";

const Edit = ({ attributes, setAttributes }) => {
	const {
		columns,
		features,
		pricing,
		featureLabelPlacement,
		animationEntrance,
		animationDuration,
		animationDelay,
	} = attributes;

	const [featuresOpen, setFeaturesOpen] = useState([]);

	useEffect(() => {
		setFeaturesOpen(prev => features.map((f, i) => prev[i] ?? true));
	}, [features.length]);

	const addColumn = () => {
		const newColumns = [...columns, { title: 'New Plan', buttonText: 'Buy Now', buttonLink: '#', price: '0', priceYear: '0', best: false }];
		const newFeatures = features.map(f => ({
			...f,
			items: [...f.items, { value: '', icon: false }]
		}));
		setAttributes({ columns: newColumns, features: newFeatures });
	};

	const updateColumn = (index, key, value) => {
		let updated = [...columns];

		if (key === 'best' && value === true) {
			updated = updated.map((col, i) => ({ ...col, best: i === index, }));
		} else {
			updated[index][key] = value;
		}

		setAttributes({ columns: updated });
	};

	const removeColumn = (index) => {
		const updatedCols = [...columns];
		updatedCols.splice(index, 1);

		const updatedFeatures = features.map(f => {
			const items = [...f.items];
			items.splice(index, 1);
			return { ...f, items };
		});

		setAttributes({ columns: updatedCols, features: updatedFeatures });
	};

	const addFeature = () => {
		setAttributes({
			features: [
				...features,
				{ title: 'New Feature', items: columns.map(() => ({ value: '', icon: false })) }
			]
		});
	};

	const updateFeatureTitle = (fIndex, value) => {
		const updated = [...features];
		updated[fIndex].title = value;
		setAttributes({ features: updated });
	};

	const updateFeatureValue = (fIndex, cIndex, key, value) => {
		const updated = [...features];
		updated[fIndex].items[cIndex][key] = value;
		setAttributes({ features: updated });
	};

	const removeFeature = (fIndex) => {
		const updated = [...features];
		updated.splice(fIndex, 1);
		setAttributes({ features: updated });
	};

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className="lg-block lg-block__pricing-list"
		>
			<InspectorControls>
				<PanelBody title={__('Columns', 'lead-gen')} initialOpen>
					{columns.map((col, i) => (
						<div key={i} style={{ border: '1px solid #eee', padding: 8, marginBottom: 8 }}>
							<TextControl
								label={__('Title', 'lead-gen')}
								value={col.title}
								onChange={val => updateColumn(i, 'title', val)}
							/>
							{pricing.monthly && (
								<TextControl
									label={__('Price Monthly', 'lead-gen')}
									value={col.price}
									onChange={val => updateColumn(i, 'price', val)}
								/>
							)}
							{pricing.yearly && (
								<TextControl
									label={__('Price Yearly', 'lead-gen')}
									value={col.priceYear}
									onChange={val => updateColumn(i, 'priceYear', val)}
								/>
							)}
							<TextControl
								label={__('Button Text', 'lead-gen')}
								value={col.buttonText}
								onChange={val => updateColumn(i, 'buttonText', val)}
							/>
							<TextControl
								label={__('Button Link', 'lead-gen')}
								value={col.buttonLink}
								onChange={val => updateColumn(i, 'buttonLink', val)}
							/>
							<ToggleControl
								label={__('Best Offer', 'lead-gen')}
								checked={col.best || false}
								onChange={(val) => updateColumn(i, 'best', val)}
							/>
							<Button isDestructive onClick={() => removeColumn(i)} style={{ padding: 0 }}>Remove Column</Button>
						</div>
					))}
					<Button isPrimary onClick={addColumn}>Add Column</Button>
				</PanelBody>

				<PanelBody title={__('Features', 'lead-gen')} initialOpen>
					<SelectControl
						label={__('Feature Label Placement', 'lead-gen')}
						value={featureLabelPlacement}
						options={[
							{ label: __('Sidebar', 'lead-gen'), value: 'sidebar' },
							{ label: __('Inline with icon', 'lead-gen'), value: 'inline' },
						]}
						onChange={(val) => setAttributes({ featureLabelPlacement: val })}
					/>

					{features.map((f, fi) => (
						<div key={fi} style={{ border: '1px solid #eee', padding: 8, marginBottom: 8, position: 'relative' }}>
							<Button
								isSecondary
								isSmall
								isLink
								onClick={() =>
									setFeaturesOpen(prev => {
										const updated = [...prev];
										updated[fi] = !updated[fi];
										return updated;
									})
								}
								style={{ position: 'absolute', top: 0, right: 0 }}
							>
								{featuresOpen[fi] ? __('Collapse', 'lead-gen') : __('Expand', 'lead-gen')}
							</Button>
							<TextControl
								label={__('Feature Title', 'lead-gen')}
								value={f.title}
								onChange={val => updateFeatureTitle(fi, val)}
							/>

							{featuresOpen[fi] && (
								<>
									{columns.map((col, ci) => (
										<div key={ci} style={{ border: '1px solid #eee', padding: 8, marginBottom: 8 }}>
											<p>Value for {col.title || `Column #${fi}`}</p>
											<ToggleControl
												label={__('Icon', 'lead-gen')}
												checked={f.items[ci].icon || false}
												onChange={(val) => updateFeatureValue(fi, ci, 'icon', val)}
											/>
											{f.items[ci].icon && (
												<ToggleControl
													label={__('Included?', 'lead-gen')}
													checked={f.items[ci].included || false}
													onChange={(val) => updateFeatureValue(fi, ci, 'included', val)}
												/>
											)}
											<TextControl
												value={f.items[ci].value}
												onChange={(v) => updateFeatureValue(fi, ci, 'value', v)}
											/>
										</div>
									))}
									<Button isDestructive onClick={() => removeFeature(fi)} style={{ padding: 0 }}>Remove Feature</Button>
								</>
							)}
						</div>
					))}
					<Button isPrimary onClick={addFeature}>Add Feature</Button>
				</PanelBody>
				<PanelBody title={__('Currency Options', 'lead-gen')} initialOpen>
					<SelectControl
						label={ __('Currency Symbol', 'lead-gen') }
						value={ pricing.currency }
						options={[
							{ label: '$ Dollar', value: '$' },
							{ label: '€ Euro', value: '€' },
							{ label: '£ Pound', value: '£' },
							{ label: '¥ Yen', value: '¥' },
							{ label: 'Custom', value: 'custom' },
						]}
						onChange={(val) => setAttributes({ pricing: { ...pricing, currency: val } })}
					/>

					{ pricing.currency === 'custom' && (
						<TextControl
							label={ __('Custom Symbol', 'lead-gen') }
							value={ pricing.customCurrency || '' }
							onChange={(val) =>
								setAttributes({ pricing: { ...pricing, customCurrency: val } })
							}
						/>
					) }
					<SelectControl
						label={ __('Currency Format', 'lead-gen') }
						value={ pricing.currencyFormat }
						options={[
							{ label: '1,234.56 (Default)', value: 'us' },
							{ label: '1.234,56', value: 'eu' },
						]}
						onChange={(val) =>
							setAttributes({ pricing: { ...pricing, currencyFormat: val } })
						}
					/>
					<ToggleControl
						label={ __('Monthly Pricing', 'lead-gen') }
						checked={ pricing.monthly || false }
						onChange={(val) =>
							setAttributes({ pricing: { ...pricing, monthly: val } })
						}
					/>
					{pricing.monthly && (
						<TextControl
							label={ __('Monthly Period', 'lead-gen') }
							value={ pricing.monthPeriod }
							onChange={ (val) =>
								setAttributes({ pricing: { ...pricing, monthPeriod: val } })
							}
						/>
					)}

					<ToggleControl
						label={ __('Yearly Pricing', 'lead-gen') }
						checked={ pricing.yearly || false }
						onChange={(val) =>
							setAttributes({ pricing: { ...pricing, yearly: val } })
						}
					/>
					{pricing.yearly && (
						<TextControl
							label={ __('Yearly Period', 'lead-gen') }
							value={ pricing.yearPeriod }
							onChange={ (val) =>
								setAttributes({ pricing: { ...pricing, yearPeriod: val } })
							}
						/>
					)}
				</PanelBody>

				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			<PricingListPreview attributes={attributes} />
		</AnimatedBlockWrapper>
	);
};

export default Edit;
