import { __ } from '@wordpress/i18n';
import { useSelect } from "@wordpress/data";
import { InspectorControls, MediaUpload, MediaUploadCheck } from '@wordpress/block-editor';
import { PanelBody, TextControl, SelectControl, ToggleControl, Button, BaseControl } from '@wordpress/components';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import VisibilityControls from "../components/visibility/VisibilityControls";
import AnimationControls from '../components/animations/AnimationControl';
import { Link } from "@10up/block-components";

const HTML_TAGS = [
	{ label: 'H2', value: 'h2' },
	{ label: 'H3', value: 'h3' },
	{ label: 'H4', value: 'h4' },
	{ label: 'H5', value: 'h5' },
	{ label: 'H6', value: 'h6' },
];

const Edit = ({ attributes, setAttributes }) => {
	const {
		header,
		pricing,
		featureIcons,
		features,
		footer,
		ribbon,
		animationEntrance,
		animationDuration,
		animationDelay
	} = attributes;

	const image = useSelect(
		(select) => (header.imageId ? select('core').getMedia(header.imageId) : null),
		[header.imageId]
	);
	const imageUrl = image?.source_url;

	const addFeature = () => {
		setAttributes({ features: [...features, { text: '', included: false }] });
	};

	const updateFeature = (index, key, value) => {
		const updated = [...features];
		updated[index][key] = value;
		setAttributes({ features: updated });
	};

	const removeFeature = (index) => {
		const updated = [...features];
		updated.splice(index, 1);
		setAttributes({ features: updated });
	};

	const formatPrice = (price, currencyFormat) => {
		if (!price) return '';

		const hasSeparator = price.toString().includes('.') || price.toString().includes(',');
		let normalized = parseFloat(price.toString().replace(',', '.'));
		if (isNaN(normalized)) normalized = 0;
		normalized = hasSeparator ? normalized.toFixed(2) : Math.trunc(normalized).toString();

		if (currencyFormat === 'us') {
			if (hasSeparator) {
				const parts = normalized.split('.');
				return (
					<>
						{parts[0]}
						<sup>{parts[1]}</sup>
					</>
				);
			}
			return normalized;
		} else {
			return hasSeparator ? normalized.replace('.', ',') : normalized;
		}
	};

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className="lg-block lg-block__pricing"
		>
			<InspectorControls>
				<PanelBody title={__('Header Options', 'lead-gen')} initialOpen>
					<BaseControl label={__('Image', 'lead-gen')}>
						<MediaUploadCheck>
							<MediaUpload
								onSelect={(media) => setAttributes({ header: { ...header, imageId: media?.id || null } })}
								allowedTypes={['image']}
								value={header.imageId}
								render={({ open }) => (
									<div className="upload-image">
										<Button onClick={open} variant="secondary">
											{header.imageId ? __('Replace Image', 'lead-gen') : __('Select Image', 'lead-gen')}
										</Button>
										{header.imageId ? (
											<Button onClick={() => setAttributes({ header: { ...header, imageId: null }})} isDestructive variant="link">
												{__('Remove Image', 'lead-gen')}
											</Button>
										) : ''}
									</div>
								)}
							/>
						</MediaUploadCheck>
					</BaseControl>
					<TextControl
						label={__('Title', 'lead-gen')}
						value={header.title}
						onChange={(val) => setAttributes({ header: { ...header, title: val } })}
					/>
					<TextControl
						label={__('Description', 'lead-gen')}
						value={header.description}
						onChange={(val) => setAttributes({ header: { ...header, description: val } })}
					/>
					<SelectControl
						label={ __('Title HTML Tag', 'lead-gen') }
						value={ header.tag }
						options={HTML_TAGS}
						onChange={(val) => setAttributes({ header: { ...header, tag: val } })}
					/>
				</PanelBody>

				<PanelBody title={__('Pricing Options', 'lead-gen')} initialOpen>
					<SelectControl
						label={ __('Currency Symbol', 'lead-gen') }
						value={ pricing.currency }
						options={[
							{ label: '$ Dollar', value: '$' },
							{ label: '€ Euro', value: '€' },
							{ label: '£ Pound', value: '£' },
							{ label: '¥ Yen', value: '¥' },
							{ label: 'Custom', value: 'custom' },
						]}
						onChange={(val) => setAttributes({ pricing: { ...pricing, currency: val } })}
					/>

					{ pricing.currency === 'custom' && (
						<TextControl
							label={ __('Custom Symbol', 'lead-gen') }
							value={ pricing.customCurrency || '' }
							onChange={(val) =>
								setAttributes({ pricing: { ...pricing, customCurrency: val } })
							}
						/>
					) }
					<TextControl
						label={__('Price', 'lead-gen')}
						value={pricing.price}
						onChange={(val) => setAttributes({ pricing: { ...pricing, price: val } })}
					/>
					<SelectControl
						label={ __('Currency Format', 'lead-gen') }
						value={ pricing.currencyFormat }
						options={[
							{ label: '1,234.56 (Default)', value: 'us' },
							{ label: '1.234,56', value: 'eu' },
						]}
						onChange={(val) =>
							setAttributes({ pricing: { ...pricing, currencyFormat: val } })
						}
					/>
					<ToggleControl
						label={__('On Sale', 'lead-gen')}
						checked={pricing.sale}
						onChange={(val) => setAttributes({ pricing: { ...pricing, sale: val } })}
					/>

					{ pricing.sale && (
						<TextControl
							label={ __('Original Price', 'lead-gen') }
							value={ pricing.originalPrice }
							onChange={ (val) =>
								setAttributes({ pricing: { ...pricing, originalPrice: val } })
							}
						/>
					) }
					<TextControl
						label={ __('Period', 'lead-gen') }
						value={ pricing.period }
						onChange={ (val) =>
							setAttributes({ pricing: { ...pricing, period: val } })
						}
					/>
				</PanelBody>

				<PanelBody title={__('Features', 'lead-gen')} initialOpen>
					<ToggleControl
						label={__('Show Icons', 'lead-gen')}
						checked={featureIcons}
						onChange={(val) => setAttributes({ featureIcons: val })}
					/>
					{features.map((feature, index) => (
						<div key={index} style={{ border: '1px solid #e0e0e0', padding: '8px', marginBottom: '8px' }}>
							<TextControl
								label={__('Text', 'lead-gen')}
								value={feature.text}
								onChange={(val) => updateFeature(index, 'text', val)}
							/>
							{ featureIcons && (
								<ToggleControl
									label={__('Included?', 'lead-gen')}
									checked={feature.included}
									onChange={(val) => updateFeature(index, 'included', val)}
								/>
							)}
							<Button isDestructive onClick={() => removeFeature(index)} style={{ padding: 0, height: '20px' }}>
								{__('Remove Feature', 'lead-gen')}
							</Button>
						</div>
					))}
					<Button isPrimary onClick={addFeature}>{__('Add Feature', 'lead-gen')}</Button>
				</PanelBody>

				<PanelBody title={__('Footer Options', 'lead-gen')} initialOpen>
					<BaseControl label={"Button URL"}>
						<Link
							value={footer.buttonText}
							url={footer?.buttonLink || ''}
							opensInNewTab={footer.opensInNewTab}
							onTextChange={(val) => setAttributes({ footer: { ...footer, buttonText: val ?? footer.buttonText }})}
							onLinkChange={(val) => setAttributes({ footer: { ...footer, buttonLink: val?.url, buttonText: val?.title ?? footer.buttonText, opensInNewTab: val?.opensInNewTab } })}
							onLinkRemove={() => setAttributes({ footer: { ...footer, buttonLink: null } })}
							placeholder={ __('Enter URL', 'lead-gen') }
						/>
					</BaseControl>
					<TextControl
						label={__('Additional Info', 'lead-gen')}
						value={footer.additionalInfo}
						onChange={(val) => setAttributes({ footer: { ...footer, additionalInfo: val } })}
					/>
				</PanelBody>

				<PanelBody title={__('Ribbon Options', 'lead-gen')} initialOpen>
					<ToggleControl
						label={__('Show Ribbon', 'lead-gen')}
						checked={ribbon.show}
						onChange={(val) => setAttributes({ ribbon: { ...ribbon, show: val } })}
					/>
					{ribbon.show && (
						<>
							<TextControl
								label={__('Ribbon Title', 'lead-gen')}
								value={ribbon.title}
								onChange={(val) => setAttributes({ ribbon: { ...ribbon, title: val } })}
							/>
							<SelectControl
								label={__('Ribbon Position', 'lead-gen')}
								value={ribbon.position}
								options={[
									{ label: 'Left', value: 'left' },
									{ label: 'Right', value: 'right' }
								]}
								onChange={(val) => setAttributes({ ribbon: { ...ribbon, position: val } })}
							/>
						</>
					)}
				</PanelBody>

				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			{imageUrl && (
				<div className="pricing-block__image">
					<img src={imageUrl} alt={image?.alt_text || 'Price card image'} />
				</div>
			)}
			<div className="pricing-block__body">
				{ribbon.show && (
					<div className={`pricing-block__ribbon pricing-block__ribbon--${ribbon.position || 'left'}`}>
						<div className="pricing-block__ribbon-item">{ribbon.title}</div>
					</div>
				)}

				<div className="pricing-block__price">
					{pricing.sale && pricing.originalPrice && (
						<span className="pricing-block__sale">
							<del>
								{pricing.currency && <span className="pricing-block__currency">{pricing.currency === 'custom' ? pricing.customCurrency : pricing.currency}</span>}
								{formatPrice(String(pricing.originalPrice), pricing.currencyFormat)}
							</del>
						</span>
					)}

					{pricing.currency && <span className="pricing-block__currency">{pricing.currency === 'custom' ? pricing.customCurrency : pricing.currency}</span>}
					<span className="pricing-block__value" >{formatPrice(String(pricing.price), pricing.currencyFormat)}</span>
					{pricing.period && <span className="pricing-block__period">{pricing.period}</span>}
				</div>

				<div className="pricing-block__header">
					{header.title && React.createElement(
						header.tag || 'h3',
						{ className: 'pricing-block__title' },
						header.title
					)}
					{header.description && <p className="pricing-block__description">{header.description}</p>}
				</div>

				{features.length > 0 && (
					<ul className="pricing-block__features">
						{features.map((feature, index) => (
							<li
								key={index}
								className={`pricing-block__feature ${feature.included ? 'pricing-block__feature--included' : ''}`}
							>
								{featureIcons && (
									<span className="icon-list__icon">{feature.included ? '✓' : '✗'}</span>
								)}
								<span className="pricing-block__feature-text">{feature.text}</span>
							</li>
						))}
					</ul>
				)}

				{footer.additionalInfo && (
					<p className="pricing-block__info">{footer.additionalInfo}</p>
				)}
			</div>
			{footer.buttonText && footer.buttonLink && (
				<Link
					tagName="span"
					value={footer.buttonText}
					url={footer.buttonLink}
					opensInNewTab={footer?.opensInNewTab || false}
					onTextChange={(val) => setAttributes({ footer: { ...footer, buttonText: val } })}
					onLinkChange={(val) => setAttributes({ footer: { buttonLink: val?.url, buttonText: val?.title, opensInNewTab: val?.opensInNewTab } })}
					onLinkRemove={() => setAttributes({ footer: { ...footer, buttonLink: null, opensInNewTab: null } })}
					className="pricing-block__button"
					placeholder="Button Text"
				/>
			)}

		</AnimatedBlockWrapper>
	);
};

export default Edit;
