import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import { useSelect } from '@wordpress/data';
import {
	PanelBody,
	Button
} from '@wordpress/components';
import {
	InspectorControls,
	MediaUpload,
	MediaUploadCheck
} from '@wordpress/block-editor';
import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import VisibilityControls from "../components/visibility/VisibilityControls";
import { getBlockStyleClasses, getNativeWPClasses, getVisibilityClasses } from '../components/blockStyleUtils';

const Edit = ({ attributes, setAttributes }) => {
	const {
		beforeImage,
		afterImage,
		animationEntrance = '',
		animationDuration = '',
		animationDelay = ''
	} = attributes;

	const [reveal, setReveal] = useState(50);

	const combinedClasses = [
		'lg-block',
		'lg-block__image-comparison',
		'image-comparison',
		getBlockStyleClasses(attributes),
		getNativeWPClasses(attributes),
		getVisibilityClasses(attributes)
	].filter(Boolean).join(' ');

	const onSelectImage = (attribute) => (media) => {
		setAttributes({
			[attribute]: media.id
		});
	};

	const onRemoveImage = (attribute) => () => {
		setAttributes({
			[attribute]: null
		});
	};

	const getMediaData = (id) =>
		useSelect(
			(select) => {
				if (!id) return null;
				const media = select('core').getMedia(id);
				if (!media) return null;

				return {
					url: media.source_url,
					alt: media.alt_text || ''
				};
			},
			[id]
		);

	const beforeMedia = getMediaData(beforeImage);
	const afterMedia = getMediaData(afterImage);

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={combinedClasses}
			tagName="figure"
			style={{ '--reveal': `${reveal}%` }}
		>
			<InspectorControls>
				<PanelBody title={__('Image Settings', 'lead-gen')} initialOpen={true}>
					<MediaUploadCheck>
						<MediaUpload
							onSelect={onSelectImage('beforeImage')}
							allowedTypes={['image']}
							value={beforeImage}
							render={({ open }) => (
								<Button onClick={open} isPrimary>
									{beforeImage ? __('Replace Before Image', 'lead-gen') : __('Select or Upload Before Image', 'lead-gen')}
								</Button>
							)}
						/>
						{beforeImage && (
							<Button onClick={onRemoveImage("beforeImage")} isDestructive style={{ marginTop: '10px' }}>
								{__('Remove Before Image', 'lead-gen')}
							</Button>
						)}
						<hr />
					</MediaUploadCheck>
					<MediaUploadCheck>
						<MediaUpload
							onSelect={onSelectImage("afterImage")}
							allowedTypes={['image']}
							value={afterImage}
							render={({ open }) => (
								<Button onClick={open} isPrimary>
									{afterImage ? __('Replace After Image', 'lead-gen') : __('Select or Upload After Image', 'lead-gen')}
								</Button>
							)}
						/>
						{afterImage && (
							<Button onClick={onRemoveImage("afterImage")} isDestructive style={{ marginTop: '10px' }}>
								{__('Remove After Image', 'lead-gen')}
							</Button>
						)}
					</MediaUploadCheck>
				</PanelBody>
				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			<div className="image-comparison__images">
				{beforeMedia ? (
					<img
						className="image-comparison__image image-comparison__image--before"
						src={beforeMedia.url}
						alt={beforeMedia.alt}
					/>
				) : (
					<img
						className="image-comparison__image image-comparison__image--before"
						src={`https://placehold.co/1200x600/ffffff/000000/png`}
						alt="Placeholder"
					/>
				)}

				{afterMedia ? (
					<img
						className="image-comparison__image image-comparison__image--after"
						src={afterMedia.url}
						alt={afterMedia.alt}
					/>
				) : (
					<img
						className="image-comparison__image image-comparison__image--after"
						src={`https://placehold.co/1200x600/000000/ffffff/png`}
						alt="Placeholder"
					/>
				)}
			</div>

			<div className="image-comparison__handle btn btn-primary" aria-hidden="true">
				<span>Before</span>
				<span className="image-comparison__handle-icon">← →</span>
				<span>After</span>
			</div>

			<figcaption className="image-comparison__controls">
				<label htmlFor="compare-range" className="sr-only">
					{__('Image comparison slider', 'lead-gen')}
				</label>
				<input
					id="compare-range"
					className="image-comparison__range"
					type="range"
					min="0"
					max="100"
					value={reveal}
					onChange={(e) => setReveal(e.target.value)}
					aria-valuemin="0"
					aria-valuemax="100"
					aria-valuenow="50"
					aria-label="Reveal after image"
				/>
			</figcaption>
		</AnimatedBlockWrapper>
	);
};

export default Edit;
