<?php
/**
 * Google Map Block
 *
 * @package StrategyBlocks
 *
 * @param array  $attributes  The block's attributes.
 * @param string $content     The block's inner content (unused here).
 * @param object $block       The block object (contains anchor, etc.).
 */

use StrategyBlocks\Utility;

$id                 = $attributes['anchor'] ?? '';
$alignment          = $attributes['alignment'];
$size               = $attributes['size'];
$image_id           = $attributes['staticImage'] ?? 0;
$markers            = $attributes['markers'];
$set_zoom           = $attributes['overrideZoom'];
$zoom               = $attributes['zoom'] ?? 16;
$lazy_load          = ( 'full' === $size && ! $image_id ) ? false : $attributes['lazyLoad'];
$primary_color      = $attributes['primaryColor'];
$secondary_color    = $attributes['secondaryColor'];
$pin_color          = $attributes['pinColor'];
$design_settings    = get_field( 'design_settings', 'options' );
if ( ! $design_settings ) {
	$design_settings = [
		'colors' => [
			'primary'   => '#000',
			'secondary' => '#000',
		],
	];
}

$normalize = static fn( $color ) => str_replace( '#', '0x', strtoupper( $color ) );

$hex_colors = [
	'primary'   => $normalize( $design_settings['colors'][ str_replace( '-', '_', $primary_color ) ] ?? '#000' ),
	'secondary' => $normalize( $design_settings['colors'][ str_replace( '-', '_', $secondary_color ) ] ?? '#000' ),
];

$animations = Utility\get_animation_data( $attributes );
$class_name = implode(
	' ',
	array_filter(
		[
			'lg-block',
			'lg-block__content',
			'lg-block__gmap',
			'lg-block__gmap--align-' . $alignment,
			Utility\get_block_style_classes( $attributes ),
			Utility\get_native_wp_classes( $attributes ),
			Utility\get_visibility_classes( $attributes ),
		]
	)
);

$map_class  = 'lg-map lg-map-size-' . $size;
$alt_text   = $image_id ? get_post_meta( $image_id, '_wp_attachment_image_alt', true ) : 'Placeholder - Please Replace Me';
$width      = null;
$height     = null;
$font_size  = 17;
// TODO: ANY CHANGES TO SIZE STYLING MUST BE REFLECTED HERE
$sizes = [
	'xs' => [ 15, 10 ],
	'sm' => [ 21, 14 ],
	'md' => [ 27, 18 ],
	'lg' => [ 33, 22 ],
	'xl' => [ 39, 26 ],
];

if ( isset( $sizes[ $size ] ) ) {
	[ $w, $h ] = $sizes[ $size ];
	$width  = $font_size * $w;
	$height = $font_size * $h;
}

$params = [
	'width' => $width,
	'height' => $height,
	'zoom' => $zoom,
	'hex_colors' => $hex_colors,
	'markers' => $markers,
	'image' => $image_id,
	'font_size' => $font_size,
	'pin_color' => $pin_color,
];
$static_url = Utility\get_google_map_static_url( $params );

do_action( 'lg_block_google_map_before' );
?>
	<div
		<?php
		if ( $id ) {
			echo 'id="' . esc_attr( $id ) . '"';
		}
		?>
		class="<?php echo esc_attr( $class_name ); ?>"
		<?php
		if ( $animations ) {
			echo 'data-animation="' . esc_attr( $animations ) . '"';
		}
		?>
	>
		<img
			class="<?php echo esc_attr( $map_class ); ?> lg-map-preview"
			src="<?php echo esc_url( $static_url ); ?>"
			alt="<?php echo esc_attr( $alt_text ); ?>"
			data-lazy-load="<?php echo esc_attr( $lazy_load ? 'true' : 'false' ); ?>"
		/>
		<?php if ( ! empty( $markers ) ) : ?>
			<div
				class="<?php echo esc_attr( $map_class ); ?> lg-map-live"
				data-zoom="<?php echo esc_attr( $zoom ); ?>"
				data-primary-color="<?php echo esc_attr( $hex_colors['primary'] ); ?>"
				data-secondary-color="<?php echo esc_attr( $hex_colors['secondary'] ); ?>"
			>
				<?php
				foreach ( $markers as $marker ) :
					$title = $marker['title'] ?? '';
					$description = $marker['description'] ?? '';
					$area = filter_var( $marker['area'], FILTER_VALIDATE_BOOLEAN );
					$radius = $area ? (int) $marker['radius'] : 0;
					$show = filter_var( $marker['show'], FILTER_VALIDATE_BOOLEAN );
					$show_marker = $area ? $show : true;
					$pin = $pin_color;
					if ( ! $area ) {
						if ( 'lightblue' === $pin ) {
							$pin = 'ltblue';
						}
						$pin .= '-dot';
					}
					?>
					<div
						class="marker"
						data-lat="<?php echo esc_attr( $marker['lat'] ); ?>"
						data-lng="<?php echo esc_attr( $marker['lng'] ); ?>"
						data-radius="<?php echo esc_attr( $radius ); ?>"
						data-pin="<?php echo esc_attr( $pin ); ?>"
						data-show-marker=<?php echo esc_attr( $show_marker ? 'true' : 'false' ); ?>
					>
						<h3><?php echo esc_html( $marker['title'] ); ?></h3>
						<p><em><?php echo esc_html( $marker['address'] ); ?></em></p>
						<?php if ( isset( $marker['description'] ) ) : ?>
							<p><?php echo wp_kses_post( $marker['description'] ); ?></p>
						<?php endif; ?>
					</div>
				<?php endforeach; ?>
			</div>
		<?php endif; ?>
	</div>

<?php do_action( 'lg_block_google_map_after' ); ?>
