import { __ } from '@wordpress/i18n';
import { useEffect, useRef } from '@wordpress/element';
import { InspectorControls } from '@wordpress/block-editor';
import { PanelBody, TextControl, ToggleControl, SelectControl, __experimentalNumberControl as NumberControl } from '@wordpress/components';
import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import { CountUp } from 'countup.js';
import VisibilityControls from "../components/visibility/VisibilityControls";
import {getVisibilityClasses} from "../components/blockStyleUtils";

const Edit = ({ attributes, setAttributes }) => {
	const {
		startNumber,
		endNumber,
		prefix,
		suffix,
		duration,
		thousandSeparator,
		separator,
		title,
		titleTag,
		animationEntrance,
		animationDuration,
		animationDelay,
	} = attributes;

	const counterRef = useRef(null);

	useEffect(() => {
		if (counterRef.current) {
			const options = {
				startVal: startNumber,
				duration: duration / 1000,
				useGrouping: thousandSeparator,
				separator,
			};
			const counter = new CountUp(counterRef.current, endNumber, options);
			if (!counter.error) counter.start();
		}
	}, [startNumber, endNumber, prefix, suffix, duration, thousandSeparator, separator]);

	const headingTags = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6'];
	const headingClass = headingTags.includes(titleTag) ? titleTag : '';

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={`lg-block lg-block__counter ${headingClass} ${getVisibilityClasses(attributes)}`}
		>
			<InspectorControls>
				<PanelBody title={__('Counter Settings', 'lead-gen')} initialOpen>
					<NumberControl
						label={__('Start Number', 'lead-gen')}
						__next40pxDefaultSize
						isShiftStepEnabled={true}
						onChange={(val) => setAttributes({ startNumber: val })}
						shiftStep={1}
						value={startNumber}
					/>
					<NumberControl
						label={__('End Number', 'lead-gen')}
						__next40pxDefaultSize
						isShiftStepEnabled={true}
						onChange={(val) => setAttributes({ endNumber: val })}
						shiftStep={1}
						value={endNumber}
					/>
					<TextControl
						label={__('Number Prefix', 'lead-gen')}
						value={prefix}
						onChange={(val) => setAttributes({ prefix: val })}
					/>
					<TextControl
						label={__('Number Suffix', 'lead-gen')}
						value={suffix}
						onChange={(val) => setAttributes({ suffix: val })}
					/>
					<TextControl
						label={__('Animation Duration (ms)', 'lead-gen')}
						value={duration}
						type="number"
						onChange={(val) => setAttributes({ duration: parseInt(val, 10) })}
					/>
					<ToggleControl
						label={__('Use Thousand Separator', 'lead-gen')}
						checked={thousandSeparator}
						onChange={(val) => setAttributes({ thousandSeparator: val })}
					/>
					{thousandSeparator && (
						<SelectControl
							label={__('Separator', 'lead-gen')}
							value={separator}
							options={[
								{ label: ',', value: ',' },
								{ label: '.', value: '.' },
								{ label: 'space', value: ' ' },
								{ label: "'", value: "'" }
							]}
							onChange={(val) => setAttributes({ separator: val })}
						/>
					)}
					<TextControl
						label={__('Title', 'lead-gen')}
						value={title}
						onChange={(val) => setAttributes({ title: val })}
					/>
					<SelectControl
						label={__('Title HTML Tag', 'lead-gen')}
						value={titleTag}
						options={[
							{ label: 'H1', value: 'h1' },
							{ label: 'H2', value: 'h2' },
							{ label: 'H3', value: 'h3' },
							{ label: 'H4', value: 'h4' },
							{ label: 'H5', value: 'h5' },
							{ label: 'H6', value: 'h6' },
							{ label: 'div', value: 'div' },
							{ label: 'span', value: 'span' },
							{ label: 'p', value: 'p' },
						]}
						onChange={(val) => setAttributes({ titleTag: val })}
					/>
				</PanelBody>
				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>
			{title && React.createElement(titleTag, { className: 'counter__title' }, title)}
			<div className="counter__number-wrapper">
				<span className="counter__number-prefix">{prefix}</span>
				<span ref={counterRef} className="counter__number"></span>
				<span className="counter__number-suffix">{suffix}</span>
			</div>
		</AnimatedBlockWrapper>
	);
};

export default Edit;
