import { PanelBody, SelectControl, ToggleControl } from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';

const VISIBILITY_OPTIONS = [
	{ label: __('Mobile and Up', 'lead-gen'), value: 'xs' },
	{ label: __('Only on Mobile', 'lead-gen'), value: 'xs-only' },
	{ label: __('Tablet and Up', 'lead-gen'), value: 'sm' },
	{ label: __('Only on Tablet', 'lead-gen'), value: 'sm-only' },
	{ label: __('Large Tablet and Up', 'lead-gen'), value: 'md' },
	{ label: __('Only on Large Tablet', 'lead-gen'), value: 'md-only' },
	{ label: __('Laptop and Up', 'lead-gen'), value: 'lg' },
	{ label: __('Only on Laptop', 'lead-gen'), value: 'lg-only' },
	{ label: __('Desktop and Up', 'lead-gen'), value: 'xl' },
];

const SIMPLE_TOGGLES = [
	{ label: __('Hide On Desktop', 'lead-gen'), value: 'xl' },
	{ label: __('Hide On Laptop', 'lead-gen'), value: 'lg-only' },
	{ label: __('Hide On Large Tablet', 'lead-gen'), value: 'md-only' },
	{ label: __('Hide On Tablet', 'lead-gen'), value: 'sm-only' },
	{ label: __('Hide On Mobile', 'lead-gen'), value: 'xs-only' },
];

export default function VisibilityControls({ attributes, setAttributes }) {
	const { visibilityHide = [], visibilityShow = [] } = attributes;
	const [advanced, setAdvanced] = useState(false);

	useEffect(() => {
		const simpleValues = SIMPLE_TOGGLES.map((t) => t.value);
		const hasAdvancedHide = visibilityHide.some((v) => !simpleValues.includes(v));

		if (hasAdvancedHide || visibilityShow.length > 0) {
			setAdvanced(true);
		}
	}, []);

	const handleSimpleToggle = (value, checked) => {
		let newHide = [...visibilityHide];

		if (checked) {
			if (!newHide.includes(value)) newHide.push(value);
		} else {
			newHide = newHide.filter((v) => v !== value);
		}

		setAttributes({ visibilityHide: newHide });
	};

	return (
		<PanelBody title={__('Visibility', 'lead-gen')} initialOpen={false}>
			{!advanced && (
				<>
					{SIMPLE_TOGGLES.map((toggle) => (
						<ToggleControl
							key={toggle.value}
							label={toggle.label}
							checked={visibilityHide.includes(toggle.value)}
							onChange={(checked) => handleSimpleToggle(toggle.value, checked)}
						/>
					))}
				</>
			)}

			{advanced && (
				<>
					<SelectControl
						label={__('Hide On', 'lead-gen')}
						multiple
						value={visibilityHide}
						options={VISIBILITY_OPTIONS}
						onChange={(value) => setAttributes({ visibilityHide: value })}
					/>
					<SelectControl
						label={__('Show On', 'lead-gen')}
						multiple
						value={visibilityShow}
						options={VISIBILITY_OPTIONS}
						onChange={(value) => setAttributes({ visibilityShow: value })}
					/>
				</>
			)}

			<ToggleControl
				label={__('Show Advanced Settings', 'lead-gen')}
				checked={advanced}
				onChange={() => {
					if (advanced) {
						const simpleValues = SIMPLE_TOGGLES.map((t) => t.value);
						const newHide = visibilityHide.filter((v) => simpleValues.includes(v));

						setAttributes({ visibilityHide: newHide, visibilityShow: [] });
					}
					setAdvanced(!advanced);
				}}
			/>
		</PanelBody>
	);
}
