import { useEffect } from '@wordpress/element';

export const parseSpacingValue = (val) => {
	if (typeof val !== 'string') return val;
	return val.includes('|') ? val.split('|').pop() : val;
};

export const getBlockStyleClasses = (attributes) => {
	const classes = [];
	const spacing = attributes?.style?.spacing || {};

	for (const prop of ['padding', 'margin', 'blockGap']) {
		if (spacing[prop]) {
			const shorthand = prop === 'blockGap' ? 'g' : prop[0];
			const value = spacing[prop];

			if (typeof value === 'string') {
				const parsed = parseSpacingValue(value);
				classes.push(`${shorthand}-${parsed}`);
			} else {
				for (const [side, val] of Object.entries(value)) {
					const parsed = parseSpacingValue(val);
					classes.push(`${shorthand}${side[0]}-${parsed}`);
				}
			}
		}
	}

	return classes.join(' ');
};

export const getNativeWPClasses = (attributes) => {
	const classes = [];

	if (attributes.className) {
		classes.push(attributes.className);
	}

	if (attributes.align) {
		classes.push(`align-${attributes.align}`);
	}

	const typography = attributes?.style?.typography || {};
	if (typography.textAlign) {
		classes.push(`text-${typography.textAlign}`);
	}

	if (attributes.fontSize) {
		classes.push(`size-${attributes.fontSize}`);
	}

	if (attributes.full_height) {
		classes.push('h-full');
	}

	if (attributes.textColor) {
		classes.push(`text-color-${attributes.textColor}`);
	}

	if (attributes.backgroundColor) {
		classes.push(`bg-color-${attributes.backgroundColor}`);
	}

	const linkColor = attributes?.style?.elements?.link?.color?.text;
	if (linkColor) {
		const parsed = linkColor.split('|').pop();
		classes.push(`link-color-${parsed}`);
	}

	const linkHover = attributes?.style?.elements?.link?.[':hover']?.color?.text;
	if (linkHover) {
		const parsed = linkHover.split('|').pop();
		classes.push(`link-hover-color-${parsed}`);
	}

	return classes.join(' ');
};

export const getVisibilityClasses = (attributes) => {
	const hideClasses = attributes.visibilityHide?.map((v) => `hide-${v}`) || [];
	const showClasses = attributes.visibilityShow?.map((v) => `show-${v}`) || [];

	return [...hideClasses, ...showClasses].join(' ');
};

export function useBlockId(attributes, setAttributes, clientId) {
	useEffect(() => {
		if (!clientId) return;

		const expected = clientId.slice(0, 8);

		// First render OR duplicated block
		if (!attributes.blockId || attributes.blockId !== expected) {
			setAttributes({ blockId: expected });
		}
	}, [attributes.blockId, clientId, setAttributes]);
}

export const getInheritedValue = (values, device) => {
	const DEVICES_ORDER = ['desktop', 'laptop', 'large-tablet', 'tablet', 'mobile' ];
	const index = DEVICES_ORDER.indexOf(device);

	if (index === -1) {
		return null;
	}

	for (let i = index; i < DEVICES_ORDER.length; i++) {
		const val = values?.[DEVICES_ORDER[i]];
		if (val !== undefined && val !== '') {
			return {
				slug: DEVICES_ORDER[i],
				val
			};
		}
	}
	return null;
};

export const showInherit = (values, activeDevice) => {
	if (!values[activeDevice]) {
		const inheritedValue = getInheritedValue(values, activeDevice);
		if (inheritedValue) {
			const deviceLabel = inheritedValue.slug.replace(/-/g, ' ').replace(/\b\w/g, (c) => c.toUpperCase());
			const val = Number.isFinite(Number(inheritedValue.val)) ? `${inheritedValue.val} / 12` : inheritedValue.val;
			return `Value inherited from ${deviceLabel}: ${val}`
		}
	}
	return null;
}
