/* global tinymce */
import { useEffect, useRef } from '@wordpress/element';
import { createPortal } from '@wordpress/element';
import { Modal, Button } from '@wordpress/components';

const TinyMCEEditor = ({ id, value, onSave, onClose }) => {
  const containerRef = useRef(null);
  const editorRef = useRef(null);
  const portalRoot = getEditorPortalRoot();

  useEffect(() => {
    const parentWin = window.parent || window;

    if (!parentWin.wp?.editor?.initialize) {
      console.error('wp.editor is not available');
      return;
    }

    const textarea = parentWin.document.createElement('textarea');
    textarea.id = id;
    textarea.value = value || '';
    containerRef.current.appendChild(textarea);

    parentWin.wp.editor.initialize(id, {
      tinymce: {
        wpautop: false,
        entity_encoding: 'raw',
        toolbar1:
          'formatselect bold italic underline bullist numlist blockquote alignleft aligncenter alignright link unlink | pastetext removeformat charmap outdent indent undo redo',
        setup: (editor) => {
          editorRef.current = editor;
          // Preserve non-breaking spaces through all serialization paths
          // (getContent, save, visual-to-text mode switch)
          editor.on('PostProcess', (e) => {
            e.content = e.content.replace(/\u00a0/g, '&nbsp;');
          });
        },
      },
      quicktags: true,
      mediaButtons: true,
    });

    return () => {
      if (editorRef.current) {
        parentWin.wp.editor.remove(id);
        editorRef.current = null;
      }
      if (containerRef.current) {
        containerRef.current.innerHTML = '';
      }
    };
  }, []);

	const handleDone = () => {
		const container = containerRef.current;
		if (!container) return;

		let content = '';
		const isVisualActive = container.querySelector(`.wp-editor-wrap.tmce-active`);

		if (isVisualActive && editorRef.current && editorRef.current.getContent) {
			content = editorRef.current.getContent();
		} else {
			const textarea = container.querySelector('textarea');
			if (textarea) {
				content = textarea.value;
			}
		}

		// Ensure non-breaking spaces survive as &nbsp; entities
		content = content.replace(/\u00a0/g, '&nbsp;');

		onSave(content);
		onClose();
	};

  return createPortal(
    <Modal
      title="Edit Content"
      onRequestClose={onClose}
      className="leadgen-tinymce-modal"
      shouldCloseOnClickOutside={false}
    >
      <div ref={containerRef} />
      <div style={{ marginTop: '1rem', textAlign: 'right' }}>
        <Button variant="primary" onClick={handleDone}>
          Done
        </Button>
      </div>
    </Modal>,
    portalRoot
  );
};

function getEditorPortalRoot() {
  const parentDoc = window.parent?.document || document;
  let root = parentDoc.getElementById('lg-editor-portal-root');

  if (!root) {
    root = parentDoc.createElement('div');
    root.id = 'lg-editor-portal-root';
    parentDoc.body.appendChild(root);
  }

  return root;
}

export default TinyMCEEditor;
