import { __ } from '@wordpress/i18n';
import { useEffect, useState } from '@wordpress/element';
import { InnerBlocks, InspectorControls } from '@wordpress/block-editor';

import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import { DeviceContext } from '../components/responsive/deviceContext';
import VisibilityControls from '../components/visibility/VisibilityControls';
import LegacyFlexAlignmentControls from '../components/alignment/LegacyFlexAlignmentControls';
import {getVisibilityClasses} from "../components/blockStyleUtils";

const BREAKPOINTS = ['xl', 'lg', 'md', 'sm', 'xs'];
const DEVICE_MAP = {
	xs: 'mobile',
	sm: 'tablet',
	md: 'large-tablet',
	lg: 'laptop',
	xl: 'desktop',
};

const buildResponsiveClasses = (attributes) => {
	let classes = '';
	let xsCascade = '';
	let valueBuffer = '';

	for (const bp of BREAKPOINTS) {
		const device = DEVICE_MAP[bp];
		const vertical = attributes.verticalAlign?.[device] || '';
		const horizontal = attributes.horizontalAlign?.[device] || '';
		const column = attributes.columnWidth?.[device] || '';

		const customized = vertical !== '' || horizontal !== '' || column !== '';

		if (customized) {
			classes += valueBuffer;
			valueBuffer = '';
			xsCascade = '';
		}

		if (vertical !== '') {
			valueBuffer += ` ${vertical}-${bp}`;
			xsCascade += ` ${vertical}-xs`;
		}
		if (horizontal !== '') {
			valueBuffer += ` ${horizontal}-${bp}`;
			xsCascade += ` ${horizontal}-xs`;
		}
		if (column !== '') {
			if (column === 'full' || column === 'auto') {
				valueBuffer += ` col-${column}-${bp}`;
				xsCascade += ` col-${column}-xs`;
			} else {
				valueBuffer += ` col-${bp}-${column}`;
				xsCascade += ` col-xs-${column}`;
			}
		}
	}

	classes += valueBuffer + xsCascade;
	classes += ' ' + getVisibilityClasses(attributes);

	return classes.trim();
};

const Edit = ({ attributes, setAttributes, clientId }) => {
	const {
		id,
		animationEntrance,
		animationDuration,
		animationDelay
	} = attributes;

	const [currentDevice, setCurrentDevice] = useState('desktop');

	useEffect(() => {
		if (!id && clientId) {
			setAttributes({ id: clientId });
		}
	}, [id, clientId]);

	return (
		<DeviceContext.Provider value={[currentDevice, setCurrentDevice]}>
			<AnimatedBlockWrapper
				animationEntrance={animationEntrance}
				animationDuration={animationDuration}
				animationDelay={animationDelay}
				attributes={attributes}
				className={`lg-block lg-block__col col ${buildResponsiveClasses(attributes)}`}
			>
				<InspectorControls>
					<LegacyFlexAlignmentControls attributes={attributes} setAttributes={setAttributes} />
					<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
					<AnimationControls attributes={attributes} setAttributes={setAttributes} />
				</InspectorControls>
				<InnerBlocks
					template={[['core/paragraph', {}]]}
					orientation="vertical"
				/>
			</AnimatedBlockWrapper>
		</DeviceContext.Provider>
	);
};

export default Edit;
