import { __ } from '@wordpress/i18n';
import {
	MediaUpload,
	MediaUploadCheck,
	InnerBlocks,
	InspectorControls
} from '@wordpress/block-editor';
import {
	Button,
	PanelBody,
	SelectControl,
	ToggleControl,
	BaseControl
} from '@wordpress/components';
import { useSelect } from '@wordpress/data';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import PlaceholderDimensionControls from '../components/PlaceholderDimension/PlaceholderDimensionControls';
import AnimationControls from '../components/animations/AnimationControl';
import VisibilityControls from "../components/visibility/VisibilityControls";
import {getVisibilityClasses} from "../components/blockStyleUtils";

const PLACEMENT_OPTIONS = [
	{ label: __('Top', 'lead-gen'), value: 'top' },
	{ label: __('Left', 'lead-gen'), value: 'left' },
	{ label: __('Right', 'lead-gen'), value: 'right' },
	{ label: __('Background', 'lead-gen'), value: 'background' },
	{ label: __('None', 'lead-gen'), value: 'none' },
];

const SIZE_OPTIONS = [
	{ label: __('Extra Small', 'lead-gen'), value: 'xs' },
	{ label: __('Small', 'lead-gen'), value: 'sm' },
	{ label: __('Medium', 'lead-gen'), value: 'md' },
	{ label: __('Large', 'lead-gen'), value: 'lg' },
	{ label: __('Extra Large', 'lead-gen'), value: 'xl' },
];

const BORDER_COLORS = [
	{ label: __('Primary', 'lead-gen'), value: 'primary' },
	{ label: __('Primary Tint 1', 'lead-gen'), value: 'primary-tint-1' },
	{ label: __('Primary Tint 2', 'lead-gen'), value: 'primary-tint-2' },
	{ label: __('Primary Shade 1', 'lead-gen'), value: 'primary-shade-1' },
	{ label: __('Primary Shade 2', 'lead-gen'), value: 'primary-shade-2' },
	{ label: __('Secondary', 'lead-gen'), value: 'secondary' },
	{ label: __('Secondary Tint 1', 'lead-gen'), value: 'secondary-tint-1' },
	{ label: __('Secondary Tint 2', 'lead-gen'), value: 'secondary-tint-2' },
	{ label: __('Secondary Shade 1', 'lead-gen'), value: 'secondary-shade-1' },
	{ label: __('Secondary Shade 2', 'lead-gen'), value: 'secondary-shade-2' },
	{ label: __('Accent 1', 'lead-gen'), value: 'accent-1' },
	{ label: __('Accent 1 Tint 1', 'lead-gen'), value: 'accent-1-tint-1' },
	{ label: __('Accent 1 Tint 2', 'lead-gen'), value: 'accent-1-tint-2' },
	{ label: __('Accent 1 Shade 1', 'lead-gen'), value: 'accent-1-shade-1' },
	{ label: __('Accent 1 Shade 2', 'lead-gen'), value: 'accent-1-shade-2' },
	{ label: __('Accent 2', 'lead-gen'), value: 'accent-2' },
	{ label: __('Accent 2 Tint 1', 'lead-gen'), value: 'accent-2-tint-1' },
	{ label: __('Accent 2 Tint 2', 'lead-gen'), value: 'accent-2-tint-2' },
	{ label: __('Accent 2 Shade 1', 'lead-gen'), value: 'accent-2-shade-1' },
	{ label: __('Accent 2 Shade 2', 'lead-gen'), value: 'accent-2-shade-2' },
];

const Edit = ({ attributes, setAttributes }) => {
	const {
		imageId,
		imagePlacement = 'top',
		imageSize = 'md',
		border,
		borderColor,
		borderRadius = 'none',
		placeholderDimensions,
		animationEntrance,
		animationDuration,
		animationDelay
	} = attributes;

	const width = placeholderDimensions?.width || 640;
	const height = placeholderDimensions?.height || 360;

	const image = useSelect(
		(select) => (imageId ? select('core').getMedia(imageId) : null),
		[imageId]
	);
	const imageUrl = image?.source_url;

	const classNames = [
		'lg-block lg-block__card card',
		imagePlacement === 'top' && 'card--vertical',
		['left', 'right'].includes(imagePlacement) && 'card--horizontal',
		imagePlacement === 'right' && 'reverse',
		imagePlacement === 'background' && 'card--vertical card--background',
		border && 'card--bordered',
		borderColor && `border-color-${borderColor}`,
		borderRadius !== 'none' && `rounded-${borderRadius}`,
		getVisibilityClasses(attributes),
	].filter(Boolean).join(' ');

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={classNames}
		>
			<InspectorControls>
				<PanelBody title={__('Card Settings', 'lead-gen')}>
					<SelectControl
						label={__('Image Placement', 'lead-gen')}
						value={imagePlacement}
						options={PLACEMENT_OPTIONS}
						onChange={(value) => setAttributes({ imagePlacement: value })}
					/>

					{imagePlacement !== 'none' && (
						<>
							<BaseControl label={__('Image', 'lead-gen')}>
								<MediaUploadCheck>
									<MediaUpload
										onSelect={(media) => setAttributes({ imageId: media?.id || null })}
										allowedTypes={['image']}
										value={imageId}
										render={({ open }) => (
											<div class="upload-image">
												<Button onClick={open} variant="secondary">
													{imageId ? __('Replace Image', 'lead-gen') : __('Select Image', 'lead-gen')}
												</Button>
												{imageId ? (
													<Button onClick={() => setAttributes({ imageId: null })} isDestructive variant="link">
														{__('Remove Image', 'lead-gen')}
													</Button>
												) : ''}
											</div>
										)}
									/>
								</MediaUploadCheck>
							</BaseControl>

							<SelectControl
								label={__('Image Size', 'lead-gen')}
								value={imageSize}
								options={SIZE_OPTIONS}
								onChange={(value) => setAttributes({ imageSize: value })}
							/>
						</>
					)}
				</PanelBody>

				{imagePlacement !== 'none' && !imageId ? (
					<PlaceholderDimensionControls attributes={attributes} setAttributes={setAttributes} />
				) : ''}

				<PanelBody title={__('Border Settings', 'lead-gen')}>
					<ToggleControl
						label={__('Border', 'lead-gen')}
						checked={border}
						onChange={(value) => setAttributes({ border: value })}
					/>
					{border && (
						<SelectControl
							label={__('Border Color', 'lead-gen')}
							value={borderColor || 'primary'}
							options={BORDER_COLORS}
							onChange={(value) => setAttributes({ borderColor: value })}
						/>
					)}
					<SelectControl
						label={__('Border Radius', 'lead-gen')}
						value={borderRadius}
						options={[{ label: __('None', 'lead-gen'), value: 'none' }, ...SIZE_OPTIONS]}
						onChange={(value) => setAttributes({ borderRadius: value })}
					/>
				</PanelBody>

				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			{imagePlacement !== 'none' && (
				<div className="card__media">
					<img
						src={imageUrl || `https://placehold.co/${width}x${height}`}
						alt={image?.alt_text || 'Card image'}
						className={`card__media-img media__image card__media-img--size-${imageSize}`}
					/>
				</div>
			)}

			<div className="card__content col">
				<InnerBlocks
					template={[['leadgen/text-editor']]}
				/>
			</div>
		</AnimatedBlockWrapper>
	);
};

export default Edit;
