import apiFetch from '@wordpress/api-fetch';
import { __ } from '@wordpress/i18n';
import { InspectorControls } from '@wordpress/block-editor';
import { useEffect, useRef, useState } from "@wordpress/element";
import {
	PanelBody,
	SelectControl,
	RangeControl,
	TextControl,
	FormTokenField, BaseControl, __experimentalNumberControl as NumberControl, ToggleControl,
} from '@wordpress/components';
import ServerSideRender from '@wordpress/server-side-render';
import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import VisibilityControls from "../components/visibility/VisibilityControls";
import { getVisibilityClasses } from "../components/blockStyleUtils";
import { ReorderableMultiSelect } from "../components/ReorderableMultiSelect";
import { COLORS, IMAGE_PLACEMENT_OPTIONS, SIZE_OPTIONS, TAGS } from "../components/constants";
import { initSwiper } from "../../../assets/js/slider";

const ELEMENT_OPTIONS = [
	{ key: 'title', label: __('Title', 'lead-gen') },
	{ key: 'name', label: __('Name', 'lead-gen') },
	{ key: 'content', label: __('Content', 'lead-gen') },
	{ key: 'socialLinks', label: __('Social Links', 'lead-gen') },
	{ key: 'categories', label: __('Categories', 'lead-gen') },
	{ key: 'read_more', label: __('Read More', 'lead-gen') },
];

const Edit = ({ attributes, setAttributes, name }) => {
	const {
		displayType,
		sliderControls,
		selector,
		elements,
		teamMembers = [],
		categories = [],
		size,
		border,
		borderColor,
		corners,
		perRow,
		btnText,
		imageEnabled,
		imagePlacement = 'background',
		imageSize = 'full',
		textAlign,
		titleTag,
		excerptLength,
		animationEntrance,
		animationDuration,
		animationDelay
	} = attributes;

	const [catOptions, setCatOptions] = useState([]);
	const [teamOptions, setTeamOptions] = useState([]);
	const wrapperRef = useRef(null);

	useEffect(() => {
		if (selector === 'category') {
			apiFetch({ path: '/wp/v2/team-member-categories' })
				.then((res) => {
					setCatOptions(res.map((cat) => ({ label: cat.name, value: String(cat.id) })));
				});
		}
		if (selector === 'name') {
			apiFetch({ path: '/wp/v2/team-members' })
				.then((res) => {
					setTeamOptions(res.map((post) => ({ label: post.title.rendered, value: String(post.id) })));
				});
		}
	}, [selector]);

	useEffect(() => {
		if (displayType !== 'slider') return;
		if (!wrapperRef.current) return;

		const observer = new MutationObserver(() => {
			const slider = wrapperRef.current.querySelector('.bio__slider');
			if (slider && !slider.dataset.initialized) {
				initSwiper(slider, { autoHeight: false, allowTouchMove: false });
				slider.dataset.initialized = '1';
			}
		});

		observer.observe(wrapperRef.current, {
			childList: true,
			subtree: true,
		});

		return () => observer.disconnect();
	}, [displayType]);

	const mapValuesToLabels = (options, values) =>
		Array.isArray(values) ? values.map(v => {
			const found = options.find(o => o.value === v);
			return found ? found.label : v;
		}) : [];

	const mapLabelsToValues = (options, labels) =>
		labels.map(l => {
			const found = options.find(o => o.label === l);
			return found ? found.value : l;
		});

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={`${getVisibilityClasses(attributes)}`}
		>
			<InspectorControls>
				<PanelBody title={__('Bio Settings', 'lead-gen')} initialOpen={true}>
					<SelectControl
						label={__('Display Type', 'lead-gen')}
						value={displayType}
						options={[
							{ label: __('Gallery', 'lead-gen'), value: 'gallery' },
							{ label: __('Grid', 'lead-gen'), value: 'grid' },
							{ label: __('Slider', 'lead-gen'), value: 'slider' },
						]}
						onChange={(val) => setAttributes({ displayType: val })}
					/>
					{'slider' === displayType && (
						<ToggleControl
							label={__('Slider Controls', 'lead-gen')}
							checked={!!sliderControls}
							onChange={(val) => setAttributes({ sliderControls: val })}
							help={__(' Enables the slider arrow/bullet controls.', 'lead-gen')}
						/>
					)}
					<SelectControl
						label={__('Selector', 'lead-gen')}
						value={selector}
						options={[
							{ label: __('All', 'lead-gen'), value: 'all' },
							{ label: __('Categories', 'lead-gen'), value: 'category' },
							{ label: __('Single Members', 'lead-gen'), value: 'name' },
						]}
						onChange={(val) => setAttributes({ selector: val })}
					/>

					{selector === 'category' && (
						<FormTokenField
							label={__('Choose Categories', 'lead-gen')}
							value={mapValuesToLabels(catOptions, categories)}
							suggestions={catOptions.map(o => o.label)}
							__experimentalExpandOnFocus={true}
							onChange={(tokens) => setAttributes({ categories: mapLabelsToValues(catOptions, tokens) })}
						/>
					)}

					{selector === 'name' && (
						<FormTokenField
							label={__('Choose Members', 'lead-gen')}
							value={mapValuesToLabels(teamOptions, teamMembers)}
							suggestions={teamOptions.map(o => o.label)}
							__experimentalExpandOnFocus={true}
							onChange={(tokens) => setAttributes({ teamMembers: mapLabelsToValues(teamOptions, tokens) })}
						/>
					)}

					<RangeControl
						label={__('slider' === displayType ? 'Slides Per View' : 'Cards Per Row', 'lead-gen')}
						value={perRow}
						onChange={(val) => setAttributes({ perRow: val })}
						min={1}
						max={4}
					/>
				</PanelBody>
				<PanelBody title={__('Card Settings', 'lead-gen')} initialOpen={true}>
					{'gallery' === displayType && (
						<SelectControl
							label={__('Gallery Card Size', 'lead-gen')}
							value={size}
							options={[
								{ label: 'Small', value: 'sm' },
								{ label: 'Medium', value: 'md' },
								{ label: 'Large', value: 'lg' },
							]}
							onChange={(val) => setAttributes({ size: val })}
						/>
					)}

					<ToggleControl
						label={__('Border', 'lead-gen')}
						checked={border}
						onChange={(value) => setAttributes({ border: value })}
					/>
					{border && (
						<SelectControl
							label={__('Border Color', 'lead-gen')}
							value={borderColor || 'primary'}
							options={COLORS}
							onChange={(value) => setAttributes({ borderColor: value })}
						/>
					)}
					<RangeControl
						label={__('Corner Rounding', 'lead-gen')}
						value={corners}
						onChange={(val) => setAttributes({ corners: val })}
						min={0}
						max={3}
					/>

				</PanelBody>
				{'gallery' !== displayType && (
					<PanelBody title={__('Team Member Settings', 'lead-gen')} initialOpen={true}>
						<BaseControl label="Order of Elements">
							<ReorderableMultiSelect
								options={ELEMENT_OPTIONS}
								selectedKeys={elements}
								onChangeSelected={(next) => setAttributes({ elements: next })}
							/>

							<hr/>

							<ToggleControl
								label={__('Image Enabled', 'lead-gen')}
								checked={!!imageEnabled}
								onChange={(val) => setAttributes({ imageEnabled: val })}
							/>

							{imageEnabled && (
								<>
									<SelectControl
										label={__('Image Placement', 'lead-gen')}
										value={imagePlacement}
										options={IMAGE_PLACEMENT_OPTIONS}
										onChange={(value) => setAttributes({ imagePlacement: value })}
									/>
									{imagePlacement !== 'none' && imagePlacement !== 'background' && (
										<SelectControl
											label={__('Image Size', 'lead-gen')}
											value={imageSize}
											options={[{ label: __('Full', 'lead-gen'), value: 'full' }, ...SIZE_OPTIONS]}
											onChange={(value) => setAttributes({ imageSize: value })}
										/>
									)}
								</>
							)}

							<SelectControl
								label={__('Text Align', 'lead-gen')}
								value={textAlign}
								options={[
									{ label: 'Left', value: 'left' },
									{ label: 'Center', value: 'center' },
									{ label: 'Right', value: 'right' },
								]}
								onChange={(val) => setAttributes({ textAlign: val })}
							/>

							{elements.includes('title') && (
								<SelectControl
									label={__('Title HTML Tag', 'lead-gen')}
									value={titleTag}
									options={TAGS}
									onChange={(val) => setAttributes({ titleTag: val })}
								/>
							)}
							{elements.includes('content') && (
								<NumberControl
									label={__('Excerpt Length', 'lead-gen')}
									__next40pxDefaultSize
									isShiftStepEnabled
									onChange={(val) => setAttributes({ excerptLength: val })}
									shiftStep={1}
									value={excerptLength}
								/>
							)}
							{elements.includes('read_more') && (
								<TextControl
									label={__('Button Text', 'lead-gen')}
									value={btnText}
									onChange={(val) => setAttributes({ btnText: val })}
								/>
							)}
						</BaseControl>
					</PanelBody>
				)}

				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			<div ref={wrapperRef}>
				<ServerSideRender
					block={name}
					attributes={attributes}
				/>
			</div>
		</AnimatedBlockWrapper>
	);
};

export default Edit;
