<?php
/**
 * WAI-ARIA Navigation Menu template functions
 *
 * @package StrategyBlocks
 * @see wp-includes/nav-menu-template.php
 */

namespace StrategyBlocks;

use StrategyBlocks\Utility;

/**
 * Create HTML list of nav menu items.
 *
 * @see wp-includes/nav-menu-template.php
 * @since 1.0.0
 * @uses Walker
 * @uses Walker_Nav_Menu
 */
class Aria_Walker_Nav_Menu extends \Walker_Nav_Menu {

	/**
	 * The style of dropdown icons to use.
	 *
	 * @var string Dropdown Style
	 */
	private $dropdown_style;

	/**
	 * The reference to the fontawesome object.
	 *
	 * @var FontAwesomeSVG FontAwesome object
	 */
	private $font_awesome;

	/**
	 * A defined constructor for the nav menu that allows fontawesome and a dropdown style to be defined on construction.
	 *
	 * @param FontAwesomeSVG $font_awesome   The reference to the fontawesome object.
	 * @param string         $dropdown_style The style of dropdown icons to use.
	 */
	public function __construct( $font_awesome, $dropdown_style = 'arrows' ) {
		$this->font_awesome = $font_awesome;
		$this->dropdown_style = $dropdown_style;
	}

	/**
	 * Starts the element output.
	 *
	 * @since 3.0.0
	 * @since 4.4.0 The {@see 'nav_menu_item_args'} filter was added.
	 * @since 5.9.0 Renamed `$item` to `$data_object` and `$id` to `$current_object_id`
	 *              to match parent class for PHP 8 named parameter support.
	 *
	 * @see Walker::start_el()
	 *
	 * @param string   $output            Used to append additional content (passed by reference).
	 * @param WP_Post  $data_object       Menu item data object.
	 * @param int      $depth             Depth of menu item. Used for padding.
	 * @param stdClass $args              An object of wp_nav_menu() arguments.
	 * @param int      $current_object_id Optional. ID of the current menu item. Default 0.
	 */
	public function start_el( &$output, $data_object, $depth = 0, $args = array(), $current_object_id = 0 ) {
		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';

		$classes = empty( $data_object->classes ) ? array() : (array) $data_object->classes;
		$classes[] = 'menu-item-' . $data_object->ID;

		$indent = ( $depth > 0 ? str_repeat( "\t", $depth ) : '' ); // code indent

		// Depth-dependent classes.
		$depth_classes = array(
			'navigation__dropdown_depth-' . $depth,
		);
		$depth_class_names = esc_attr( implode( ' ', $depth_classes ) );

		/**
		 * Filter the arguments for a single nav menu item.
		 *
		 * @since 4.4.0
		 *
		 * @param array  $args  An array of arguments.
		 * @param object $data_object  Menu item data object.
		 * @param int    $depth Depth of menu item. Used for padding.
		 */
		$args = apply_filters( 'nav_menu_item_args', $args, $data_object, $depth );

		/**
		 * Filter the CSS class(es) applied to a menu item's list item element.
		 *
		 * @since 3.0.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array  $classes The CSS classes that are applied to the menu item's `<li>` element.
		 * @param object $data_object    The current menu item.
		 * @param array  $args    An array of {@see wp_nav_menu()} arguments.
		 * @param int    $depth   Depth of menu item. Used for padding.
		 */
		$class_names = join( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $data_object, $args, $depth ) );
		$class_names = $class_names ? ' class="' . $depth_class_names . ' ' . esc_attr( $class_names ) . ' ' . get_field( 'menu_style_option', $data_object->ID ) . '"' : '';

		/**
		 * Filter the ID applied to a menu item's list item element.
		 *
		 * @since 3.0.1
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param string $menu_id The ID that is applied to the menu item's `<li>` element.
		 * @param object $data_object    The current menu item.
		 * @param array  $args    An array of {@see wp_nav_menu()} arguments.
		 * @param int    $depth   Depth of menu item. Used for padding.
		 */
		$current_object_id = apply_filters( 'nav_menu_item_id', 'menu-item-' . $data_object->ID, $data_object, $args, $depth );
		$current_object_id = $current_object_id ? ' id="' . esc_attr( $current_object_id ) . '"' : '';

		$has_children = in_array( 'menu-item-has-children', $data_object->classes );
		$output .= sprintf(
			'%s<li%s%s%s>',
			$indent,
			$current_object_id,
			$class_names,
			$has_children ? ' role="menuitem" aria-haspopup="true" aria-expanded="false" tabindex="0"' : ' role="menuitem"'
		);

		$atts = array();
		$atts['title']  = ! empty( $data_object->attr_title ) ? $data_object->attr_title : '';
		$atts['target'] = ! empty( $data_object->target ) ? $data_object->target : '';
		$atts['rel']    = ! empty( $data_object->xfn ) ? $data_object->xfn : '';
		$atts['href']   = ! empty( $data_object->url ) ? $data_object->url : '';

		/**
		 * Filter the HTML attributes applied to a menu item's anchor element.
		 *
		 * @since 3.6.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array $atts {
		 *     The HTML attributes applied to the menu item's `<a>` element, empty strings are ignored.
		 *
		 *     @type string $title  Title attribute.
		 *     @type string $target Target attribute.
		 *     @type string $rel    The rel attribute.
		 *     @type string $href   The href attribute.
		 * }
		 * @param object $data_object  The current menu item.
		 * @param array  $args  An array of {@see wp_nav_menu()} arguments.
		 * @param int    $depth Depth of menu item. Used for padding.
		 */
		$atts = apply_filters( 'nav_menu_link_attributes', $atts, $data_object, $args, $depth );

		$attributes = '';
		foreach ( $atts as $attr => $value ) {
			if ( ! empty( $value ) ) {
				$value = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
				$attributes .= ' ' . $attr . '="' . $value . '"';
			}
		}

		/** This filter is documented in wp-includes/post-template.php */
		$title = apply_filters( 'the_title', $data_object->title, $data_object->ID );

		/**
		 * Filter a menu item's title.
		 *
		 * @since 4.4.0
		 *
		 * @param string $title The menu item's title.
		 * @param object $data_object  The current menu item.
		 * @param array  $args  An array of {@see wp_nav_menu()} arguments.
		 * @param int    $depth Depth of menu item. Used for padding.
		 */
		$title = apply_filters( 'nav_menu_item_title', $title, $data_object, $args, $depth );

		$item_output = $args->before;

		if ( get_field( 'heading_icon', $data_object->ID ) ) {
			$item_output .= wp_get_attachment_image( get_field( 'heading_icon', $data_object->ID ), [ '75px', '75px' ], false, [ 'class' => 'navigation__icon' ] );
		}
		if ( get_field( 'remove_link', $data_object->ID ) ) {
			$item_output .= '<button' . $attributes . '>';
			$item_output .= $args->link_before . $title . $args->link_after . '</button>';
			if ( $has_children ) {
				$item_output .= '<span class="navigation__dropdown-toggle">';
				$item_output .= $this->output_dropdown_icon();
				$item_output .= '</span>';
			}
			// $item_output .= '</button>';
		} else {
			$item_output .= '<a' . $attributes . '>';
			$item_output .= $args->link_before . $title . $args->link_after . '</a>';
			if ( $has_children ) {
				$item_output .= '<button class="navigation__dropdown-toggle">';
				$item_output .= '<span class="sr-only">Expand Dropdown</span>';
				$item_output .= $this->output_dropdown_icon();
				$item_output .= '</button>';
			}
			// $item_output .= '</a>';
		}
		$item_output .= $args->after;

		/**
		 * Filter a menu item's starting output.
		 *
		 * The menu item's starting output only includes `$args->before`, the opening `<a>`,
		 * the menu item's title, the closing `</a>`, and `$args->after`. Currently, there is
		 * no filter for modifying the opening and closing `<li>` for a menu item.
		 *
		 * @since 3.0.0
		 *
		 * @param string $item_output The menu item's starting HTML output.
		 * @param object $data_object        Menu item data object.
		 * @param int    $depth       Depth of menu item. Used for padding.
		 * @param array  $args        An array of {@see wp_nav_menu()} arguments.
		 */
		$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $data_object, $depth, $args );
	}

	/**
	 * Outputs the SVG icon that is defined when the class is constructed.
	 */
	private function output_dropdown_icon() {
		ob_start();
		switch ( $this->dropdown_style ) {
			case 'plus-minus':
				$plus = $this->font_awesome->get_svg( 'fa-solid fa-plus', [ 'class' => 'navigation__dropdown-toggle-icon plus' ] );
				Utility\sanitize_and_output_svg( $plus );
				$minus = $this->font_awesome->get_svg( 'fa-solid fa-minus', [ 'class' => 'navigation__dropdown-toggle-icon minus' ] );
				Utility\sanitize_and_output_svg( $minus );
				break;
			default:
				$chevron = $this->font_awesome->get_svg( 'fa-solid fa-chevron-down', [ 'class' => 'navigation__dropdown-toggle-icon' ] );
				Utility\sanitize_and_output_svg( $chevron );
				break;
		}
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
}
