<?php
/**
 * Bio Block
 *
 * @package StrategyBlocks
 *
 * @param array  $attributes  Block attributes.
 * @param string $content
 * @param object $block
 */

use StrategyBlocks\Utility;
use StrategyBlocks\FontAwesomeSVG;

$id               = $attributes['anchor'] ?? '';
$display_type     = $attributes['displayType'] ?? 'gallery';
$selector         = $attributes['selector'] ?? 'all';
$size             = $attributes['size'] ?? 'md';
$corners          = $attributes['corners'] ?? 0;
$border           = $attributes['border'] ?? false;
$border_color     = $attributes['borderColor'] ?? '';
$per_row          = $attributes['perRow'] ?? 3;
$btn_text         = $attributes['btnText'] ?? '';
$categories       = $attributes['categories'] ?? [];
$team_ids         = $attributes['teamMembers'] ?? [];
$image_enabled    = 'gallery' === $display_type ? true : $attributes['imageEnabled'] ?? false;
$image_placement  = $attributes['imagePlacement'] ?? 'background';
$image_size       = $attributes['imageSize'] ?? 'full';
$text_align       = $attributes['textAlign'] ?? 'left';
$title_tag        = $attributes['titleTag'] ?? 'h3';
$elements         = $attributes['elements'] ?? [];
$excerpt_length   = $attributes['excerptLength'] ?? 10;

$class_name = implode(
	' ',
	array_filter(
		[
			'lg-block',
			'lg-block__content',
			'lg-block__bio--container',
			'lg-block__bio--' . $display_type,
			'lg-block__bio--' . $per_row,
			'lg-block__bio--' . $image_placement,
			'lg-block__bio--img-' . $image_size,
			'slider' === $display_type && ! empty( $attributes['sliderControls'] ) ? 'swiper--nav' : '',
			Utility\get_block_style_classes( $attributes ),
			Utility\get_visibility_classes( $attributes ),
		]
	)
);

$component_classes = [
	'background' => null,
	'overlay'    => null,
	'text'       => null,
	'link'       => null,
	'button'     => null,
];

$wp_classes = explode( ' ', trim( Utility\get_native_wp_classes( $attributes ) ) );
foreach ( $wp_classes as $class ) {
	if ( str_contains( $class, 'text-color' ) ) {
		$component_classes['text'] = $class;
	} elseif ( str_contains( $class, 'bg-color' ) ) {
		$component_classes['background'] = $class;
		$component_classes['overlay'] = $class;
		if ( str_contains( $class, 'primary' ) ) {
			$component_classes['button'] = 'btn-secondary';
		} elseif ( str_contains( $class, 'secondary' ) ) {
			$component_classes['button'] = 'btn-primary';
		} else {
			$component_classes['button'] = 'btn-tertiary';
		}
	} elseif ( str_contains( $class, 'link-color' ) ) {
		$component_classes['link'] = $class;
	} else {
		$class_name .= ' ' . $class;
	}
}

$svg_args = [
	'svg'   => [
		'class'           => true,
		'aria-hidden'     => true,
		'aria-labelledby' => true,
		'role'            => true,
		'xmlns'           => true,
		'width'           => true,
		'height'          => true,
		'viewbox'         => true,
	],
	'g'     => [ 'fill'  => true ],
	'title' => [ 'title' => true ],
	'path'  => [
		'd'               => true,
		'fill'            => true,
	],
];

$design_classes = [
	'background' => $component_classes['background'] ?? 'bg-color-secondary',
	'overlay'    => $component_classes['overlay'] ?? 'bg-color-secondary',
	'text'       => $component_classes['text'] ? $component_classes['text'] . ' text-align--' . $text_align : 'text-color-primary text-align--' . $text_align,
	'link'       => $component_classes['link'] ?? 'link-color-accent-1',
	'button'     => $component_classes['button'] ?? 'btn-primary',
];

$common_classes = implode(
	' ',
	array_filter(
		[
			'lg-bio',
			'lg-bio--corners-' . $corners,
			$design_classes['background'],
			$border ? 'card--bordered' : '',
			$border_color ? 'border-color-' . $border_color : '',
		]
	)
);

$open_class   = $common_classes . ' lg-bio--open lg-bio--open--size-' . $size;
$closed_class = $common_classes . ' lg-bio--closed lg-bio--closed--size-' . $size;
$animations   = Utility\get_animation_data( $attributes );

$query_args = [
	'post_type'      => 'team-members',
	'post_status'    => 'publish',
	'posts_per_page' => -1,
];

if ( 'name' === $selector && ! empty( $team_ids ) ) {
	$query_args['post__in'] = $team_ids;
} elseif ( 'category' === $selector && ! empty( $categories ) ) {
	$query_args['tax_query'] = [
		[
			'taxonomy' => 'team-member-categories',
			'field'    => 'id',
			'terms'    => $categories,
		],
	];
}

$query         = new WP_Query( $query_args );
$font_awesome  = new FontAwesomeSVG( STRATEGY_BLOCKS_DIST_PATH . 'font-awesome/svgs' );
$is_slider     = 'slider' === $display_type;

do_action( 'lg_block_bio_before' );

if ( ! function_exists( 'render_bio_element' ) ) {
	/**
	 * Render a single Bio block element based on the element type.
	 *
	 * Outputs the appropriate markup for a given Bio component such as title,
	 * name, social links, categories, excerpt content, or read more button.
	 * All output is properly escaped according to WordPress coding standards.
	 *
	 * @param string $element         Element identifier (e.g. title, name, socialLinks, categories, content, read_more).
	 * @param int    $post_id         Post ID of the team member.
	 * @param string $title_tag       HTML tag used for the title element.
	 * @param string $btn_text        Text for the read more button.
	 * @param array  $design_classes  Array of design-related CSS classes.
	 * @param string $link            URL to the single post page.
	 * @param array  $social_media    Social media links array (key => url).
	 * @param array  $categories      Term objects associated with the post.
	 * @param int    $excerpt_length  Length of the generated excerpt.
	 * @param array  $svg_args        Allowed SVG attributes for wp_kses.
	 *
	 * @return void
	 */
	function render_bio_element( $element, $post_id, $title_tag, $btn_text, $design_classes, $link, $social_media, $categories, $excerpt_length, $svg_args ) {
		switch ( $element ) {
			case 'title':
				echo '<' . esc_html( $title_tag ) . ' class="lg-bio__text lg-bio__text--name">' . esc_html( get_the_title( $post_id ) ) . '</' . esc_html( $title_tag ) . '>';
				break;

			case 'name':
				$title = get_field( 'title', $post_id );
				echo '<p class="lg-bio__text lg-bio__text--title">' . esc_html( $title ) . '</p>';
				break;

			case 'socialLinks':
				if ( ! empty( $social_media ) && is_array( $social_media ) ) {
					echo '<ul class="lg-bio__media">';
					foreach ( $social_media as $key => $url ) {
						if ( $url ) {
							echo '<li class="lg-bio__media-icon">';
							echo '<span class="sr-only">' . esc_html( $key ) . '</span>';
							echo '<a href="' . esc_url( $url ) . '" target="_blank" rel="noreferrer noopener" title="(opens in a new window)" class="lg-bio__media-link">';
							echo wp_kses( Utility\get_social_icon( $key ), $svg_args );
							echo '</a></li>';
						}
					}
					echo '</ul>';
				}
				break;

			case 'categories':
				if ( ! empty( $categories ) && is_array( $categories ) ) {
					echo '<p class="lg-bio__categories size-xs">';
					echo '<span class="sr-only">Posted In: </span>';
					foreach ( $categories as $category ) {
						printf(
							'<a href="%1$s" class="post__category">%2$s</a>',
							esc_url( get_category_link( $category->term_id ) ),
							esc_html( $category->name )
						);
					}
					echo '</p>';
				}
				break;

			case 'content':
				echo '<div class="post__excerpt">';
				Utility\sanitize_and_output_html_content( Utility\get_custom_excerpt( $excerpt_length ) );
				echo '</div>';
				break;

			case 'read_more':
				if ( $link ) {
					echo '<div class="lg-bio__footer">';
					echo '<a class="btn ' . esc_attr( $design_classes['button'] ) . ' lg-bio__btn" href="' . esc_url( $link ) . '">' . esc_html( $btn_text ) . '</a>';
					echo '</div>';
				}
				break;
		}
	}
}

if ( $query->have_posts() ) : ?>
	<div <?php echo $id ? 'id="' . esc_attr( $id ) . '"' : ''; ?> class="<?php echo esc_attr( $class_name ); ?>">

		<?php if ( $is_slider ) : ?>
		<div class="bio__slider swiper" data-swiper-perview="<?php echo esc_attr( $per_row ); ?>">
			<div class="swiper-wrapper">
				<?php endif; ?>

				<?php
				while ( $query->have_posts() ) :
					$query->the_post();
					$post_id = get_the_ID();
					$categories = get_the_terms( get_the_ID(), 'team-member-categories' );
					$bio = get_the_excerpt( $post_id );
					$show_bio = $bio && 'gallery' === $display_type;
					$social_media = get_field( 'member_social_media', $post_id );
					$link = get_field( 'link_to_single_page', $post_id ) ? get_permalink( $post_id ) : null;
					$attachment_id = get_post_thumbnail_id( $post_id );
					?>

					<?php if ( $is_slider ) : ?><div class="swiper-slide"><?php endif; ?>

					<div
						class="lg-block__bio lg-block__bio--closed lg-block__bio--closed--size-<?php echo esc_attr( $size ); ?> row-max--closed-<?php echo esc_attr( $per_row ); ?>"
						<?php echo $animations ? 'data-animation="' . esc_attr( $animations ) . '"' : ''; ?>
					>
						<div class="<?php echo esc_attr( $closed_class ); ?>">
							<?php if ( $image_enabled && $attachment_id ) : ?>
								<div class="lg-bio__image--container">
									<?php
									echo wp_get_attachment_image(
										$attachment_id,
										'full',
										false,
										[
											'class'   => 'lg-bio__image lg-bio--corners-' . $corners,
											'loading' => 'lazy',
										],
									);
									?>
									<div class="lg-bio__overlay lg-bio--corners-<?php echo esc_attr( $corners ); ?> <?php echo esc_attr( $design_classes['overlay'] ); ?>"></div>
								</div>
							<?php endif; ?>

							<?php if ( ! empty( $elements ) && 'gallery' !== $display_type ) : ?>
								<div class="lg-bio__text lg-bio__text--container <?php echo esc_attr( $design_classes['text'] ); ?>">
									<?php
									foreach ( $elements as $element ) {
										render_bio_element( $element, $post_id, $title_tag, $btn_text, $design_classes, $link, $social_media, $categories, $excerpt_length, $svg_args );
									}
									?>
								</div>
							<?php endif; ?>
							<?php if ( 'gallery' === $display_type ) : ?>
								<div class="lg-bio__text lg-bio__text--container <?php echo esc_attr( $design_classes['text'] ); ?>">
									<?php
									$post_title = get_the_title( $id );
									$title = get_field( 'title', $id );
									?>
									<span class="lg-bio__text lg-bio__text--name"><?php echo esc_html( $post_title ); ?></span>
									<span class="lg-bio__text lg-bio__text--title"><?php echo esc_html( $title ); ?></span>
								</div>
								<?php if ( $bio ) : ?>
									<a class="btn <?php echo esc_attr( $design_classes['button'] ); ?> lg-bio__btn lg-bio__btn--open">
										<?php echo esc_html( $btn_text ); ?>
									</a>
								<?php elseif ( $link ) : ?>
									<a class="btn <?php echo esc_attr( $design_classes['button'] ); ?> lg-bio__btn" href="<?php echo esc_url( $link ); ?>">
										<?php echo esc_html( $btn_text ); ?>
									</a>
								<?php endif; ?>
							<?php endif; ?>
						</div>

						<?php if ( $show_bio ) : ?>
							<div class="<?php echo esc_attr( $open_class ); ?> <?php echo esc_attr( $design_classes['text'] ); ?> <?php echo esc_attr( $design_classes['link'] ); ?>">
								<div class="lg-bio__text lg-bio__text--bio">
									<?php Utility\sanitize_and_output_html_content( $bio ); ?>
								</div>
								<?php if ( $link ) : ?>
									<a class="lg-bio__text lg-bio__text--link" href="<?php echo esc_url( $link ); ?>">
										<?php esc_html_e( 'Read More >', 'lead-gen' ); ?>
									</a>
								<?php endif; ?>
							</div>
						<?php endif; ?>
					</div>

					<?php if ( $is_slider ) : ?></div><?php endif; ?>
				<?php endwhile; ?>

				<?php if ( $is_slider ) : ?>
			</div>
					<?php
					if ( ! empty( $attributes['sliderControls'] ) ) {
						Utility\render_slider_controls( $font_awesome );
					}
					?>
		</div>
				<?php endif; ?>

	</div>
	<?php
		endif;
		wp_reset_postdata();
?>

<?php do_action( 'lg_block_bio_after' ); ?>
