<?php
/**
 * NavigationMenus
 *
 * @package StrategyBlocks
 */

namespace StrategyBlocks;

/**
 * Creates the StrategyBlocks post type and shortcodes to output them.
 */
class NavigationMenus extends \StrategyBlocks\Module {

	/**
	 * Registers instances where we will override default WP Core behavior.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_filter( 'nav_menu_item_id', '__return_null', 10, 3 );
		add_filter( 'nav_menu_css_class', [ $this,  'clean_menu_css_classes' ], 10, 2 );
		add_filter( 'nav_menu_submenu_css_class', [ $this,  'nav_menu_submenu_css_class' ] );
		add_filter( 'nav_menu_link_attributes', [ $this,  'add_class_to_menu_links' ], 10, 3 );
	}

	/**
	 * Filters the CSS classes applied to a menu item's list item element.
	 *
	 * @since 3.0.0
	 * @since 4.1.0 The `$depth` parameter was added.
	 *
	 * @param string[] $classes   Array of the CSS classes that are applied to the menu item's `<li>` element.
	 * @param WP_Post  $menu_item The current menu item object.
	 */
	public function clean_menu_css_classes( $classes, $menu_item ) {
		$slug = sanitize_title( $menu_item->title );
		$cpt = get_post_type();

		$is_cpt = in_array( $cpt, get_post_types( array( '_builtin' => false ) ) );

		// Fix core `active` behavior for custom post types
		if ( $is_cpt ) {
			$classes = str_replace( 'current_page_parent', '', $classes );
			$link = get_post_type_archive_link( $cpt );

			if ( false !== $link && ! is_search() ) {
				if ( parse_url( $link, PHP_URL_PATH ) === parse_url( $menu_item->url, PHP_URL_PATH ) ) {
					$classes[] = 'active';
				}
			}
		}

		// Remove most core classes
		$classes = preg_replace( '/(current(-menu-|[-_]page[-_])(item|parent|ancestor))/', 'active', $classes );
		$classes = preg_replace( '/^((menu|page)[-_\w+]+)+/', '', $classes );

		// Re-add core `navigation__item` class
		$classes[] = 'navigation__item';

		// Re-add core `navigation__item-has-children` class on parent elements
		if ( array_search( 'menu-item-has-children', $menu_item->classes ) !== false ) {
			$classes[] = 'navigation__item-has-children';
		}

		// Add `menu-<slug>` class
		$classes[] = 'menu-' . $slug;

		$classes = array_unique( $classes );
		$classes = array_map( 'trim', $classes );

		return array_filter( $classes );
	}

	/**
	 * Renames CSS class on ul.sub-menu to navigation__dropdown
	 *
	 * @since 4.8.0
	 *
	 * @param string[] $classes Array of the CSS classes that are applied to the menu `<ul>` element.
	 */
	public function nav_menu_submenu_css_class( $classes ) {
		$classes = [ 'navigation__dropdown' ];
		return $classes;
	}

	/**
	 * Adds class to a tag in WordPress menus
	 *
	 * @since 3.6.0
	 * @since 4.1.0 The `$depth` parameter was added.
	 *
	 * @param array    $atts {
	 *        The HTML attributes applied to the menu item's `<a>` element, empty strings are ignored.
	 *
	 *     @type string $title        Title attribute.
	 *     @type string $target       Target attribute.
	 *     @type string $rel          The rel attribute.
	 *     @type string $href         The href attribute.
	 *     @type string $aria-current The aria-current attribute.
	 * }
	 * @param WP_Post  $menu_item The current menu item object.
	 * @param stdClass $args      An object of wp_nav_menu() arguments.
	 */
	public function add_class_to_menu_links( $atts, $menu_item, $args ) {
		$atts['class'] = 'navigation__link';
		return $atts;
	}
}
