import { __ } from '@wordpress/i18n';
import { useRef, useEffect } from "@wordpress/element";
import { InspectorControls, MediaUpload, MediaUploadCheck } from '@wordpress/block-editor';
import { PanelBody, ToggleControl, RangeControl, SelectControl, Button } from '@wordpress/components';
import ServerSideRender from '@wordpress/server-side-render';
import AnimationControls from '../components/animations/AnimationControl';
import AnimatedBlockWrapper from '../components/animations/AnimatedBlockWrapper';
import { initSwiper } from "../../../assets/js/slider";
import BorderPanel from "../components/BorderPanel/BorderPanel";
import VisibilityControls from "../components/visibility/VisibilityControls";
import {getVisibilityClasses} from "../components/blockStyleUtils";

const Edit = ({ attributes, setAttributes, name }) => {
	const {
		content,
		sliderControls,
		imageFit,
		pageSize,
		sliderHeight,
		animationEntrance,
		animationDuration,
		animationDelay
	} = attributes;

	const wrapperRef = useRef(null);

	useEffect(() => {
		if (!wrapperRef.current) return;

		const observer = new MutationObserver(() => {
			const slider = wrapperRef.current.querySelector('.image-slider');
			if (slider && !slider.dataset.initialized) {
				initSwiper( slider, { autoHeight: false, allowTouchMove: false } );
				slider.dataset.initialized = '1';
			}
		});

		observer.observe(wrapperRef.current, {
			childList: true,
			subtree: true,
		});

		return () => observer.disconnect();
	}, []);

	return (
		<AnimatedBlockWrapper
			animationEntrance={animationEntrance}
			animationDuration={animationDuration}
			animationDelay={animationDelay}
			attributes={attributes}
			className={`lg-block lg-block__image-slider ${getVisibilityClasses(attributes)}`}
		>
			<InspectorControls>
				<PanelBody title={__('Image Slider Settings', 'lead-gen')} initialOpen={true}>
					<MediaUploadCheck>
						<MediaUpload
							onSelect={(images) =>
								setAttributes({ content: images.map((img) => img.id) })
							}
							allowedTypes={['image']}
							multiple
							gallery
							value={content}
							render={({ open }) => (
								<div class="upload-image">
									<Button onClick={open} variant="secondary">
										{content?.length ? __('Edit Images', 'lead-gen') : __('Select Images', 'lead-gen')}
									</Button>
									{content?.length ? (
										<Button onClick={() => setAttributes({ content: [] })} isDestructive variant="link">
											{__('Remove Images', 'lead-gen')}
										</Button>
									) : ''}
								</div>
							)}
						/>
					</MediaUploadCheck>
					<hr/>

					<ToggleControl
						label={__('Slider Controls', 'lead-gen')}
						checked={!!sliderControls}
						onChange={(val) => setAttributes({ sliderControls: val })}
						help={__('Enables the slider arrow/bullet controls.', 'lead-gen')}
					/>

					<SelectControl
						label={__('Image Fit', 'lead-gen')}
						value={imageFit}
						options={[
							{ label: __('Cover', 'lead-gen'), value: 'cover' },
							{ label: __('Contain', 'lead-gen'), value: 'contain' },
						]}
						onChange={(val) => setAttributes({ imageFit: val })}
					/>

					<RangeControl
						label={__('Per View', 'lead-gen')}
						value={pageSize}
						onChange={(val) => setAttributes({ pageSize: val })}
						min={1}
						max={50}
					/>

					<SelectControl
						label={__('Slider Height', 'lead-gen')}
						value={sliderHeight}
						options={[
							{ label: __('XS', 'lead-gen'), value: 'xs' },
							{ label: __('SM', 'lead-gen'), value: 'sm' },
							{ label: __('MD', 'lead-gen'), value: 'md' },
							{ label: __('LG', 'lead-gen'), value: 'lg' },
							{ label: __('XL', 'lead-gen'), value: 'xl' },
						]}
						onChange={(val) => setAttributes({ sliderHeight: val })}
					/>
				</PanelBody>

				<BorderPanel
					attributes={attributes}
					setAttributes={setAttributes}
				/>

				<VisibilityControls attributes={attributes} setAttributes={setAttributes} />
				<AnimationControls attributes={attributes} setAttributes={setAttributes} />
			</InspectorControls>

			<div ref={wrapperRef}>
				<ServerSideRender
					block={name}
					attributes={attributes}
				/>
			</div>
		</AnimatedBlockWrapper>
	);
};

export default Edit;
