<?php
/**
 * Image Slider
 *
 * @package StrategyBlocks
 *
 * @param array  $attributes  The block's attributes.
 * @param string $content     The block's inner content (unused here).
 * @param object $block       The block object (contains anchor, etc.).
 */

use StrategyBlocks\Utility;
use StrategyBlocks\FontAwesomeSVG;

$id               = $attributes['anchor'] ?? '';
$image_ids        = $attributes['content'] ?? [];
$slider_controls  = $attributes['sliderControls'] ?? true;
$image_fit        = $attributes['imageFit'] ?? 'cover';
$page_size        = max( 1, intval( $attributes['pageSize'] ?? 3 ) );
$slider_height    = $attributes['sliderHeight'] ?? 'md';
$visible_border   = $attributes['border'] ?? false;
$border_radius    = $attributes['borderRadius'] ?? 'none';
$border_color     = $attributes['borderColor'] ?? '';

$class_name = implode(
	' ',
	array_filter(
		[
			'lg-block',
			'lg-block__image-slider',
			! empty( $attributes['sliderControls'] ) ? 'swiper--nav' : '',
			'row',
			Utility\get_block_style_classes( $attributes ),
			Utility\get_native_wp_classes( $attributes ),
			Utility\get_visibility_classes( $attributes ),
		]
	)
);

$slider_class = implode(
	' ',
	array_filter(
		[
			'image-slider__card',
			'image-slider__card__border--' . $border_radius,
			'image-slider__card__size--' . $slider_height,
			$visible_border ? 'border-color-' . $border_color : 'image-slider__card__border--invis',
		]
	)
);

$animations  = Utility\get_animation_data( $attributes );
$placeholder = true;

if ( ! empty( $image_ids ) && is_array( $image_ids ) ) {
	foreach ( $image_ids as $image_id ) {
		if ( intval( $image_id ) > 0 ) {
			$placeholder = false;
			break;
		}
	}
}

// Duplicate images until we have at least $page_size + 1 slides,
if ( ! empty( $image_ids ) && is_array( $image_ids ) ) {
	$target     = $page_size + 1;
	$image_count = count( $image_ids );

	while ( $image_count < $target ) {
		$missing     = $target - $image_count;
		$image_ids   = array_merge( $image_ids, array_slice( $image_ids, 0, $missing ) );
		$image_count = count( $image_ids );
	}
}

$content_length   = $placeholder ? $page_size + 1 : count( $image_ids );
$pages            = (int) ceil( $content_length / $page_size );
$font_awesome     = new FontAwesomeSVG( STRATEGY_BLOCKS_DIST_PATH . 'font-awesome/svgs' );

do_action( 'lg_block_image_slider_before' );
?>

<div
	<?php
	if ( $id ) {
		echo 'id="' . esc_attr( $id ) . '"';
	}
	?>
	class="<?php echo esc_attr( $class_name ); ?>"
	<?php
	if ( $animations ) {
		echo 'data-animation="' . esc_attr( $animations ) . '"';
	}
	?>
>
	<div class="image-slider swiper" data-swiper-perview="<?php echo esc_attr( $page_size ); ?>">
		<div class="swiper-wrapper">
			<?php for ( $i = 0; $i < $content_length; $i++ ) { ?>
				<div class="swiper-slide <?php echo esc_attr( $slider_class ); ?>">
					<?php
					if ( $placeholder ) {
						if ( 'lg' === $slider_height && 3 === $page_size ) {
							?>
							<img
								class="image-slider_card--image image-slider__card--image--fit-<?php echo esc_attr( $image_fit ); ?>"
								src="<?php echo esc_url( 'https://placehold.co/405x306' ); ?>"
								alt="Placeholder - Please Replace Me"
							/>
							<?php
						} else {
							?>
							<div class="image-slider__card--image image-slider__card--placeholder"></div>
							<?php
						}
					} else {
						echo wp_get_attachment_image(
							$image_ids[ $i ],
							'large',
							false,
							[
								'class' => 'image-slider__card--image image-slider__card--image--fit-' . esc_attr( $image_fit ),
							]
						);
					}
					?>
				</div>
			<?php } ?>
		</div>
		<?php if ( ! empty( $attributes['sliderControls'] ) ) : ?>
			<?php Utility\render_slider_controls( $font_awesome ); ?>
		<?php endif; ?>
	</div>
</div>

<?php do_action( 'lg_block_image_slider_after' ); ?>
