<?php
/**
 * Plugin Name: Seamless Login
 * Plugin URI:  https://strategynewmedia.com
 * Description: Seamless Login
 * Version:     1.1.0
 * Author:      Strategy, LLC
 *
 * @package     Strategy
 */

namespace Strategy\SeamlessLogin;

require_once __DIR__ . '/inc/security-checks.php';
\Strategy\SeamlessLogin\check_security();

require_once __DIR__ . '/inc/user-nonce-helper.php';
require_once __DIR__ . '/inc/logger.php';
require_once __DIR__ . '/inc/sign-on-user-provider.php';
require_once __DIR__ . '/inc/custom-exceptions.php';

use WP_Error;
use Strategy\SeamlessLogin\UserNonceHelper;
use Strategy\SeamlessLogin\Logger;
use Strategy\SeamlessLogin\SignOnUserProvider;

const BASE_URL = 'seamless_login/v1';

StrategySeamlessLogin::initialize();

class StrategySeamlessLogin {
	const REDIRECT_URL_ON_ERROR    = '/wp-login.php';
	const WP_CLI_COMMAND_NAME      = 'seamless-login';
	const WP_CLI_EMAIL_ARG         = 'user-email';
	const WP_CLI_FIRST_NAME_ARG    = 'first-name';
	const WP_CLI_LAST_NAME_ARG     = 'last-name';
	const WP_CLI_USER_ROLE_ARG     = 'user-role';
	const WP_CLI_DASHBOARD_ENV_ARG = 'dashboard-env';
	const REDIRECT_URL_ON_SUCCESS  = '/wp-admin/';
	const USER_PORTAL_HOSTNAME_PRD = 'api.leadfuel.app';
	const USER_PORTAL_HOSTNAME_DEV = 'staging-api.leadfuel.app';
	const USER_PORTAL_HOSTNAME_LOCAL = 'api.leadfuel.local';

	public static $instance;

	private $login_route  = '/' . BASE_URL . '/login';
	private $sign_on_user_provider;

	private $user_nonce_helper;

	public function __construct( $sign_on_user_provider, $user_nonce_helper ) {
		$this->sign_on_user_provider  = $sign_on_user_provider;
		$this->user_nonce_helper      = $user_nonce_helper;
	}

	public static function initialize( $sign_on_user_provider = null, $user_nonce_helper = null ) {
		$sign_on_user_provider  = $sign_on_user_provider ?? new SignOnUserProvider();
		$user_nonce_helper      = $user_nonce_helper ?? new UserNonceHelper();
		self::$instance         = new self( $sign_on_user_provider, $user_nonce_helper );

		// <domain_name>/index.php?rest_route=/<BASE_URL>/<endpoint>
		add_action(
			'rest_api_init',
			function () {
				register_rest_route(
					BASE_URL,
					'/login',
					array(
						'methods'             => 'GET',
						'callback'            => array( self::$instance, 'login' ),
						'permission_callback' => '__return_true',
					)
				);
				register_rest_route(
					BASE_URL,
					'/one-click-login-url',
					array(
						'methods'             => 'GET',
						'callback'            => array( self::$instance, 'one_click_login_url' ),
						'permission_callback' => array( self::$instance, 'permission_check' ),
					)
				);
			}
		);

		if ( defined( 'WP_CLI' ) && \WP_CLI ) {
			\WP_CLI::add_command( self::WP_CLI_COMMAND_NAME, self::$instance );
		}

	}

	public function permission_check( $request ) {

		$referer = $request->get_header( 'referer' );
		if ( null !== $referer && strpos( $referer, 'wp-json' ) !== false ) {
			Logger::log( Logger::WP_JSON_REFERER_ERROR, 'Referer check did not pass.' );
			return new WP_Error( 'bad_request', __( 'Referer check did not pass.' ), array( 'status' => 400 ) );
		}

		if ( null !== $referer && ! $this->referer_is_user_portal( $referer ) ) {
			Logger::log( Logger::WP_JSON_REFERER_ERROR, 'Request is not coming from LeadFuel Dashboard' );
			return new WP_Error( 'bad_request', __( 'Request is not coming from LeadFuel Dashboard' ), array( 'status' => 400 ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			Logger::log( Logger::WP_JSON_REFERER_ERROR, 'User with application password cannot log users in.' );
			return new WP_Error( 'bad_request', __( 'User with application password cannot log users in.' ), array( 'status' => 403 ) );
		}

		return true;
	}

	public function login( $request ) {
		$time_start = round( microtime( true ) * 1000 );

		try {
			if ( is_multisite() ) {
				throw new MultisiteEnabledException();
			}

			if ( ! is_ssl() && force_ssl_admin() ) {
				return $this->generate_https_redirect( $request->get_query_params() );
			}

			list( $nonce, $user_email ) = $this->get_params_from_login_request( $request );

			if ( ! $this->validate_non_empty_string( $user_email ) ) {
				throw new \Exception( " User email ({$user_email}) is blank " );
			}

			$user       = $this->sign_on_user_provider->get_wp_user( $user_email );
			$nonce_data = $this->user_nonce_helper->get_nonce_data( $user->ID );

			if ( empty( $nonce_data ) ) {
				throw new NonceMetaDataValidationException( "Empty nonce data retrieved for User ({$user_email}) during login." );
			}
			
			// TODO: Send permission check request back to leadfuel api to ensure user has access to this site.
			$is_valid = $this->user_nonce_helper->validate_nonce( $user->ID, $nonce, $nonce_data );
			if ( $is_valid ) {
				$this->sign_on_user_provider->login_user( $user, $time_start );
				$redirect_url = self::REDIRECT_URL_ON_SUCCESS;
			}
		} catch ( NonceMetaDataValidationException $e ) {
			Logger::log( Logger::NONCE_META_DATA_VALIDATION_ERROR, $e->getMessage(), $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
		} catch ( MultisiteEnabledException $e ) {
			Logger::log( Logger::MULTISITE_ENABLED_ERROR, $e->getMessage(), $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
		} catch ( \Exception $e ) {
			Logger::log( Logger::GENERAL_EXCEPTION_ERROR, $e->getMessage() . $e->getTraceAsString(), isset( $user_email ) ? $user_email : 'Unknown' );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
		}

		$response = new \WP_REST_Response( null, 307, array( 'Location' => $redirect_url ?? self::REDIRECT_URL_ON_ERROR ) );

		return $response;
	}

	public function one_click_login_url( $request ) {
		try {
			if ( is_multisite() ) {
				throw new MultisiteEnabledException();
			}

			if ( ! is_ssl() && force_ssl_admin() ) {
				return $this->generate_https_redirect( $request->get_query_params() );
			}

			list( $user_email, $first_name, $last_name, $role, $dashboard_env ) = $this->get_params_from_one_click_login_url_request( $request );

			$this->validate_login_params( $user_email, $first_name, $last_name, $role, $dashboard_env );

			$this->double_check_dashboard_permissions( $user_email, $role, $dashboard_env );

			$user = $this->sign_on_user_provider->get_or_create_wp_user( $user_email, $first_name, $last_name, $role );

			$nonce_array = $this->user_nonce_helper->generate_nonce( $user->ID );
			$nonce       = $nonce_array['nonce'];
			$expiration  = $nonce_array['expiration'];

			$successfully_added = $this->user_nonce_helper->add_nonce( $user->ID, $nonce, $expiration );

			if ( ! $successfully_added ) {
				throw new UserMetaAdditionException( "Nonce ({$nonce}) was not added successfully to users ({$user_email}) meta data" );
			}

			$redirect_url = get_rest_url( null, $this->login_route );
			$query_params = array( 'user_email' => $user_email, 'nonce' => $nonce );

		} catch ( UserCreationException $e ) {
			$this->sign_on_user_provider->rollback_user_creation( $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::USER_CREATE_ERROR . ": {$e->getMessage()}";
		} catch ( UserMetaAdditionException $e ) {
			$this->sign_on_user_provider->rollback_user_creation( $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::ADD_USER_META_ERROR . ": {$e->getMessage()}";
		} catch ( NonceMetaDataValidationException $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::NONCE_META_DATA_VALIDATION_ERROR . ": {$e->getMessage()}";
		} catch ( MultisiteEnabledException $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::MULTISITE_ENABLED_ERROR . ": {$e->getMessage()}";
		} catch ( \Exception $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::GENERAL_EXCEPTION_ERROR . ": {$e->getMessage()}";
		}
		
		if ( isset( $error ) ) {
			return new \WP_REST_Response( $error, 400 );
		}
		
		$data = array(
			'redirect_url' => $redirect_url . '?' . http_build_query( $query_params ),
		);

		return new \WP_REST_Response( $data, 200 );
	}

	public function __invoke( $args, $assoc_args ) {
		echo wp_json_encode( $this->strategy_sso( $assoc_args ) );
	}

	private function referer_is_user_portal( $referer ) {
		if ( wp_parse_url( $referer ) === false ) {
			return false;
		}

		$hostname = wp_parse_url( $referer, PHP_URL_HOST );
		$path     = wp_parse_url( $referer, PHP_URL_PATH );

		$is_not_prd_hostname 	= self::USER_PORTAL_HOSTNAME_PRD !== $hostname;
		$is_not_dev_referrer 	= self::USER_PORTAL_HOSTNAME_DEV !== $hostname;
		$is_not_local_referrer 	= self::USER_PORTAL_HOSTNAME_LOCAL !== $hostname;

		if ( !$is_not_local_referrer && WP_ENV === 'local' ) {
			return true;
		}

		if ( $is_not_prd_hostname && $is_not_dev_referrer ) {
			return false;
		}

		return true;
	}
	private function get_params_from_login_request( $request ) {
		$nonce		  = $request->get_param( 'nonce' );
		$user_email   = $request->get_param( 'user_email' );

		return array( $nonce, $user_email );
	}

	private function get_params_from_one_click_login_url_request( $request ) {
		$user_email   = $request->get_param( 'user_email' );
		$first_name	  = $request->get_param( 'first_name' );
		$last_name	  = $request->get_param( 'last_name' );
		$role		  = $request->get_param( 'role' );
		$dashboard_env = $request->get_param( 'dashboard_env' );

		return array( $user_email, $first_name, $last_name, $role, $dashboard_env );
	}

	private function strategy_sso( $assoc_args ) {
		try {
			if ( is_multisite() ) {
				throw new MultisiteEnabledException();
			}

			$user_email   = $assoc_args[ self::WP_CLI_EMAIL_ARG ];
			$first_name   = $assoc_args[ self::WP_CLI_FIRST_NAME_ARG ];
			$last_name    = $assoc_args[ self::WP_CLI_LAST_NAME_ARG ];
			$role         = $assoc_args[ self::WP_CLI_USER_ROLE_ARG ];
			$dashboard_env = $assoc_args[ self::WP_CLI_DASHBOARD_ENV_ARG ];

			$this->validate_login_params( $user_email, $first_name, $last_name, $role, $dashboard_env );
			
			$this->double_check_dashboard_permissions( $user_email, $role, $dashboard_env );

			$user = $this->sign_on_user_provider->get_or_create_wp_user( $user_email, $first_name, $last_name, $role );

			$nonce_array = $this->user_nonce_helper->generate_nonce( $user->ID );
			$nonce       = $nonce_array['nonce'];
			$expiration  = $nonce_array['expiration'];

			$successfully_added = $this->user_nonce_helper->add_nonce( $user->ID, $nonce, $expiration );

			if ( ! $successfully_added ) {
				throw new UserMetaAdditionException( "Nonce ({$nonce}) was not added successfully to users ({$user_email}) meta data" );
			}

			$redirect_url = get_rest_url( null, $this->login_route );
			$query_params = array( 'user_email' => $user_email, 'nonce' => $nonce );

		} catch ( UserCreationException $e ) {
			$this->sign_on_user_provider->rollback_user_creation( $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::USER_CREATE_ERROR . ": {$e->getMessage()}";
		} catch ( UserMetaAdditionException $e ) {
			$this->sign_on_user_provider->rollback_user_creation( $user_email );
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::ADD_USER_META_ERROR . ": {$e->getMessage()}";
		} catch ( ImpersonatedUserException $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::IMPERSONATED_USER_ERROR . ": {$e->getMessage()}";
		} catch ( MultisiteEnabledException $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::MULTISITE_ENABLED_ERROR . ": {$e->getMessage()}";
		} catch ( \Exception $e ) {
			$redirect_url = self::REDIRECT_URL_ON_ERROR;
			$error        = Logger::GENERAL_EXCEPTION_ERROR . ": {$e->getMessage()}";
		}

		$data = array(
			'redirect_url' => $redirect_url . '?' . http_build_query( $query_params ),
		);

		if ( isset( $error ) ) {
			$this->add_error_field_to_cli_command_return( $data, $error );
		}

		return $data;
	}

	private function add_error_field_to_cli_command_return( &$array, $error ) {
		$array['error_message'] = $error;
	}

	private function validate_login_params( $user_email, $first_name, $last_name, $role, $dashboard_env ) {
		if ( ! $this->validate_non_empty_string( $user_email ) ) {
			throw new \Exception( "User email ({$user_email}) is blank" );
		}

		if ( ! $this->validate_non_empty_string( $first_name ) ) {
			throw new \Exception( 'Validation of CLI command parameters failed as first name was blank.' );
		}

		if ( ! $this->validate_non_empty_string( $last_name ) ) {
			throw new \Exception( 'Validation of CLI command parameters failed as last name was blank.' );
		}

		if ( ! $this->sign_on_user_provider->validate_role( $role ) ) {
			throw new \Exception( "Validation of user role ({$role}) failed as it is not a known WordPress role " );
		}

		switch ( $dashboard_env ) {
			case 'local':
			case 'development':
			case 'staging':
			case 'production':
				break;
			default:
				throw new \Exception( "Validation of dashboard environment ({$dashboard_env}) failed as it is not a valid environment. " );
		}		
	}

	private function double_check_dashboard_permissions( $user_email, $role, $dashboard_env ) {
		$domain = '';

		switch ( $dashboard_env ) {
			case 'local':
				$domain = 'http://host.docker.internal:3030';
				break;
			case 'development':
			case 'staging':
				$domain = 'https://' . self::USER_PORTAL_HOSTNAME_DEV;
				break;
			case 'production':
				$domain = 'https://' . self::USER_PORTAL_HOSTNAME_PRD;
				break;
		}				

		$body = wp_json_encode([
			'email' => $user_email,
			'role' => $role
		]);

		$options = [
			'body'        => $body,
			'headers'     => [
				'Content-Type' => 'application/json',
			],
			'timeout'     => 60,
			'redirection' => 5,
			'blocking'    => true,
			'httpversion' => '1.0',
			'sslverify'   => WP_ENV !== 'local',
			'data_format' => 'body',
		];

		$response = wp_remote_post( $domain . '/v1/auth/access', $options );

		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			throw new \Exception( $error_message );
		}

		$body = json_decode( wp_remote_retrieve_body( $response ) );

		if ( ! $body->success ) {
			throw new \Exception( 'User cannot access this website.' );
		}
	}

	private function validate_non_empty_string( $string ) {
		return is_string( $string ) && ! empty( trim( $string ) );
	}

	private function generate_https_redirect( $query_params ) {
		$query_string = http_build_query( $query_params );
		$redirect_url = get_site_url( null, $this->login_route, 'https' );
		$response     = new \WP_REST_Response( null, 307, array( 'Location' => $redirect_url . '?' . $query_string ) );
		return $response;
	}
}
