<?php

class Holler_Pro_Integrations {

	public function __construct() {
		add_action( 'hollerbox/register_integrations', [ $this, 'init' ] );

		// Survey
		add_filter( 'hollerbox/integrations/webhook/body', [ $this, 'survey_webhook' ], 10, 4 );
		add_filter( 'hollerbox/integrations/email/replacements', [ $this, 'survey_replacements' ], 10, 3 );
	}

	/**
	 * Add a replacement code to show survey answers
	 *
	 * @param $replacements
	 * @param $props
	 * @param $lead
	 *
	 * @return mixed
	 */
	public function survey_replacements( $replacements, $props, $lead ) {

		$replacements['{{survey_answers}}'] = "";

		return $replacements;
	}

	/**
	 * Add survey answers to the webhook body
	 *
	 * @param $body  array
	 * @param $props
	 * @param $lead  Holler_Lead
	 * @param $popup Holler_Popup
	 */
	public function survey_webhook( $body, $props, $lead, $popup ) {

		if ( $popup->template !== 'popup_with_survey' || ! $lead->request->has_param( 'survey_answers' ) ) {
			return $body;
		}

		$body['survey_answers'] = map_deep( $lead->request->get_param( 'survey_answers' ), 'sanitize_textarea_field' );

		return $body;
	}

	/**
	 * Register the basic integration types
	 */
	public function init() {

		Holler_Integrations::register( 'activecampaign', [ $this, 'activecampaign' ] );
		Holler_Integrations::register( 'convertkit', [ $this, 'convertkit' ] );
		Holler_Integrations::register( 'drip', [ $this, 'drip' ] );
		Holler_Integrations::register( 'mailchimp', [ $this, 'mailchimp' ] );
		Holler_Integrations::register( 'mailpoet', [ $this, 'mailpoet' ] );
		Holler_Integrations::register( 'hubspot', [ $this, 'hubspot' ] );

		// Todo
		Holler_Integrations::register( 'keap', [ $this, 'keap' ] );

	}

	/**
	 * Send lead data to ActiveCampaign
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return true|WP_Error
	 */
	public function activecampaign( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'account' => '',
			'key'     => '',
			'tags'    => []
		] );

		$json = Holler_Pro_CRMs::activecampaign_v1( [
			'endpoint' => 'contact_sync',
			'method'   => 'POST',
			'key'      => $integration['key'],
			'account'  => $integration['account'],
			'body'     => [
				'first_name' => $lead->first_name,
				'last_name'  => $lead->last_name,
				'email'      => $lead->email,
				'phone'      => $lead->phone,
				'ip4'        => $lead->get_ip()
			]
		] );

		// Something went wrong here
		if ( is_wp_error( $json ) ) {
			return $json;
		}

		Holler_Pro_CRMs::activecampaign_v1( [
			'endpoint' => 'contact_tag_add',
			'method'   => 'POST',
			'key'      => $integration['key'],
			'account'  => $integration['account'],
			'body'     => [
				'email' => $lead->email,
				'tags'  => wp_list_pluck( $integration['tags'], 'text' ),
			]
		] );

		return true;
	}

	/**
	 * Send lead data to ConvertKit
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return true|WP_Error
	 */
	public function convertkit( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'key'   => '',
			'forms' => []
		] );

		foreach ( $integration['forms'] as $form ) {
			$form_id = absint( $form['id'] );

			$json = Holler_Pro_CRMs::convertkit( [
				'method'   => 'POST',
				'endpoint' => '/forms/' . $form_id . '/subscribe/',
				'key'      => $integration['key'],
				'body'     => [
					'email'      => $lead->email,
					'first_name' => $lead->first_name,
					'fields'     => [
						'last_name' => $lead->last_name,
						'phone'     => $lead->phone,
						'ip4'       => $lead->get_ip()
					]
				]
			] );

			if ( is_wp_error( $json ) ) {
				return $json;
			}
		}

		return true;
	}

	/**
	 * Send lead data to Drip
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return WP_Error|true
	 */
	public function drip( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'account' => '',
			'key'     => '',
			'tags'    => []
		] );

		$result = Holler_Pro_CRMs::drip( [
			'endpoint' => 'subscribers',
			'method'   => 'POST',
			'key'      => $integration['key'],
			'account'  => $integration['account'],
			'body'     => [
				'subscribers' => [
					[
						'first_name' => $lead->first_name,
						'last_name'  => $lead->last_name,
						'email'      => $lead->email,
						'phone'      => $lead->phone,
						'ip_address' => $lead->get_ip(),
						'tags'       => wp_list_pluck( $integration['tags'], 'text' )
					]
				],
			]
		] );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Send lead data to Keap
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return void
	 */
	public function keap( $integration, $lead ) {

	}

	/**
	 * Send lead data to MailChimp
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return true|WP_Error
	 */
	public function mailchimp( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'key'   => '',
			'lists' => [],
			'tags'  => []
		] );

		$tags = wp_list_pluck( $integration['tags'], 'id' );
		$tags = array_map( function ( $tag ) {
			return [ 'name' => $tag, 'status' => 'active' ];
		}, $tags );

		$sub_hash = md5( $lead->email );

		$last_result = false;

		foreach ( $integration['lists'] as $list ) {
			$list_id = $list['id'];

			$last_result = Holler_Pro_CRMs::mailchimp( [
				'endpoint' => 'lists/' . $list_id . '/members/' . $sub_hash,
				'method'   => 'PUT',
				'key'      => $integration['key'],
				'body'     => [
					'email_address' => $lead->email,
					'ip_signup'     => $lead->get_ip(),
					'status'        => isset( $integration['double_optin'] ) && $integration['double_optin'] ? 'pending' : 'subscribed',
					'merge_fields'  => [
						'FNAME' => $lead->first_name,
						'LNAME' => $lead->last_name,
						'PHONE' => $lead->phone
					]
				]
			] );

			if ( is_wp_error( $last_result ) ) {
				continue;
			}

			if ( empty( $tags ) ) {
				continue;
			}

			$last_result = Holler_Pro_CRMs::mailchimp( [
				'endpoint' => 'lists/' . $list_id . '/members/' . $sub_hash . '/tags',
				'method'   => 'POST',
				'key'      => $integration['key'],
				'body'     => [
					'tags' => $tags
				]
			] );

			// Apply tags has no JSON response
			if ( $last_result->get_error_code() === 'not_json' ) {
				$last_result = false;
			}
		}

		return is_wp_error( $last_result ) ? $last_result : true;
	}

	/**
	 * Send lead data to HubSpot
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 *
	 * @return bool|WP_Error
	 */
	public function hubspot( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'account' => '',
			'key'     => '',
			'lists'   => []
		] );

		$props = [
			'firstname' => $lead->first_name,
			'lastname'  => $lead->last_name,
			'email'     => $lead->email,
			'phone'     => $lead->email,
		];

		$_props = [];

		foreach ( $props as $prop => $value ) {
			$_props[] = [
				'property' => $prop,
				'value'    => $value
			];
		}

		$result = Holler_Pro_CRMs::hubspot( [
			'endpoint' => 'contacts/v1/contact/createOrUpdate/email/' . $lead->email,
			'method'   => 'POST',
			'key'      => $integration['key'],
			'body'     => [
				'properties' => $_props
			]
		] );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		foreach ( $integration['lists'] as $list ) {
			$list_id = $list['id'];

			Holler_Pro_CRMs::hubspot( [
				'endpoint' => 'contacts/v1/lists/' . $list_id . '/add/',
				'method'   => 'POST',
				'key'      => $integration['key'],
				'body'     => [
					'emails' => [
						$lead->email
					]
				]
			] );
		}

		return true;
	}


	/**
	 * Send lead data to MailPoet
	 *
	 * @param $integration
	 * @param $lead Holler_Lead
	 *
	 * @return bool
	 */
	public function mailpoet( $integration, $lead ) {

		$integration = wp_parse_args( $integration, [
			'lists' => []
		] );

		if ( ! class_exists( \MailPoet\API\API::class ) ) {
			return false;
		}

		try {
			$api = \MailPoet\API\API::MP( 'v1' );
		} catch ( Exception $e ) {
			return false;
		}

		$list_ids = wp_parse_id_list( wp_list_pluck( $integration['lists'], 'id' ) );

		try {

			// Add the subscriber
			$api->addSubscriber( [
				'first_name' => $lead->first_name,
				'last_name'  => $lead->last_name,
				'email'      => $lead->email,
				'phone'      => $lead->phone
			], $list_ids );

		} catch ( \MailPoet\API\MP\v1\APIException $e ) {

			switch ( $e->getCode() ) {
				// Subscriber exists iin the list already
				case 12:

					try {
						$api->subscribeToLists( $lead->email, $list_ids );
					} catch ( \MailPoet\API\MP\v1\APIException $e ) {
						return false;
					}

					break;
				default:
					return false;
			}

		}

		return true;
	}


}

