<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class Holler_Pro_Fomo {

	public function __construct() {
		add_action( 'rest_api_init', [ $this, 'init' ] );
		add_filter( 'hollerbox/popup/can_show', [ $this, 'maybe_show_fomo' ], 10, 2 );
	}

	/**
	 * If FOMO has no items, don't show
	 *
	 * @param $can_show bool
	 * @param $popup    Holler_Popup
	 *
	 * @return bool
	 */
	public function maybe_show_fomo( $can_show, $popup ) {
		if ( ! $can_show || $popup->template !== 'fomo' ) {
			return $can_show;
		}

		$items = $this->get_fomo_items( $popup );

		return ! empty( $items );
	}

	public function init() {

		register_rest_route( 'hollerbox', '/fomo', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'fomo_rest' ],
				'permission_callback' => '__return_true'
			],
		] );

	}

	/**
	 * @var DateTime
	 */
	protected $after;
	protected $image_type;
	protected $text;
	protected $integration;

	public function fomo_rest( WP_REST_Request $request ) {

		$id = absint( $request->get_param( 'popup_id' ) );

		$popup = new Holler_Popup( $id );

		if ( ! $popup->exists() ) {
			return Holler_Api::ERROR_404();
		}

		return rest_ensure_response( [
			'items' => $this->get_fomo_items( $popup )
		] );
	}

	/**
	 * @throws Exception
	 *
	 * @param $popup Holler_Popup
	 *
	 * @return array[]
	 */
	public function get_fomo_items( $popup ) {

		$time_ago          = $popup->fomo_time_ago ?: 14;
		$this->after       = new DateTime( "$time_ago days ago", wp_timezone() );
		$this->image_type  = $popup->fomo_image ?: 'product';
		$this->integration = $popup->fomo_integration ?: '';

		$method = 'get_' . $this->integration . '_items';

		// Default display item so not empty
		$items = [
			[
				'name'     => wp_get_current_user()->display_name,
				'product'  => 'Product with a really long name',
				'url'      => 'https://hollerwp.com',
				'image'    => get_avatar_url( wp_get_current_user()->user_email ),
				'time_ago' => $this->time_ago( '5 minutes ago' ),
			],
		];

		if ( method_exists( $this, $method ) ) {
			$items = call_user_func( [ $this, $method ] );
		}

		/**
		 * Filter the fomo items
		 *
		 * @param $items array[] list of fomo items
		 * @param $popup Holler_Popup the current popup
		 */
		return apply_filters( 'hollerbox/pro/fomo_items', $items, $popup );
	}

	/**
	 * Returns a human diff time for the time ago string
	 *
	 * @throws Exception
	 *
	 * @param $date_or_timestamp
	 *
	 * @return string
	 */
	protected function time_ago( $date_or_timestamp ) {

		if ( is_a( $date_or_timestamp, DateTime::class ) ) {
			$date = $date_or_timestamp;
		} else if ( is_int( $date_or_timestamp ) ) {
			$date = new DateTime( 'now', wp_timezone() );
			$date->setTimestamp( $date_or_timestamp );
		} else if ( is_string( $date_or_timestamp ) ) {
			$date = new DateTime( $date_or_timestamp, wp_timezone() );
		}

		return sprintf( __( '%s ago', 'holler-box-pro' ), human_time_diff( $date->getTimestamp(), time() ) );
	}

	/**
	 * Will return the gravatar URL, or the thumbnail src for the given post id depending on image type
	 *
	 * @param $email
	 * @param $post_id
	 *
	 * @return string
	 */
	protected function get_image( $email, $post_id ) {
		return $this->image_type === 'product' ? get_the_post_thumbnail_url( $post_id ) : get_avatar_url( $email );
	}

	/**
	 * Get EDD sale items
	 *
	 * @param $args
	 *
	 * @return array
	 */
	protected function get_edd_items() {

		$payments = edd_get_payments( [
			'date_query' => [
				'after' => $this->after->format( 'Y-m-d H:i:s' )
			],
			'status'     => 'publish',
			'number'     => 99,
			'output'     => 'payments'
		] );

		$items = [];

		foreach ( $payments as $payment ) {

			$download = $payment->downloads[0];

			$name    = $payment->first_name;
			$product = get_the_title( $download['id'] );
			$url     = get_permalink( $download['id'] );

			$items[] = [
				'name'     => $name,
				'product'  => $product,
				'url'      => $url,
				'image'    => $this->get_image( $payment->email, $download['id'] ),
				'time_ago' => $this->time_ago( $payment->completed_date ),
			];
		}

		return $items;
	}

	/**
	 * Get Give donation items
	 *
	 * @param $args
	 *
	 * @return array
	 */
	protected function get_give_items() {

		$payments = give_get_payments( [
			'date_query' => [
				'after' => $this->after->format( 'Y-m-d H:i:s' )
			],
			'status'     => 'publish',
			'number'     => 99,
			'output'     => 'payments'
		] );

		foreach ( $payments as &$payment ) {

			$payment = [
				'name'     => $payment->first_name,
				'product'  => $payment->form_title,
				'url'      => get_permalink( $payment->form_id ),
				'image'    => $this->get_image( $payment->email, $payment->form_id ),
				'time_ago' => $this->time_ago( $payment->completed_date ),
			];
		}

		return $payments;
	}

	/**
	 * Get WooCommerce sale items
	 *
	 * @param $args
	 *
	 * @return array
	 */
	protected function get_woo_items() {

		$orders = wc_get_orders( [
			'status'         => 'completed',
			'type'           => 'shop_order',
			'date_completed' => '>' . $this->after->getTimestamp(),
			'posts_per_page' => 99
		] );

		foreach ( $orders as &$order ) {

			$items = $order->get_items();
			$item  = $items[0];

			// Make sure order has items
			if ( ! $item || ! is_a( $item, WC_Order_Item::class ) || ! method_exists( $item, 'get_name' ) ){
				continue;
			}

			$order = [
				'name'     => $order->get_billing_first_name(),
				'product'  => $item->get_name(),
				'url'      => get_permalink( $item['product_id'] ),
				'image'    => $this->get_image( $order->get_billing_email(), $item['product_id'] ),
				'time_ago' => $this->time_ago( $order->get_date_completed() ),
			];
		}

		return $orders;
	}

	/**
	 * Get MemberPress  items
	 *
	 * @param $args
	 *
	 * @return array
	 */
	protected function get_pmp_items() {

		global $wpdb;

		$query  = "SELECT user_id, membership_id, timestamp FROM {$wpdb->pmpro_membership_orders} ORDER BY timestamp DESC LIMIT 99";
		$orders = $wpdb->get_results( $query );

		$items = [];

		foreach ( $orders as $order ) {

			$user = get_userdata( $order->user_id );

			if ( ! $user ) {
				continue;
			}

			$membership_level = pmpro_getMembershipLevelForUser( $user->ID );
			$checkout_url     = pmpro_url( 'checkout', '?level=' . $order->membership_id );

			$items[] = [
				'name'     => $user->first_name ?: $user->display_name,
				'product'  => $membership_level->name,
				'url'      => $checkout_url,
				'image'    => get_avatar_url( $user->user_email ),
				'time_ago' => $this->time_ago( $order->timestamp ),
			];
		}

		return $items;
	}

	/**
	 * Get MemberPress  items
	 *
	 * @param $args
	 *
	 * @return array
	 */
	protected function get_memberpress_items() {

		global $wpdb;

		$mepr_db = new MeprDb();

		$query  = "SELECT id FROM {$mepr_db->transactions} WHERE status ='complete' ORDER BY created_at DESC LIMIT 99";
		$orders = wp_parse_id_list( wp_list_pluck( $wpdb->get_results( $query ), 'id' ) );

		$items = [];

		foreach ( $orders as $order_id ) {

			$transaction = new MeprTransaction( $order_id );
			$user        = $transaction->user();
			$membership  = $transaction->product();

			if ( ! $user || ! $membership ) {
				continue;
			}

			$items[] = [
				'name'     => $user->first_name,
				'product'  => $membership->post_title,
				'url'      => $membership->url(),
				'image'    => $this->get_image( $user->user_email, $membership->ID ),
				'time_ago' => $this->time_ago( $transaction->created_at ),
			];
		}

		return $items;
	}


}
