<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class Holler_Pro_CRMs {

	public static function post( $url = '', $body = [], $method = 'POST', $headers = [], $as_array = false ) {

		$method = strtoupper( $method );

		$args = [
			'method'      => $method,
			'headers'     => $headers,
			'body'        => $body,
			'data_format' => 'body',
			'sslverify'   => true,
			'user-agent'  => 'HollerBox/' . HOLLERBOX_VERSION . '; ' . home_url()
		];

		if ( $method === 'GET' ) {
			$response = wp_remote_get( $url, $args );
		} else {
			$response = wp_remote_post( $url, $args );
		}

		if ( ! $response ) {
			return new WP_Error( 'unknown_error', sprintf( 'Failed to initialize remote %s.', $method ), [
				'response' => $response,
				'url'      => $url
			] );
		}

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$_body = wp_remote_retrieve_body( $response );
		$json  = json_decode( $_body, $as_array );

		if ( ! $json ) {
			return new WP_Error( 'not_json', sprintf( 'No JSON response from remote %s.', $method ), [
				'response' => $_body,
				'url'      => $url
			] );
		}

		return $json;
	}

	/**
	 * Remote post x-www-form-urlencoded content
	 * Glorified wp_remote_post wrapper
	 *
	 * @param string $url
	 * @param array  $body
	 * @param string $method
	 * @param array  $headers
	 * @param bool   $as_array
	 *
	 * @return array|bool|WP_Error|object
	 */
	public static function post_form( $url = '', $body = [], $method = 'POST', $headers = [], $as_array = false ) {
		$headers['Content-type'] = 'application/x-www-form-urlencoded';

		return self::post( $url, $body, $method, $headers, $as_array );
	}

	/**
	 * Remote post json content
	 * Glorified wp_remote_post wrapper
	 *
	 * @param string $url
	 * @param array  $body
	 * @param string $method
	 * @param array  $headers
	 * @param bool   $as_array
	 *
	 * @return array|bool|WP_Error|object
	 */
	public static function post_json( $url = '', $body = [], $method = 'POST', $headers = [], $as_array = false ) {

		$method = strtoupper( $method );

		switch ( $method ) {
			case 'POST':
			case 'PUT':
			case 'PATCH':
			case 'DELETE':
				$headers['Content-type'] = sprintf( 'application/json; charset=%s', get_bloginfo( 'charset' ) );
				$body                    = is_array( $body ) ? wp_json_encode( $body ) : $body;
				break;
		}

		return self::post( $url, $body, $method, $headers, $as_array );
	}

	/**
	 * Get data from ActiveCampaign api v1
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function activecampaign_v1( $request = [] ) {

		$request = wp_parse_args( $request, [
			'account'  => '',
			'endpoint' => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$base_name = sprintf( 'https://%s.api-us1.com/admin/api.php?api_action=', $request['account'] );

		$json = self::post_form( $base_name . $request['endpoint'], array_merge( [
			'api_key'    => $request['key'],
			'api_output' => 'json',
		], $request['body'] ), $request['method'] );

		if ( is_wp_error( $json ) ) {
			return $json;
		}

		if ( $json->result_code != 1 ) {
			return new WP_Error( 'api_error', $json->result_message );
		}

		return $json;
	}

	/**
	 * Get data from ActiveCampaign api v3
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function activecampaign_v3( $request = [] ) {

		$request = wp_parse_args( $request, [
			'account'  => '',
			'endpoint' => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$base_name = sprintf( 'https://%s.api-us1.com/api/3/', $request['account'] );

		return self::post_json( $base_name . $request['endpoint'], $request['body'], $request['method'], [
			'api-token' => $request['key']
		] );
	}

	/**
	 * Get data from ConvertKit
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function convertkit( $request = [] ) {

		$request = wp_parse_args( $request, [
			'endpoint' => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$base_name = sprintf( 'https://api.convertkit.com/v3/%s', $request['endpoint'] );

		$body = array_merge( $request['body'], [
			'api_key' => $request['key']
		] );

		$json = self::post_json( $base_name, $body, $request['method'] );

		if ( is_wp_error( $json ) ) {
			return $json;
		}

		if ( $json->error ) {
			return new WP_Error( $json->error, $json->message );
		}

		return $json;
	}

	/**
	 * Get data from Drip
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function drip( $request = [] ) {

		$request = wp_parse_args( $request, [
			'endpoint' => '',
			'account'  => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$base_name = sprintf( 'https://api.getdrip.com/v2/%s/%s', $request['account'], $request['endpoint'] );

		$json = self::post_json( $base_name, $request['body'], $request['method'], [
			'Authorization' => 'Basic ' . base64_encode( $request['key'] . ':' )
		] );

		if ( is_wp_error( $json ) ) {
			return $json;
		}

		if ( $json->errors ) {
			return new WP_Error( $json->errors[0]->code, $json->errors[0]->message );
		}

		return $json;
	}

	/**
	 * Get data from MailChimp
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function mailchimp( $request = [] ) {

		$request = wp_parse_args( $request, [
			'endpoint' => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$data_center = substr( $request['key'], strpos( $request['key'], '-' ) + 1 );;

		$base_name = sprintf( 'https://%s.api.mailchimp.com/3.0/%s', $data_center, $request['endpoint'] );

		$json = self::post_json( $base_name, $request['body'], $request['method'], [
			'Authorization' => 'Bearer ' . $request['key']
		] );

		if ( is_wp_error( $json ) ) {
			return $json;
		}

		if ( $json->status && absint( $json->status ) >= 400 ) {
			return new WP_Error( $json->title, $json->detail, $json );
		}

		return $json;
	}

	/**
	 * Get data from HubSpot
	 *
	 * @param array $request
	 *
	 * @return object|WP_Error
	 */
	public static function hubspot( $request = [] ) {

		$request = wp_parse_args( $request, [
			'endpoint' => '',
			'key'      => '',
			'body'     => [],
			'method'   => 'GET',
		] );

		$base_name = sprintf( 'https://api.hubapi.com/%s', $request['endpoint'] );

		$json = self::post_json( $base_name, $request['body'], $request['method'], [
			'Authorization' => 'Bearer ' . $request['key']
		] );

		if ( is_wp_error( $json ) ) {
			return $json;
		}

		if ( $json->status === 'error' ) {
			$error = new WP_Error( 'hubspot_error', $json->message );

			foreach ( $json->validationResults as $result ){
				$error->add( $result->error, $result->message );
			}

			return $error;
		}

		return $json;
	}


}
