<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class Holler_Pro_Api {

	public function __construct() {
		add_action( 'rest_api_init', [ $this, 'init' ] );
	}

	public function init() {

		register_rest_route( 'hollerbox', '/activecampaign', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'activecampaign' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/activecampaign/lists', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'activecampaign_lists' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/convertkit', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'convertkit' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/drip', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'drip' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/mailchimp/lists', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'mailchimp_lists' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/mailchimp/tags', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'mailchimp_tags' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/hubspot', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'hubspot' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

		register_rest_route( 'hollerbox', '/mailpoet', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'mailpoet' ],
				'permission_callback' => [ $this, 'permission_callback' ]
			],
		] );

	}

	/**
	 * Return a credentials error
	 *
	 * @return WP_Error
	 */
	public static function credentials_error() {
		return new WP_Error( 'invalid_credentials', 'Check your api credentials' );
	}

	/**
	 * Get tags from ActiveCampaign
	 *
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function activecampaign( WP_REST_Request $request ) {

		$account = sanitize_text_field( $request->get_param( 'account' ) );
		$key     = sanitize_text_field( $request->get_param( 'key' ) );

		if ( ! $account || ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::activecampaign_v3( [
			'endpoint' => 'tags',
			'account'  => $account,
			'key'      => $key,
			'method'   => 'GET',
			'body'     => [
				'search' => sanitize_text_field( $request->get_param( 'search' ) )
			]
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		return rest_ensure_response( [
			'tags' => array_filter( $res->tags, function ( $t ) {
				return $t->tagType == 'contact';
			} )
		] );
	}

	/**
	 * Get tags from ActiveCampaign
	 *
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function activecampaign_lists( WP_REST_Request $request ) {

		$account = sanitize_text_field( $request->get_param( 'account' ) );
		$key     = sanitize_text_field( $request->get_param( 'key' ) );
		$search  = sanitize_text_field( $request->get_param( 'search' ) );

		if ( ! $account || ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::activecampaign_v1( [
			'endpoint' => 'list_list',
			'account'  => $account,
			'key'      => $key,
			'method'   => 'GET',
			'body'     => [
				'filters' => [
					'name' => $search
				],
			]
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		$lists = array_filter( get_object_vars( $res ), 'is_object' );

		return rest_ensure_response( [
			'lists' => $lists
		] );
	}

	/**
	 * Get forms from convert kit
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function convertkit( WP_REST_Request $request ) {

		$key    = sanitize_text_field( $request->get_param( 'key' ) );
		$search = strtolower( sanitize_text_field( $request->get_param( 'search' ) ) );

		if ( ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::convertkit( [
			'endpoint' => 'forms',
			'key'      => $key,
			'method'   => 'GET',
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		$forms = $res->forms;

		if ( ! $forms ) {
			return new WP_Error( 'no_forms', 'No forms found', $res );
		}

		if ( $search ) {
			$forms = array_filter( $forms, function ( $form ) use ( $search ) {
				return strpos( strtolower( $form->name ), $search ) !== false;
			} );
		}

		return rest_ensure_response( [
			'forms' => array_values( $forms )
		] );
	}

	/**
	 * Get forms from drip
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function drip( WP_REST_Request $request ) {

		$account = sanitize_text_field( $request->get_param( 'account' ) );
		$key     = sanitize_text_field( $request->get_param( 'key' ) );
		$search  = strtolower( sanitize_text_field( $request->get_param( 'search' ) ) );

		if ( ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::drip( [
			'endpoint' => 'tags',
			'key'      => $key,
			'account'  => $account,
			'method'   => 'GET',
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		$tags = $res->tags;

		if ( ! $tags ) {
			return new WP_Error( 'no_tags', 'No tags found', $res );
		}

		if ( $search ) {
			$tags = array_filter( $tags, function ( $tag ) use ( $search ) {
				return strpos( strtolower( $tag ), $search ) !== false;
			} );
		}

		return rest_ensure_response( [
			'tags' => array_values( $tags )
		] );
	}

	/**
	 * Get forms from MailChimp
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function mailchimp_lists( WP_REST_Request $request ) {

		$key    = sanitize_text_field( $request->get_param( 'key' ) );
		$search = strtolower( sanitize_text_field( $request->get_param( 'search' ) ) );

		if ( ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::mailchimp( [
			'endpoint' => 'lists',
			'key'      => $key,
			'method'   => 'GET',
			'body'     => [
				'fields' => 'lists.name,lists.id',
//				'count'  => 1000
			]
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		$lists = $res->lists;

		if ( ! $lists ) {
			return new WP_Error( 'no_lists', 'No lists found', $res );
		}

		if ( $search ) {
			$lists = array_filter( $lists, function ( $list ) use ( $search ) {
				return strpos( strtolower( $list->name ), $search ) !== false;
			} );
		}

		return rest_ensure_response( [
			'lists' => array_values( $lists )
		] );
	}

	/**
	 * Get forms from MailChimp
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function mailchimp_tags( WP_REST_Request $request ) {

		$key    = sanitize_text_field( $request->get_param( 'key' ) );
		$lists  = map_deep( $request->get_param( 'lists' ), 'sanitize_text_field' );
		$search = sanitize_text_field( $request->get_param( 'search' ) );

		if ( ! $key ) {
			return self::credentials_error();
		}

		if ( ! $lists ) {
			return new WP_Error( 'no_lists', 'No lists specified' );
		}

		$tags = [];

		foreach ( $lists as $list ) {
			$res = Holler_Pro_CRMs::mailchimp( [
				'endpoint' => 'lists/' . $list . '/tag-search',
				'key'      => $key,
				'method'   => 'GET',
				'body'     => [
					'name' => $search
				]
			] );

			if ( is_wp_error( $res ) ) {
				return $res;
			}

			if ( ! $res->tags ) {
				return new WP_Error( 'no_tags', 'No tags were found', $res );
			}

			$_tags = wp_list_pluck( $res->tags, 'name' );
			$tags  = array_merge( $tags, $_tags );
		}

		return rest_ensure_response( [
			'tags' => array_unique( $tags )
		] );
	}

	/**
	 * Get forms from MailChimp
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function hubspot( WP_REST_Request $request ) {

		$key    = sanitize_text_field( $request->get_param( 'key' ) );
		$search = strtolower( sanitize_text_field( $request->get_param( 'search' ) ) );

		if ( ! $key ) {
			return self::credentials_error();
		}

		$res = Holler_Pro_CRMs::hubspot( [
			'endpoint' => 'contacts/v1/lists/static',
			'key'      => $key,
			'method'   => 'GET',
			'body'     => [
				'count' => 250
			]
		] );

		if ( is_wp_error( $res ) ) {
			return $res;
		}

		$lists = $res->lists;

		if ( ! $lists ) {
			return new WP_Error( 'no_lists', 'No lists found', $res );
		}

		if ( $search ) {
			$lists = array_filter( $lists, function ( $list ) use ( $search ) {
				return strpos( strtolower( $list->name ), $search ) !== false;
			} );
		}

		return rest_ensure_response( [
			'lists' => array_values( $lists )
		] );
	}

	/**
	 * Get lists from MailPoet
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error
	 */
	public function mailpoet( WP_REST_Request $request ) {
		$search = strtolower( sanitize_text_field( $request->get_param( 'search' ) ) );

		if ( ! class_exists( \MailPoet\API\API::class ) ) {
			return new WP_Error( 'mailpoet', 'MailPoet is not installed' );
		}

		try {
			$api   = \MailPoet\API\API::MP( 'v1' );
			$lists = $api->getLists();
		} catch ( Exception $e ) {
			return new WP_Error( 'mailpoet', 'MailPoet API could not be initialized' );
		}

		if ( ! $lists ) {
			return new WP_Error( 'no_lists', 'No lists found' );
		}

		if ( $search ) {
			$lists = array_filter( $lists, function ( $list ) use ( $search ) {
				return strpos( strtolower( $list['name'] ), $search ) !== false;
			} );
		}

		return rest_ensure_response( [
			'lists' => array_values( $lists )
		] );
	}


	public function permission_callback() {
		return current_user_can( 'manage_options' );
	}

}
