<?php
/**
 * Plugin Name:     HollerBox - Pro
 * Plugin URI:      http://hollerwp.com
 * Description:     Premium features for the HollerBox plugin.
 * Version:         2.1
 * Author:          Groundhogg Inc.
 * Author URI:      http://groundhogg.io
 * Text Domain:     hollerbox-pro
 *
 * @author          Groundhogg Inc.
 * @copyright       Copyright (c) Groundhogg Inc. 2022
 */


// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'HOLLERBOX_PRO_VERSION', '2.1' );

if ( ! class_exists( 'Holler_Box_Pro' ) ) {

	/**
	 * Main Holler_Box class
	 *
	 * @since       0.1.0
	 */
	class Holler_Box_Pro {

		/**
		 * @var         Holler_Box $instance The one true Holler_Box
		 */
		private static $instance;


		/**
		 * Get active instance
		 *
		 * @access      public
		 * @return      object self::$instance The one true Holler_Box
		 */
		public static function instance() {
			if ( ! self::$instance ) {
				self::$instance = new Holler_Box_Pro();
			}

			return self::$instance;
		}

		public function __construct() {
			$this->setup_constants();
			$this->includes();
			$this->load_textdomain();
			$this->hooks();

			new Holler_Pro_Api();
			new Holler_Pro_Fomo();
			new Holler_Pro_Integrations();

			Holler_Licensing::instance()->edd_updater( __FILE__, Holler_Box_Pro_VER );
		}

		/**
		 * Setup plugin constants
		 *
		 * @access      private
		 * @since       0.1.0
		 * @return      void
		 */
		private function setup_constants() {
			// Plugin version
			define( 'Holler_Box_Pro_VER', HOLLERBOX_PRO_VERSION );

			// Plugin path
			define( 'Holler_Box_Pro_DIR', plugin_dir_path( __FILE__ ) );

			// Plugin URL
			define( 'Holler_Box_Pro_URL', plugin_dir_url( __FILE__ ) );

			define( 'Holler_Box_Pro_File', __FILE__ );

			// the name of your product. This is the title of your product in EDD and should match the download title in EDD exactly
			define( 'HWP_PRO_EDD_ID', 132 );
		}


		/**
		 * Include necessary files
		 *
		 * @access      private
		 * @since       0.1.0
		 * @return      void
		 */
		private function includes() {
			include __DIR__ . '/includes/class-holler-pro-integrations.php';
			include __DIR__ . '/includes/class-holler-pro-crms.php';
			include __DIR__ . '/includes/class-holler-pro-api.php';
			include __DIR__ . '/includes/class-holler-pro-fomo.php';

		}


		/**
		 * Run action and filter hooks
		 *
		 * @access      private
		 * @since       0.1.0
		 * @return      void
		 *
		 *
		 */
		private function hooks() {
			add_action( 'hollerbox/scripts', [ $this, 'frontend_scripts' ] );
			add_action( 'hollerbox/admin/scripts', [ $this, 'frontend_scripts' ] );
			add_action( 'hollerbox/admin/scripts', [ $this, 'admin_scripts' ] );
			add_action( 'hollerbox/init_display_conditions', [ $this, 'register_conditions' ] );

			if ( defined( 'PMPRO_VERSION' ) ) {
				add_action( 'hollerbox/popup/__construct', [ $this, 'maybe_upgrade_pmpro_integration' ] );
			}

			add_shortcode( 'holler-upgrade', [ $this, 'holler_upgrade_shortcode' ] );
		}

		/**
		 * The popup
		 *
		 * @param $popup Holler_Popup
		 *
		 * @return void
		 */
		public function maybe_upgrade_pmpro_integration( $popup ) {

			if ( ! defined( 'PMPRO_VERSION' ) ) {
				return;
			}

			// Levels integration was already upgraded
			if ( $popup->pmpro_updated ) {
				return;
			}

			$selected = [];

			$levels = pmpro_getAllLevels( true, true );

			// Check if the level is required and add it to the selection
			foreach ( $levels as $key => $value ) {
				if ( get_post_meta( $popup->ID, 'pmpro_membership_level_' . $value->id, true ) ) {
					$selected[] = [
						'id'   => $value->id,
						'text' => $value->name
					];
				}
			}

			// handle non-member selection
			if ( get_post_meta( $popup->ID, 'pmpro_membership_no_level', true ) ) {
				$selected[] = [
					'id'   => 'non_member',
					'text' => __( 'Non Member' )
				];
			}

			if ( empty( $selected ) ) {
				return;
			}

			$rules = get_post_meta( $popup->ID, 'advanced_rules', true );

			if ( ! is_array( $rules ) ) {
				$rules = [];
			}

			$rules['pmpro_level'] = [
				'selected' => $selected,
				'enabled'  => true,
			];

			// Update the rules with the PMPro settings
			update_post_meta( $popup->ID, 'advanced_rules', $rules );

			// falg that we did this and to not do it again
			add_post_meta( $popup->ID, 'pmpro_updated', true );
		}

		/*
		* Shortcode for content upgrades
		*/
		public function holler_upgrade_shortcode( $atts, $content = null ) {

			$a = shortcode_atts( [
				'link' => 'Click here.',
				'id'   => '',
			], $atts );

			return '<p class="holler-upgrade">' . $content . ' <a class="holler-show" data-id="' . $a['id'] . '" href="javascript:void(0)">' . $a['link'] . '</a></p>';

		}

		/**
		 * Register the pro scheduling conditions
		 *
		 * @return void
		 */
		public function register_conditions() {

			// Advanced Conditions
			Holler_Popup::add_display_condition( 'show_until_date', function ( $filter ) {
				$date = new DateTime( $filter['date'], wp_timezone() );
				$curr = new DateTime( 'now', wp_timezone() );

				return $date->getTimestamp() > $curr->getTimestamp();
			} );

			Holler_Popup::add_display_condition( 'show_after_date', function ( $filter ) {
				$date = new DateTime( $filter['date'], wp_timezone() );
				$curr = new DateTime( 'now', wp_timezone() );

				return $date->getTimestamp() < $curr->getTimestamp();
			} );

			Holler_Popup::add_display_condition( 'user_roles', function ( $filter ) {

				// Ignore if roles is not defined
				if ( ! is_array( $filter['selected'] ) ) {
					return true;
				}

				$roles = wp_list_pluck( $filter['selected'], 'id' );

				// if not logged in
				if ( ! is_user_logged_in() ) {

					// Guest is selected
					if ( in_array( 'guest', $roles ) ){
						return true;
					}

					return false;
				}

				$user     = wp_get_current_user();
				$my_roles = array_values( $user->roles );

				return count( array_intersect( $my_roles, $roles ) ) > 0;
			} );

			if ( defined( 'PMPRO_VERSION' ) ) {

				Holler_Popup::add_display_condition( 'pmpro_level', function ( $filter ) {

					// Ignore if levels is not defined
					if ( ! is_array( $filter['selected'] ) ) {
						return true;
					}

					$levels = wp_list_pluck( $filter['selected'], 'id' );

					foreach ( $levels as $level ) {

						switch ( $level ) {
							case 'any':
								// Any Level
								if ( pmpro_hasMembershipLevel() ) {
									return true;
								}
								break;
							case 'non_member':
								// No levels
								if ( ! pmpro_hasMembershipLevel() ) {
									return true;
								}
								break;
							default:
								// Specific level
								if ( pmpro_hasMembershipLevel( $level ) ) {
									return true;
								}
								break;
						}
					}

					return false;
				} );
			}
		}

		/**
		 * Frontend scripts
		 *
		 * @return void
		 */
		public function frontend_scripts() {

			$dot_min = Holler_Settings::instance()->get( 'script_debug_mode' ) ? '' : '.min';

			wp_register_style( 'hollerbox-pro', Holler_Box_Pro_URL . '/assets/css/holler-pro.css', [], HOLLERBOX_PRO_VERSION );

			wp_register_script( 'hollerbox-pro', Holler_Box_Pro_URL . '/assets/js/holler-pro' . $dot_min . '.js', [
				'hollerbox-popups'
			], HOLLERBOX_PRO_VERSION, true );

			wp_enqueue_style( 'hollerbox-pro' );
			wp_enqueue_script( 'hollerbox-pro' );

			wp_localize_script( 'hollerbox-pro', 'HollerBoxPro', [
				'routes' => [
					'fomo' => rest_url( 'hollerbox/fomo' ),
				]
			] );
		}

		/**
		 * Returns an array of roles used for select elements.
		 *
		 * @return string[]
		 */
		public function get_roles_for_select() {
			$editable_roles = array_reverse( get_editable_roles() );

			$roles = [];

			foreach ( $editable_roles as $role => $details ) {
				$name    = translate_user_role( $details['name'] );
				$roles[] = [ 'id' => $role, 'text' => $name ];
			}

			return $roles;
		}

		/**
		 * Admin Scripts
		 *
		 * @return void
		 */
		public function admin_scripts() {

			$dot_min = Holler_Settings::instance()->get( 'script_debug_mode' ) ? '' : '.min';

			wp_register_style( 'hollerbox-admin-pro', Holler_Box_Pro_URL . '/assets/css/holler-pro-admin.css', [], HOLLERBOX_PRO_VERSION );

			wp_register_script( 'hollerbox-admin-pro', Holler_Box_Pro_URL . '/assets/js/holler-pro-admin' . $dot_min . '.js', [
				'hollerbox-builder',
				'hollerbox-pro'
			], HOLLERBOX_PRO_VERSION );

			wp_localize_script( 'hollerbox-admin-pro', 'HollerBoxPro', [
				'assets'    => [
					'img' => [
						'root' => Holler_Box_Pro_URL . 'assets/img',
						'pdf'  => Holler_Box_Pro_URL . 'assets/img/pdf-download.png',
					]
				],
				'installed' => [
					'mailpoet'    => defined( 'MAILPOET_VERSION' ),
//					'fluentcrm'   => defined( '' ),
					'edd'         => defined( 'EDD_VERSION' ),
					'pmp'         => defined( 'PMPRO_VERSION' ),
					'memberpress' => defined( 'MEPR_VERSION' ),
					'woo'         => defined( 'WC_PLUGIN_FILE' ),
					'give'        => defined( 'GIVE_VERSION' ),
				],
				'roles'     => $this->get_roles_for_select()
			] );

			if ( defined( 'PMPRO_VERSION' ) ) {
				wp_localize_script( 'hollerbox-admin-pro', 'HollerBoxProPMP', array_values( pmpro_getAllLevels( true, true ) ) );
			}

			wp_enqueue_script( 'hollerbox-admin-pro' );
			wp_enqueue_style( 'hollerbox-admin-pro' );

		}

		/**
		 * Internationalization
		 *
		 * @access      public
		 * @since       0.1.0
		 * @return      void
		 */
		public function load_textdomain() {
			load_plugin_textdomain( 'hollerbox-pro' );
		}

		/**
		 * Holler Box Free Required admin notice
		 *
		 * @access      public
		 * @since       0.1.0
		 * @return      void
		 */
		public static function hwp_required() {
			echo '<div id="message" class="error"><p>' . sprintf( __( '%1$s requires the HollerBox (free version) plugin to be installed/activated. Please install under Plugins => Add New => Search for HollerBox. %1$s has been deactivated.', 'hollerbox' ), '<b>HollerBox - Pro</b> ' ) . '</p></div>';
			deactivate_plugins( plugin_basename( __FILE__ ), true );
		}

	}

	/**
	 *
	 * @since       0.1.0
	 * @return      void
	 *
	 */
	function holler_box_pro_load() {

		// is main plugin active? If not, throw a notice and deactivate
		if ( ! class_exists( 'Holler_Box' ) ) {
			add_action( 'all_admin_notices', array( 'Holler_Box_Pro', 'hwp_required' ) );

			return;
		}

		if ( did_action( 'hollerbox/loaded' ) ) {
			Holler_Box_Pro::instance();

			return;
		}

		add_action( 'hollerbox/loaded', function () {
			Holler_Box_Pro::instance();
		} );
	}

	add_action( 'plugins_loaded', 'holler_box_pro_load' );

}

