(() => {

  const PopupTemplates = HollerBox.templates
  const TriggerCallbacks = HollerBox._frontend.TriggerCallbacks

  const {
    CommonActions,
    SubmitActions,
    closeButton,
    credit,
    form,
    nameInput,
    emailInput,
    __content,
    submitButton,
    overlay,
    isBuilderPreview,
    makeURL
  } = HollerBox._frontend

  function isString (string) {
    return typeof string === 'string'
  }

  /**
   * If it's not a string just return the value
   *
   * @param string
   * @returns {*}
   */
  const specialChars = (string) => {
    if (!isString(string)) {
      return string
    }

    return string.replace(/&/g, '&amp;').
    replace(/>/g, '&gt;').
    replace(/</g, '&lt;').
    replace(/"/g, '&quot;')
  }

  function ApiError (message) {
    this.name = 'ApiError'
    this.message = message
  }

  ApiError.prototype = Error.prototype

  /**
   * Fetch stuff from the API
   * @param route
   * @param params
   * @param opts
   */
  async function apiGet (route, params = {}, opts = {}) {

    let __params = new URLSearchParams()

    Object.keys(params).forEach(k => {
      __params.append(k, params[k])
    })

    const response = await fetch(route + '?' + __params, {
      headers: {
        'X-WP-Nonce': HollerBox.nonces._wprest,
      },
      ...opts,
    })

    let json = await response.json()

    if (!response.ok) {
      console.log(json)
      throw new ApiError(json.message)
    }

    return json
  }

  const NoActions = {
    ...SubmitActions,
    redirect: (popup) => {

      let { no_redirect_url = '' } = popup

      let url = new URL(no_redirect_url)
      let home = new URL(HollerBox.home_url)

      if (HollerBox.is_builder_preview) {

        if (url.hostname !== home.hostname) {
          popup.close()
          return
        }

        url.searchParams.append('suppress_hollerbox', 1)
      }

      window.open(url, '_self')
    },
  }

  let embedScripts = []

  /**
   * Turn string HTML into a Dom element
   * @param html
   * @return {Element}
   */
  const makeElement = html => {
    let div = document.createElement('div')
    let innerDiv = document.createElement('div')
    innerDiv.classList.add('holler-embed-wrap')
    div.append(innerDiv)
    innerDiv.innerHTML = html
    return div.firstElementChild
  }

  /**
   * Parse string HTML into an embeddable html element
   * @param html
   * @return {*}
   */
  const parseEmbed = (html) => {

    const parsers = {
      youtube: [
        html => html.includes('https://www.youtube.com/embed/'),
        html => {
          let el = makeElement(html)
          let iframe = el.querySelector('iframe')
          iframe.setAttribute('height', null)
          iframe.setAttribute('width', null)
          iframe.classList.add('holler-responsive-embed')
          return el
        },
      ],
      instagram: [
        html => html.includes('instagram-media'),
        html => {
          return makeElement(html)
        },
      ],
      default: [
        html => true,
        html => {
          return makeElement(html)
        },
      ],
    }

    for (const parser in parsers) {
      const [is, parse] = parsers[parser]
      if (is(html)) {
        html = parse(html)
        break
      }
    }

    embedScripts = Array.from(html.querySelectorAll('script'))

    return html
  }

  const ProActions = {
    bannerOpened: ({ id, position }) => {
      let height = document.querySelector(`#${id} .holler-box-modal`).offsetHeight +
        'px'
      setTimeout(() => {
        switch (position) {
          case 'bottom-left':
            document.body.style.marginBottom = height
            break
          default:
            document.body.style.marginTop = height
            break
        }
      }, 300)
    },
    bannerClosed: (popup) => {
      document.body.style.marginTop = '0'
      document.body.style.marginBottom = '0'

      document.querySelector(`#${popup.id} .animation`).
      classList.
      add('slide-out')

      return new Promise((res) => {
        setTimeout(() => {
          res()
        }, 300)
      })
    },
    yesNo: (popup) => {
      popup.querySelector('.holler-box-button.choose-yes').
      addEventListener('click', e => {
        e.currentTarget.innerHTML = '<span class="holler-spinner"></span>'
        popup.submitted = true
        popup.chose = 'yes'
        popup.converted( 'Clicked Yes Button' )

        let { after_submit = 'close' } = popup

        SubmitActions[after_submit](popup)
        document.dispatchEvent(new CustomEvent('holler_chose_yes', {
          popup,
        }))
      })

      popup.querySelector('.holler-box-button.choose-no').
      addEventListener('click', e => {
        e.currentTarget.innerHTML = '<span class="holler-spinner"></span>'
        popup.submitted = true
        popup.chose = 'no'

        let { no_chosen = 'close' } = popup

        NoActions[no_chosen](popup)
        document.dispatchEvent(new CustomEvent('holler_chose_no', {
          popup,
        }))
      })
    },
    handleEmbed: popup => {
      popup.querySelector('.embed-here').append(parseEmbed(popup.embed_code))

      embedScripts.forEach(oldScriptEl => {

        const newScriptEl = document.createElement('script')

        Array.from(oldScriptEl.attributes).forEach(attr => {
          newScriptEl.setAttribute(attr.name, attr.value)
        })

        const scriptText = document.createTextNode(oldScriptEl.innerHTML)
        newScriptEl.appendChild(scriptText)

        document.body.appendChild(newScriptEl)
      })

    },
    inverseButton: popup => {
      let button = popup.querySelector( '.holler-box-button.inverse' )

      if ( ! button || ! popup.inverse_button_link ){
        return
      }

      let url = makeURL( popup.inverse_button_link )
      let home = new URL(HollerBox.home_url)

      button.addEventListener('click', e => {
        e.preventDefault()
        button.innerHTML = '<span class="holler-spinner"></span>'
        popup.converted( 'Clicked Alternate Button' ).then(() => {

          if (isBuilderPreview()) {

            if (url.hostname !== home.hostname) {
              popup.close()
              return
            }

            url.searchParams.append('suppress_hollerbox', 1)
          }

          window.open(url, '_self')
        })
      })
    }
  }

  const yesNoButtons = ({
    yes_text = '',
    no_text = '',
    button_alignment = 'vertical',
  }) => {
    //language=HTML
    return `
		<div class="yes-no-buttons ${button_alignment}">
			<button type="button" class="holler-box-button choose-no">
				${no_text}
			</button>
			<button type="button" class="holler-box-button choose-yes">
				${yes_text}
			</button>
		</div>
    `
  }

  const LoginForm = ({
    button_text = 'Login',
    username_placeholder = 'Username or Email',
    password_placeholder = 'Password',
    rememberme_text = 'Remember Me',
    lost_password_text = 'Forgot your password?',
    enable_rememberme = false,
    enable_lost_password = false,
    enable_inverse_button = false,
    inverse_button_text = 'Create an account',
    after_login = 'reload',
    redirect_to = ''
  }) => {

    const rememberMe = () => {
      // language=HTML
      return `
		  <div class="forgetmenot">
			  <input name="rememberme" type="checkbox" id="rememberme" value="forever">
			  <label for="rememberme">${rememberme_text}</label>
		  </div>`
    }

    const lostPassword = () => {
      // language=HTML
      return `
		  <div class="lostpass">
			  <a href="${HollerBox.nav?.lost_password}">${lost_password_text}</a>
		  </div>`
    }

    const createAccount = () => {
      // language=HTML
      return `
		  <button type="button" class="holler-box-button inverse">${inverse_button_text}</button>
      `
    }

    const rememberMeAndLostPassword = () => {
      // language=HTML
      return `<div class="space-between">
					${enable_rememberme ? rememberMe() : ''}
					${enable_lost_password ? lostPassword() : ''}
				</div>`
    }

    const loginUrl = new URL(HollerBox.nav?.login ?? window.location.href)

    let _redirect_to = after_login === 'reload' ? window.location.href : makeURL( redirect_to )

    loginUrl.searchParams.append('redirect_to', _redirect_to )

    // language=HTML
    return `
		<form class="holler-box-form vertical" method="post" action="${loginUrl.toString()}">
			<div class="fields">
				<input class="holler-box-input" type="text" id="user_login" name="log"
				       placeholder="${specialChars(username_placeholder)}" autocomplete="current-password" required>
				<input class="holler-box-input" type="password" id="user_pass" name="pwd"
				       placeholder="${specialChars(password_placeholder)}" autocomplete="username" required>
				${ enable_rememberme || enable_lost_password ? rememberMeAndLostPassword() : '' }
				<button type="submit" class="holler-box-button">${button_text}</button>
				${enable_inverse_button ? createAccount() : ''}
			</div>
		</form>`
  }

  const ProTemplates = {
    popup_yes_no: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
        yes_text = 'Yes!',
        no_text = 'No...',
        submitted = false,
        success_message = '',
        no_message = '',
        button_alignment = 'vertical',
        chose = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-popup yes-no ${submitted
				? 'no-animation'
				: ''}">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							${__content(
								submitted ? (chose === 'yes'
									? success_message
									: no_message) : post_content)}
							${submitted ? '' : yesNoButtons({
								yes_text,
								no_text,
								button_alignment,
							})}
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.yesNo(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    popup_yes_no_image_left: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
        yes_text = 'Yes!',
        no_text = 'No...',
        submitted = false,
        chose = '',
        success_message = '',
        no_message = '',
        button_alignment = 'vertical',
        image_src = '',
      }) => {
        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup yes-no holler-popup-image-left">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left image-width"
								     style="background-image: url('${image_src}')">
								</div>
								<div class="right">
									${__content(submitted
										? (chose === 'yes'
											? success_message
											: no_message)
										: post_content)}
									${submitted ? '' : yesNoButtons({
										yes_text,
										no_text,
										button_alignment,
									})}
								</div>
							</div>
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.yesNo(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    popup_yes_no_image_right: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
        yes_text = 'Yes!',
        no_text = 'No...',
        submitted = false,
        chose = '',
        success_message = '',
        no_message = '',
        button_alignment = 'vertical',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup holler-popup-image-right yes-no ${submitted
				     ? 'no-animation'
				     : ''}">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									${__content(submitted
										? (chose === 'yes'
											? success_message
											: no_message)
										: post_content)}
									${submitted ? '' : yesNoButtons({
										yes_text,
										no_text,
										button_alignment,
									})}
								</div>
								<div class="right image-width"
								     style="background-image: url('${image_src}')">
								</div>
							</div>
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.yesNo(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    fomo: {
      render: (popup) => {

        let {
          id = '',
          position = 'bottom-left',
        } = popup

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-notification-box holler-fomo">
				<div class="positioner ${position}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">

							</div>
							${credit()}
						</div>
					</div>
				</div>
			</div>`
      },
      beforeOpen: (popup) => {

        if (!popup.items) {
          popup.items = []
        }

        if (popup.items.length) {
          return
        }

        return apiGet(HollerBoxPro.routes.fomo, {
          popup_id: popup.ID,
        }).then(r => {
          popup.items = r.items
        })
      },
      onOpen: (popup) => {

        if (!popup.items.length && HollerBox.currentUser) {
          popup.items.push({
            name: HollerBox.currentUser.data.display_name,
            product: 'HollerBox',
            url: 'https://hollerwp.com',
            image: HollerBox.currentUser.data.avatar,
            time_ago: '5 minutes ago',
          })
        }

        const setContent = () => {

          // Remove from top of array
          let item = popup.items.shift()

          let {
            fomo_text = '{{name}} purchased {{product}}',
          } = popup

          let { name, product, url, image, time_ago } = item

          fomo_text = fomo_text.replaceAll('{{name}}',
            `<span class="holler-fomo-name">${name}</span>`)
          fomo_text = fomo_text.replaceAll('{{product}}',
            `<span class="holler-fomo-product-name"><a href="${url}">${product}</a></span>`)

          // Add to bottom of array
          popup.items.push(item)

          popup.querySelector('.display-flex').innerHTML =
            //language=html
            `
				<div class="fomo-image"
				     style="background-image: url(${image})"></div>
				<div class="order-details">
					${__content(fomo_text)}
					<div class="time-ago">
						${time_ago}
					</div>
				</div>`
        }

        const showDelay = popup.fomo_display_time ? popup.fomo_display_time *
          1000 : 4000
        const hideDelay = popup.fomo_loop_delay
          ? popup.fomo_loop_delay * 1000
          : 3000

        const show = () => {
          popup.timeout = setTimeout(() => {
            popup.querySelector('.animation')?.classList.remove('slide-out')
            popup.querySelector('.animation')?.classList.add('slide-in')
            hide()
          }, hideDelay)
        }

        const hide = () => {
          setContent()
          popup.timeout = setTimeout(() => {
            popup.querySelector('.animation')?.classList.remove('slide-in')
            popup.querySelector('.animation')?.classList.add('slide-out')
            show()
          }, showDelay)
        }

        hide()

      },
      onClose: (popup) => {
        popup.querySelector(`.animation`).classList.add('slide-out')
        clearTimeout(popup.timeout)

        return new Promise((res) => {
          setTimeout(() => {
            res()
          }, 300)
        })
      },
      cleanup: (popup) => {
        clearTimeout(popup.timeout)
      },

    },
    banner_standard: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="holler-box-modal-content">
								${post_content}
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: ProActions.bannerOpened,
      onClose: ProActions.bannerClosed,
    },
    banner_image_right: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-image-right">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									<div class="holler-button-cta">
										${__content(post_content)}
									</div>
								</div>
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_image_left: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-image-left">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
								<div class="right">
									<div class="holler-button-cta">
										${__content(post_content)}
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_fixed_image_right: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-fixed-image">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									<div class="holler-button-cta">
										${__content(post_content)}
									</div>
								</div>
								<div class="image image-width">
									<img src="${image_src}" class="image-width">
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_fixed_image_left: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-fixed-image">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="image image-width">
									<img src="${image_src}" class="image-width">
								</div>
								<div class="right">
									<div class="holler-button-cta">
										${__content(post_content)}
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_button: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Get Started!',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-button">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								${__content(post_content)}
								<div class="holler-button-cta">
									${submitButton(button_text, 'button')}
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.buttonClicked(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_button_image_right: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Get Started!',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-banner with-button with-image-right">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									<div class="holler-button-cta">
										${__content(post_content)}
										${submitButton(button_text, 'button')}
									</div>
								</div>
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.buttonClicked(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_button_image_left: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Get Started!',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-banner with-button with-image-left">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
								<div class="right">
									<div class="holler-button-cta">
										${__content(post_content)}
										${submitButton(button_text, 'button')}
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.buttonClicked(popup)
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_form: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Subscribe!',
        name_placeholder = 'Name',
        email_placeholder = 'Email',
        submitted = false,
        success_message = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-banner with-form">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation ${submitted ? '' : 'slide-in'}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								${__content(
									submitted ? success_message : post_content)}
								${submitted ? '' : form({
									name: false,
									direction: 'horizontal',
									button_text,
									name_placeholder,
									email_placeholder,
								})}
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.formSubmitted(popup)

        document.querySelector(`#${popup.id} .holler-button-cta .holler-box-button`).
        addEventListener('click', e => {
          window.open(popup.button_link, '_blank')
        })
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_form_image_right: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Subscribe!',
        name_placeholder = 'Name',
        email_placeholder = 'Email',
        submitted = false,
        success_message = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-banner with-form with-image-right">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation ${submitted ? '' : 'slide-in'}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									${__content(submitted
										? success_message
										: post_content)}
									${submitted ? '' : form({
										name: false,
										direction: 'horizontal',
										button_text,
										name_placeholder,
										email_placeholder,
									})}
								</div>
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.formSubmitted(popup)

        document.querySelector(`#${popup.id} .holler-button-cta .holler-box-button`).
        addEventListener('click', e => {
          window.open(popup.button_link, '_blank')
        })
      },
      onClose: ProActions.bannerClosed,
    },
    banner_with_form_image_left: {
      render: ({
        id = '',
        position = 'top-left',
        post_content = '',
        button_text = 'Subscribe!',
        name_placeholder = 'Name',
        email_placeholder = 'Email',
        submitted = false,
        success_message = '',
        image_src = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-banner with-form with-image-left">
				<div class="positioner ${position === 'bottom-left'
					? 'bottom-left'
					: 'top-left'}">
					<div class="animation ${submitted ? '' : 'slide-in'}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="image image-width"
								     style="background-image: url('${image_src}')">
								</div>
								<div class="right">
									${__content(submitted
										? success_message
										: post_content)}
									${submitted ? '' : form({
										name: false,
										direction: 'horizontal',
										button_text,
										name_placeholder,
										email_placeholder,
									})}
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        ProActions.bannerOpened(popup)
        CommonActions.formSubmitted(popup)

        document.querySelector(`#${popup.id} .holler-button-cta .holler-box-button`).
        addEventListener('click', e => {
          window.open(popup.button_link, '_blank')
        })
      },
      onClose: ProActions.bannerClosed,
    },
    sidebar_standard: {
      render: ({
        id = '',
        position = 'center-right',
        post_content = '',
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-sidebar">
				<div class="positioner ${position === 'center-left'
					? 'center-left'
					: 'center-right'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							${__content(post_content)}
						</div>
					</div>
				</div>
			</div>`
      },
      onClose: (popup) => {
        document.querySelector(`#${popup.id} .animation`).
        classList.
        add('slide-out')
        return new Promise((res) => {
          setTimeout(() => res(), 300)
        })
      },
    },
    sidebar_with_form_below: {
      render: ({
        id = '',
        position = 'center-right',
        post_content = '',
        submitted = false,
        success_message = '',
        ...props
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-sidebar">
				<div class="positioner ${position === 'center-left'
					? 'center-left'
					: 'center-right'}">
					<div class="animation slide-in">
						<div class="holler-box-modal">
							${closeButton()}
							${__content(
								submitted ? success_message : post_content)}
							${submitted ? '' : form({
								direction: 'vertical',
								...props,
							})}
						</div>
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.formSubmitted(popup)
      },
      onClose: (popup) => {
        document.querySelector(`#${popup.id} .animation`).
        classList.
        add('slide-out')
        return new Promise((res) => {
          setTimeout(() => res(), 300)
        })
      },
    },
    popup_with_survey: {
      render: (popup) => {

        let {
          id = '',
          position = 'center-right',
          overlay_enabled = true,
          submitted = false,
          success_message = '',
          form_prompt = '',
          survey_button_text = 'Start Survey',
          animation = 'appear',
          questions = [],
          __state = 'start',
          post_content = '',
          allow_skipping = false,
          skip_text = 'Skip',
          ...rest
        } = popup

        let content

        if (submitted) {
          __state = 'submitted'
        }

        switch (__state) {
          default:
          case 'start':

            content = [
              __content(post_content),
              //language=HTML
              `
				  <div class="survey-start">
					  ${submitButton(survey_button_text, 'button')}
				  </div>`,
            ].join('')

            break
          case 'survey':

            let current = popup.current
            let __question, prompt

            if (questions.length === 0) {
              questions.push({
                prompt: '<p>A question about you?</p>',
                choices: [
                  'Option A',
                  'Option B',
                  'Option C',
                ],
              })
            }

            if (current > questions.length) {
              return
            }

            prompt = questions[current].prompt

            const nextButton = () => {
              return `<button class="holler-survey-next" type="submit">${questions[current].next}</button>`
            }

            switch (questions[current].type) {
              default:
              case 'choices':
                // language=HTML
                __question = `
					<form class="holler-choices">
						${questions[current].choices.map((
							choice,
							i) => {

							if (Array.isArray(choice)) {
								let [_choice = '', _custom_value] = choice
								choice = _choice
							}

							return `<button class="holler-choice" name="choice" value="${i}">${choice}</button>`
						}).
						join('')}
					</form>`
                break
              case 'text':
                __question = [
                  `<form class="text-answer">`,
                  `<input required class="holler-survey-text holler-box-input" type="text" name="answer" placeholder="${questions[current].placeholder}">`,
                  nextButton(),
                ].join('')
                break
              case 'textarea':
                __question = [
                  `<form class="text-answer">`,
                  `<textarea required class="holler-survey-text holler-box-input" name="answer" placeholder="${questions[current].placeholder}"></textarea>`,
                  nextButton(),
                ].join('')
                break
            }

            content = [
              __content(prompt),
              __question,
              allow_skipping
                ? `<div class="holler-skip-wrap"><button class="holler-skip">${skip_text}</button></div>`
                : '',
            ].join('')

            break

          case 'form':

            content = [
              __content(form_prompt),
              form({
                direction: 'vertical',
                ...rest,
              }),
            ].join('')

            break

          case 'submitted':
            content = __content(success_message)
            break
        }

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup has-survey ${__state === 'start'
				     ? ''
				     : 'no-animation'}">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							${content}
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {

        CommonActions.maybeDisableScrolling(popup)

        let { __state = 'start' } = popup

        if (popup.submitted) {
          __state = 'submitted'
          popup.__state = __state
        }

        switch (__state) {
          default:
          case 'start':

            popup.querySelector('.holler-box-button').
            addEventListener('click', e => {

              popup.__state = 'survey'
              popup.current = 0
              popup.open()

            })

            break
          case 'survey':

            if (!popup.answers) {
              popup.answers = []
            }

            const currentQuestion = () => {
              return popup.questions[popup.current]
            }

            const setAnswer = (answer) => {
              popup.answers.push({
                prompt: currentQuestion().prompt.replace(/<\/?[^>]+(>|$)/g, ''),
                answer,
              })

              nextQuestion()
            }

            const nextQuestion = () => {
              popup.current += 1

              // No more questions
              if ((popup.current + 1) > popup.questions.length) {
                popup.__state = 'form'
              }

              popup.open()
            }

            if (popup.allow_skipping) {
              popup.querySelector('.holler-skip').
              addEventListener('click', e => {
                setAnswer('')
              })
            }

            switch (currentQuestion().type) {
              default:
              case 'choices':

                let chosen

                popup.querySelectorAll('.holler-choice').forEach(choice => {
                  choice.addEventListener('click', e => {
                    chosen = currentQuestion().choices[parseInt(
                      e.currentTarget.value)]
                  })
                })

                popup.querySelector('form.holler-choices').
                addEventListener('submit', e => {
                  e.preventDefault()

                  // If custom values are present
                  if (Array.isArray(chosen)) {
                    let [choice = '', custom_value = ''] = chosen
                    chosen = custom_value ? custom_value : choice
                  }

                  setAnswer(chosen)
                })
                break
              case 'text':
              case 'textarea':
                let answer

                let input = popup.querySelector('.holler-survey-text')

                input.addEventListener('input', e => {
                  answer = e.target.value
                })

                if (!isBuilderPreview()) {
                  input.focus()
                }

                popup.querySelector('form.text-answer').
                addEventListener('submit', e => {

                  e.preventDefault()

                  setAnswer(answer)
                })

                break
            }

            break
          case 'form':

            CommonActions.formSubmitted(popup, {
              modifyPayload: (payload) => {
                payload.survey_answers = popup.answers
              },
            })

            break
          case 'submitted':
            break
        }
      },
      onClose: (popup) => {
        popup.answers = []
        popup.current = 0

        CommonActions.enableScrolling(popup)
      },
    },
    popup_with_survey_links: {
      render:  popup  => {

        let {
          id = '',
          position = 'center-right',
          overlay_enabled = true,
          submitted = false,
          success_message = '',
          form_prompt = '',
          survey_button_text = 'Start Survey',
          animation = 'appear',
          choices = [],
          __state = 'start',
          post_content = '',
          allow_skipping = false,
          skip_text = 'Skip',
          chosen_message = '',
          ...rest
        } = popup

        let content

        content = [
          __content( post_content ),
          `<div class="holler-choices">`,
          choices.map( ({choice = ''}, i) => {
          // language=HTML
          return `<button class="holler-choice" name="choice" value="${i}">${choice}</button>`
        } ).join(''),
          '</div>'
        ].join('')

        if ( chosen_message ){
          content = __content( chosen_message )
        }

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup has-survey ${!chosen_message
          ? ''
          : 'no-animation'}">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							${content}
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: popup => {
        CommonActions.maybeDisableScrolling(popup)

        popup.querySelectorAll( '.holler-choice' ).forEach( el => {
          el.addEventListener( 'click', async e => {

            let chosen = popup.choices[parseInt(e.currentTarget.value)]

            switch ( chosen.result ?? 'close' ){
              default:
              case 'close':
                popup.converted( chosen.choice )
                return popup.close()
              case 'message':
                popup.chosen_message = chosen.message
                popup.converted( chosen.choice )
                popup.open()
                break;
              case 'redirect':

                el.innerHTML = '<span class="holler-spinner"></span>'
                await popup.converted( chosen.choice )

                if ( isBuilderPreview() ){
                  setTimeout( () => popup.close(), 1000 )
                  return
                }

                window.open( makeURL( chosen.redirect ), '_self' )
                break;
            }

          })
        })

      },
      onClose: (popup) => {
        CommonActions.enableScrolling(popup)
        popup.chosen_message = ''
      },
    },
    popup_with_embed: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
      }) => {

        // language=HTML
        return `
			<div id="${id}" class="holler-box holler-popup has-embed">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							${__content(post_content)}
							<div class="embed-here"></div>
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.handleEmbed(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    popup_with_embed_right: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup has-embed embed-right">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									${__content(post_content)}
								</div>
								<div class="right">
									<div class="embed-here"></div>
								</div>
							</div>
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.handleEmbed(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    popup_with_embed_left: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup has-embed embed-left">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							<div class="display-flex">
								<div class="left">
									<div class="embed-here"></div>
								</div>
								<div class="right">
									${__content(post_content)}
								</div>
							</div>
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.handleEmbed(popup)
      },
      onClosed: CommonActions.enableScrolling,
    },
    popup_with_login: {
      render: ({
        id = '',
        position = 'center-center',
        animation = 'appear',
        post_content = '',
        overlay_enabled = true,
        ...form
      }) => {

        // language=HTML
        return `
			<div id="${id}"
			     class="holler-box holler-popup with-login">
				${overlay_enabled ? overlay() : ''}
				<div class="positioner ${position}">
					<div class="animation ${animation}">
						<div class="holler-box-modal">
							${closeButton()}
							${__content(post_content)}
							${LoginForm(form)}
						</div>
						${credit()}
					</div>
				</div>
			</div>`
      },
      onOpen: (popup) => {
        CommonActions.maybeDisableScrolling(popup)
        ProActions.inverseButton(popup)

        let loginForm = popup.querySelector('form')

        if (isBuilderPreview()) {
          loginForm.addEventListener('submit', e => {
            e.preventDefault()
            return false
          })
        } else {

          let flag = false

          loginForm.addEventListener( 'submit', e => {

            if ( flag ){
              return true
            }

            e.preventDefault()
            popup.querySelector( '.holler-box-button[type=submit]' ).innerHTML = '<span class="holler-spinner"></span>'
            popup.converted( 'Logged In' ).then( () => {
              flag = true;
              loginForm.submit()
            })
            return false
          } )
        }
      },
      onClosed: CommonActions.enableScrolling,
    },
  }

  Object.keys(ProTemplates).forEach(t => {
    PopupTemplates[t] = ProTemplates[t]
  })

  const ProTriggers = {
    after_inactivity: ({ time = 0 }, show) => {

      let timer

      const resetTimer = () => {

        if (timer) {
          clearTimeout(timer)
        }

        timer = setTimeout(() => {

          show()

        }, parseInt(time) * 1000)
      }

      document.addEventListener('scroll', resetTimer)
      document.addEventListener('click', resetTimer)
      document.addEventListener('keydown', resetTimer)

    },
  }

  Object.keys(ProTriggers).forEach(t => {
    TriggerCallbacks[t] = ProTriggers[t]
  })

})()
